/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "button.h"
#include "desktop.h"
#include "menu.h"
#include "text.h"
#include <cassert>

#define MENU_CAPTION_TEXT_SIZE (14)

namespace Graph2D
{
	Menu::Menu()
		: padding(3)
		, dirty(false)
	{
		type = TYPE_MENU;
		setFrameStyle(FRAME_STYLE_NORMAL);
		setColor(0.2f, 0.2f, 1.0f, 0.8f, 0.0f);
		setLeftFrameSize(5.f);
		setTopFrameSize(5.f);
		setRightFrameSize(5.f);
		setBottomFrameSize(5.f);

		allocate(LAYER_SIZE);
		{
			Text* caption = new Text();
			caption->setFontSize(MENU_CAPTION_TEXT_SIZE);
			set(LAYER_CAPTION, caption);
			caption->release();
		}
		set(LAYER_CLOSE_BUTTON, NULL);
		{
			Container* container = new Container();
			container->setOwnerDraw(true);
			container->setScrollable(true);
			set(LAYER_CONTAINER, container);
			container->release();
		}
		adjust();
	}

	Menu::~Menu()
	{
	}

	Text* Menu::getCaption()
	{
		Component* component = get(LAYER_CAPTION);
		assert(component && component->isKindOfA(TYPE_TEXT));
		return reinterpret_cast<Text*>(component);
	}
	
	void Menu::setCaption(Text* caption)
	{
		assert(caption);
		set(LAYER_CAPTION, caption);
		dirty = true;
	}
	
	void Menu::setCaption(const std::string& caption)
	{
		getCaption()->setString(caption);
		dirty = true;
	}
	
	ImageButton* Menu::getCloseButton()
	{
		Component* component = get(LAYER_CLOSE_BUTTON);
		if(component)
		{
			assert(component->isKindOfA(TYPE_IMAGE_BUTTON));
		}
		return reinterpret_cast<ImageButton*>(component);
	}

	void Menu::setCloseButton(ImageButton* button)
	{
		set(LAYER_CLOSE_BUTTON, button);
		dirty = true;
	}

	Container* Menu::getContainer()
	{
		Component* component = get(LAYER_CONTAINER);
		assert(component && component->isKindOfA(TYPE_CONTAINER));
		return reinterpret_cast<Container*>(component);
	}

	void Menu::setContainer(Container* container)
	{
		assert(container);
		set(LAYER_CONTAINER, container);
		dirty = true;
	}

	void Menu::addContents(Component* component, const int data)
	{
		component->setActionEvent(this);
		component->setUserData(data);
		getContainer()->append(component);
		dirty = true;
	}

	void Menu::addContents(const std::string& text, const int data)
	{
		Button* button = new Button(text);
		addContents(button, data);
		button->release();
		dirty = true;
	}

	Component* Menu::searchContents(const int data)
	{
		for(size_t i = 0; i < getCount(); i++)
		{
			Component* component = get(i);
			if(!component)
				continue;
			if(component->getUserData() == data)
				return component;
		}
		return NULL;
	}

	void Menu::clearContents()
	{
		getContainer()->clear();
	}

	unsigned short Menu::getPadding() const
	{
		return padding;
	}

	void Menu::setPadding(const unsigned short padding)
	{
		this->padding = padding;
	}

	void Menu::adjust()
	{
		const Vector2 clientSize = getClientSize();

		ImageButton* button = getCloseButton();
		if(button)
		{
			button->setPosition(
				clientSize.x - button->getSize().x,
				0.0f
			);
		}

		Text* caption = getCaption();
		caption->adjustSize();
		caption->setPosition(
			clientSize.x * 0.5f - caption->getSize().x * 0.5f,
			0.0f
		);

		Container* container = getContainer();

		float y = 0;
		for(size_t i = 0; i < container->getCount(); i++)
		{
			Component* child = container->get(i);
			if(child == NULL)
				continue;

			child->setPosition(0, y);
			child->setSize(clientSize.x, child->getSize().y);

			y += child->getSize().y + static_cast<float>(padding);
		}

		const float captionBottom = caption->getSize().y + 2;
		const float buttonBottom = button ? (button->getSize().y + 2) : 0;
		const float containerTop = captionBottom > buttonBottom ? captionBottom : buttonBottom;

		container->setPosition(0.0f, containerTop);
		container->setSize(clientSize.x, clientSize.y - containerTop);
		container->setScrollSize(clientSize.x, y);

		dirty = false;
	}

	void Menu::onSerialize(mana_stream* stream) const
	{
		super::onSerialize(stream);

		mana_stream_mark(stream);
	}

	void Menu::onDeserialize(mana_stream* stream)
	{
		super::onDeserialize(stream);

		mana_stream_check(stream);
	}

	void Menu::onDraw(const DrawRect& drawRect)
	{
		// 変更があった場合はサイズ調整
		if(dirty)
		{
			adjust();
		}
		// 親クラスの動作を継承
		super::onDraw(drawRect);
	}

#if 0
	bool Menu::onTouchesBegan(const Vector2& localPosition)
	{
		setMouseCapture();
		return super::onTouchesBegan(localPosition);
	}

	bool Menu::onTouchesEnded(const Vector2& localPosition)
	{
		releaseMouseCapture();
		return super::onTouchesEnded(localPosition);
	}

	bool Menu::onTouchesCancelled(const Vector2& localPosition)
	{
		releaseMouseCapture();
		return super::onTouchesCancelled(localPosition);
	}
#endif

	void Menu::onAction(Component* component, const unsigned int identification)
	{
		(void)identification;

		if(actionEvent)
		{
			Container* container = getContainer();
			if(container->find(component))
				actionEvent->onAction(container, component->getUserData());
		}
	}

	void Menu::onClick(Component* component)
	{
		if(component == get(LAYER_CLOSE_BUTTON))
		{
			onCloseButtonDown();
		}
	}

	void Menu::onCloseButtonDown()
	{
	}

	bool Menu::compare(const Menu& other) const
	{
		if(!super::compare(other))
			return false;

		return true;
	}
}
