/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "imageButton.h"
#include "menu.h"
#include "text.h"

namespace Graph2D
{
	ImageButton::ImageButton()
		: leftFrameSize(0)
		, rightFrameSize(0)
		, clickEvent(NULL)
	{
		type = TYPE_IMAGE_BUTTON;
		//pageSize = 1;
		memset(frameTexture, NULL, sizeof(frameTexture));

		setFrameStyle(FRAME_STYLE_NOLINE|FRAME_STYLE_FLAT);
		setColor(1, 1, 1, 1, 0);
		setTouchable(true);
	}

	bool ImageButton::loadFrameTexture(const ImageButton::Layout direction, const std::string& filename, const bool pushed)
	{
		return loadTextureCore(&frameTexture[direction][pushed ? 1 : 0], filename);
	}

	void ImageButton::unloadFrameTexture(const ImageButton::Layout direction, const bool pushed)
	{
		unloadTextureCore(&frameTexture[direction][pushed ? 1 : 0]);
	}

	const Vector2& ImageButton::getFrameTexCoord(const ImageButton::Layout direction, const bool pushed) const
	{
		return frameTexCoord[direction][pushed ? 1 : 0];
	}

	void ImageButton::setFrameTexCoord(const ImageButton::Layout direction, const Vector2& frameTexCoord, const bool pushed)
	{
		this->frameTexCoord[direction][pushed ? 1 : 0] = frameTexCoord;
	}

	void ImageButton::setFrameTexCoord(const ImageButton::Layout direction, const float u, const float v, const bool pushed)
	{
		setFrameTexCoord(direction, Vector2(u, v), pushed);
	}

	void ImageButton::onSerialize(mana_stream* stream) const
	{
		super::onSerialize(stream);
		mana_stream_mark(stream);
	}

	void ImageButton::onDeserialize(mana_stream* stream)
	{
		super::onDeserialize(stream);
		mana_stream_check(stream);
	}

	void ImageButton::onDraw(const DrawRect& drawRect)
	{
		super::onDraw(drawRect);

		if(getVisible() == false || getOwnerDraw() == true)
			return;

		const int index = (getFrameStyle() & FRAME_STYLE_PUSHED || getEnable() == false) ? 1 : 0;
		if((frameTexture[0][index] == NULL) && (frameTexture[1][index] == NULL) && (frameTexture[2][index] == NULL))
			return;

		Color baseColor = getDrawColor(drawRect);
		if(baseColor.a <= 0.0f)
			return;

		GraphicDevice::setBlendMode(static_cast<GraphicDevice::BlendMode>(blendMode));
		GraphicDevice::enableTexture(true);

		//////////////////////////////////////////////////////////////////////
		const Vector2 lt = drawRect.getDrawLeftTopPosition();
		const Vector2 rb = drawRect.getDrawRightBottomPosition();

		const float leftFrameSize = this->leftFrameSize * drawRect.scale.x;
		const float rightFrameSize = this->rightFrameSize * drawRect.scale.x;

		// 左頂点
		const Vector2 vltlt(lt.x                 , lt.y);
		const Vector2 vltrt(lt.x + leftFrameSize , lt.y);
		const Vector2 vltlb(lt.x                 , rb.y);
		const Vector2 vltrb(lt.x + leftFrameSize , rb.y);
		// 右頂点
		const Vector2 vrtlt(rb.x - rightFrameSize, lt.y);
		const Vector2 vrtrt(rb.x                 , lt.y);
		const Vector2 vrtlb(rb.x - rightFrameSize, rb.y);
		const Vector2 vrtrb(rb.x                 , rb.y);
		// 中頂点
		const Vector2 vlblt(lt.x + leftFrameSize , lt.y);
		const Vector2 vlbrt(rb.x - rightFrameSize, lt.y);
		const Vector2 vlblb(lt.x + leftFrameSize , rb.y);
		const Vector2 vlbrb(rb.x - rightFrameSize, rb.y);
		// 内部のサイズ
		const float frameWidth = vlbrt.x - vlblt.x;
		const float frameHeight = vlblb.y - vlblt.y;

		//////////////////////////////////////////////////////////////////////
		// 描画
		drawFrame(frameTexture[0][index], frameTexCoord[0][index], vlblt, vlbrt, vlblb, vlbrb, baseColor, frameWidth, frameHeight);
		drawFrame(frameTexture[1][index], frameTexCoord[1][index], vltlt, vltrt, vltlb, vltrb, baseColor, leftFrameSize, frameHeight);
		drawFrame(frameTexture[2][index], frameTexCoord[2][index], vrtlt, vrtrt, vrtlb, vrtrb, baseColor, rightFrameSize, frameHeight);
	}

	void ImageButton::drawFrame(Texture* texture, const Vector2& uv, const Vector2& v0, const Vector2& v1, const Vector2& v2, const Vector2& v3, const Color& color, const float width, const float height)
	{
		if(texture)
		{
			texture->bind();

			const Vector2 textureSize = texture->getTextureSize();
			const Vector2 uv0((uv.x        ) / textureSize.x, (uv.y         ) / textureSize.y);
			const Vector2 uv1((uv.x + width) / textureSize.x, (uv.y         ) / textureSize.y);
			const Vector2 uv2((uv.x        ) / textureSize.x, (uv.y + height) / textureSize.y);
			const Vector2 uv3((uv.x + width) / textureSize.x, (uv.y + height) / textureSize.y);

			GraphicDevice::begin(GraphicDevice::TYPE_TEXTURE_COORDINATE, GraphicDevice::TRIANGLES, color);
			GraphicDevice::addTextureCoord(uv0);
			GraphicDevice::addTextureCoord(uv1);
			GraphicDevice::addTextureCoord(uv2);
			GraphicDevice::addTextureCoord(uv1);
			GraphicDevice::addTextureCoord(uv2);
			GraphicDevice::addTextureCoord(uv3);
			GraphicDevice::addVertex(v0);
			GraphicDevice::addVertex(v1);
			GraphicDevice::addVertex(v2);
			GraphicDevice::addVertex(v1);
			GraphicDevice::addVertex(v2);
			GraphicDevice::addVertex(v3);
			GraphicDevice::end();
		}
	}

	bool ImageButton::onTouchesBegan(const Vector2& localPosition)
	{
		// マウスをキャプチャー
		setMouseCapture();
		// 押されたアニメーションを開始
		if(!setAnimation("push"))
		{
			// スタイルを押された状態に変更
			setFrameStyle(getFrameStyle() | FRAME_STYLE_PUSHED);
		}
		// 親クラスを実行
		return super::onTouchesBegan(localPosition);
	}

	bool ImageButton::onTouchesEnded(const Vector2& localPosition)
	{
		// マウスを解放
		Component::releaseMouseCapture();
		// 放したアニメーションを開始
		if(!setAnimation("release"))
		{
			// スタイルを押されていない状態に戻す
			setFrameStyle(getFrameStyle() & ~FRAME_STYLE_PUSHED);
		}
		// クリックイベントを実行
		if(clickEvent)
		{
			clickEvent->onClick(this);
		}
		// 親クラスを実行
		return super::onTouchesEnded(localPosition) || clickEvent != NULL;
	}

	bool ImageButton::onTouchesCancelled(const Vector2& localPosition)
	{
		// マウスを解放
		Component::releaseMouseCapture();
		// 放したアニメーションを開始
		if(!setAnimation("release"))
		{
			// スタイルを押されていない状態に戻す
			setFrameStyle(getFrameStyle() & ~FRAME_STYLE_PUSHED);
		}
		// 親クラスを実行
		return super::onTouchesCancelled(localPosition);
	}

	bool ImageButton::compare(const ImageButton& other) const
	{
		// 親クラスを実行
		if(!super::compare(other))
			return false;

		//ClickEvent* clickEvent;

		return true;
	}
}
