/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#if !defined(___GRAPH2D_DESKTOP_H___)
#define ___GRAPH2D_DESKTOP_H___

#include "base.h"
#include "color.h"
#include "vector2.h"

#include "abort.h"
#include "data.h"
#include "stopwatch.h"

#include "component.h"
#include "container.h"
#include "scene.h"

#include "soundPlayer.h"

#include <libmana.h>
#include <cstdarg>
#include <string>

namespace Graph2D
{
	class AudioPlayer;
	namespace Sound
	{
		class DataManager;
		class Player;
		class MusicPlayer;
	}

	// TODO:Z[uƃ[h@\

	typedef enum SoundPlayerID
	{
		SOUND_PLAYER_ID_BGM = 0,
		SOUND_PLAYER_ID_SE,
		SOUND_PLAYER_ID_SYSTEM,
		SOUND_PLAYER_ID_SIZE
	}SoundPlayerID;

	typedef struct DebugMessageInfomation
	{
		int x;
		int y;
		std::string string;
		DebugMessageInfomation* next;
	}DebugMessageInfomation;

	class Desktop : public Base
	{
		Scene* scene;										//!< ݂̃V[̃|C^
		Scene* reserveScene;								//!< \񂳂ꂽV[̃|C^
		Scene* prepareScene;								//!< ݏ̃V[
		Scene* backScene;									//!< ̃V[
		
		Color fadeColor;
		Color fadeDeltaColor;
		Color fadeTargetColor;
		float fadeColorTime;
		bool fadeColorEnable;
		
		Stopwatch lastUpdateTime;
		float refreshTime;
		
		Sound::DataManager* dataManager[SOUND_PLAYER_ID_SIZE];
		Sound::Player* player;
		Sound::MusicPlayer* musicPlayer;
		
		std::string requestDeserializeFileName;
		
		bool userControl;
		bool releaseBackScene;
		
#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
		DebugMessageInfomation* debugMessageInfomation;
		Texture* debugMessageFontTexture;
		Stopwatch lastIntervalTime;
#endif
		
		static Desktop* instance;

	protected:
		virtual ~Desktop();

	public:
		static void initialize(void* buffer, size_t size)
		{
			if(instance == NULL)
			{
				g2d_initialize(buffer, size);

				instance = new Desktop();
			}
		}

		static void finalize()
		{
			if(instance)
			{
				instance->release();
				instance = NULL;

				g2d_finalize();
			}
		}

		static Desktop& getInstance()
		{
			return *instance;
		}

		Desktop();

		void initializeResources();

		Component* find(const unsigned int identification) const
		{
			return Component::find(identification);
		}

		const Vector2& getSize() const
		{
			return GraphicDevice::getScreenSize();
		}

		bool rangeOfVisibility(const Vector2& lt, const Vector2& rb) const
		{
			const Vector2& size = GraphicDevice::getScreenSize();
			return lt.x < size.x && rb.x >= 0 && lt.y < size.y && rb.y >= 0;
		}

		Scene& getScene() const								//!< ݂̃V[擾
		{
			checkScene();
			return *scene;
		}

		void setScene(Scene* scene)						//!< V[ݒ肷
		{
			if(reserveScene)
				reserveScene->release();
			reserveScene = scene;
			if(reserveScene)
				reserveScene->retain();
		}

		Scene* getPrepareScene() const						//!< ݏ̃V[擾
		{
			return prepareScene;
		}

		void setPrepareScene(Scene* scene)					//!< ̃V[ݒ肷
		{
			if(prepareScene)
				prepareScene->release();
			prepareScene = scene;
			if(prepareScene)
				prepareScene->retain();
		}

		void swapScene(const bool releaseBackScene)
		{
			Scene* temporary = scene;
			scene = backScene;
			backScene = temporary;
			this->releaseBackScene = releaseBackScene;
		}

		Scene* getBackScene() const							//!< ̃V[擾
		{
			return backScene;
		}

		const Color& getFadeColor() const
		{
			return fadeColor;
		}

		void setFadeColor(const Color& color, const float second);

		bool loadScript(const std::string& filename)
		{
			return scene ? scene->loadScript(filename) : false;
		}

		void unloadScript()
		{
			if(scene)
				scene->unloadScript();
		}

		mana* getScript() const
		{
			return scene ? scene->getScript() : NULL;
		}

		bool getUserControl() const
		{
			return userControl;
		}

		void setUserControl(const bool userControl)
		{
			this->userControl = userControl;
		}

		static float getMinimumTime()
		{
			return 2.0f / 60.0f;
		}

		static float getMostDelayTime()
		{
#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
			return 2.0f / 60.0f;
#else
			return 8.0f / 60.0f;
#endif
		}

		float getRefreshTime() const
		{
			return refreshTime;
		}

		void setRefreshTime(const float time)
		{
			refreshTime = time;
		}

		Sound::DataManager* getSoundDataManager(const SoundPlayerID index)
		{
			return dataManager[index];
		}

		void setSoundDataManager(const SoundPlayerID index, Sound::DataManager* dataManager)
		{
			this->dataManager[index] = dataManager;
		}

		Sound::Player* getSoundPlayer()
		{
			return player;
		}

		void setSoundPlayer(Sound::Player* player)
		{
			this->player = player;
		}

		Sound::MusicPlayer* getMusicPlayer()
		{
			return musicPlayer;
		}

		void setMusicPlayer(Sound::MusicPlayer* musicPlayer)
		{
			this->musicPlayer = musicPlayer;
		}

		void serialize(mana_stream* stream) const;

		void requestDeserialize(const std::string& filename)
		{
			requestDeserializeFileName = filename;
		}

		void updateAndDraw();

		void touchesBegan(const Vector2& position);
		void touchesMoved(const Vector2& position);
		void touchesEnded(const Vector2& position);
		void touchesCancelled(const Vector2& position);

#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
		void print(const int x, const int y, const char* format, ...);
#endif

	private:
		void changeScene();								//!< V[̏

		void checkScene() const
		{
			if(scene == NULL)
			{
				abort(ERROR_CODE_ILLEGAL_SCENE_OBJECT_DETECTED);
			}
		}

		Component* deserialize(mana_stream* stream) const;

#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
		void cleanDebugMessage();
		void drawDebugMessage();
		void addDebugMessage(const int x, const int y, const std::string& message);
#endif
	};
}

#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
#define GRAPH2D_TRACE(x, y, m)	if(0)Graph2D::print
#else
#define GRAPH2D_TRACE(x, y, m)	Graph2D::print
#endif

#endif
