/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *	 claim that you wrote the original software. If you use this software
 *	 in a product, an acknowledgment in the product documentation would be
 *	 appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *	 misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *	 distribution.
 */

#include "curve.h"
#include <assert.h>
#include <math.h>

namespace Graph2D
{
	////////////////////////////////////////////////////////////////////////////////
	Curve::Curve()
	{
		rewind();
	}

	void Curve::rewind()
	{
		currentTime = 0;
	}

	void Curve::update(const float deltaTime)
	{
		currentTime += deltaTime;
	}

	bool Curve::finish()
	{
		return currentTime >= getLength();
	}

	////////////////////////////////////////////////////////////////////////////////
	Curve2::Curve2()
	{
		normalized = false;
	}

	Curve2::~Curve2()
	{
		release();
	}

	void Curve2::release()
	{
		for(int i = 0; i < curveKeys.size(); i++)
		{
			delete curveKeys[i];
		}
		curveKeys.release();

		rewind();
	}

	CurveKey2* Curve2::getKeyFrame(const int index)
	{
		return curveKeys[index];
	}

	void Curve2::addKeyFrame(CurveKey2* curveKey)
	{
		assert(curveKey);
		if(curveKey)
		{
			curveKeys.push(curveKey);
			normalized = false;
		}
	}

	float Curve2::getLength()
	{
		const int keyCount = getKeyCount();
		if(keyCount <= 1)
			return 0;

		if(!normalized)
			normalize();

		return curveKeys[0]->getTimeSpan(*curveKeys[keyCount - 1]);
	}

	int Curve2::getKeyCount() const
	{
		return curveKeys.size();
	}

	Vector2 Curve2::getValue()
	{
		const int keyCount = getKeyCount();

		if(keyCount <= 0)
			return Vector2(0, 0);

		if(!normalized)
			normalize();

		CurveKey2* skey  = NULL;
		CurveKey2* ekey  = NULL;
		CurveKey2* sskey = NULL;
		CurveKey2* eekey = NULL;

		for(int i = 0; i < keyCount; i++)
		{
			if(curveKeys[i]->getTime() > currentTime)
			{
				ekey  = curveKeys[i];
				skey  = i > 0 ? curveKeys[i - 1] : ekey;
				sskey = i > 1 ? curveKeys[i - 2] : skey;
				eekey = i < (keyCount - 1) ? curveKeys[i + 1] : ekey;
				break;
			}
		}

		if(!ekey)
		{
			return curveKeys[keyCount - 1]->getVector();
		}
		else if(skey == ekey)
		{
			return curveKeys[0]->getVector();
		}

		const Vector2& p0 = skey->getVector();
		const Vector2& p1 = ekey->getVector();
		const float st = skey->getTime();
		const float et = ekey->getTime();
		const float nt = currentTime;
		const float t  = (nt - st) / (et - st);

		switch(skey->getInterpolationType())
		{
		case INTERPOLATION_CONST:
			// 定数
			return p0;

		case INTERPOLATION_LINER:
			// 線形補間
			return p0 + (p1 - p0) * t;

		case INTERPOLATION_HERMITE:
			// エルミートスプライン補間
			{
				const Vector2& p3 = eekey->getVector();
				Vector2 v0 = p1 - p0;
				Vector2 v1 = p3 - p1;
				return (p0 * 2.0f - p1 * 2.0f + v0 + v1) * (t * t * t) + (p0 * -3.0f + p1 * 3.0f - v0 * 2.0f - v1) * (t * t) + v0 * t + p0;
	//			return (2.0f * p0 - 2.0f * p1 + v0 + v1) * (t * t * t) + (-3.0f * p0 + 3.0f * p1 - 2.0f * v0 - v1) * (t * t) + v0 * t + p0;
			}

		case INTERPOLATION_CATMULL:
			// Catmull-Romスプライン補間
			{
				const Vector2& p2 = sskey->getVector();
				const Vector2& p3 = eekey->getVector();
				Vector2 v0 = (p1 - p2) * .5f;
				Vector2 v1 = (p3 - p0) * .5f;
				return (p0 * 2.0f - p1 * 2.0f + v0 + v1) * (t * t * t) + (p0 * -3.0f + p1 * 3.0f - v0 * 2.0f - v1) * (t * t) + v0 * t + p0;
			}

		case INTERPOLATION_SLERP:
			// 球面線形補間
			{
				// theatは本来内積で求めるべきだが、クオータニオンではないのでとりあえず仮の値
				const float theta = 3.14159265358979323846f / 2.f;
				return (p0 * sinf(theta * (1 - t)) + p1 * sinf(theta * t)) / sinf(theta);
			}

		default:
			// それ以外
			return Vector2(0, 0);
		}
	}

	void Curve2::normalize()
	{
		const int keyCount = getKeyCount();
		int gap  = keyCount;
		int done = 0;
		while(gap > 1 || done == 0)
		{
			gap = (gap * 10) / 13;
			if(gap == 0)
				gap = 1;
			done = 1;
			for(int i = 0; i < keyCount - gap; ++i)
			{
				if(curveKeys[i]->getTime() > curveKeys[i + gap]->getTime())
				{
					CurveKey2* curveKey = curveKeys[i + gap];
					curveKeys.set(i + gap, curveKeys[i]);
					curveKeys.set(i, curveKey);
					done = 0;
				}
			}
		}
		normalized = true;
	}

	////////////////////////////////////////////////////////////////////////////////
	Curve4::Curve4()
	{
		normalized = false;
	}

	Curve4::~Curve4()
	{
		release();
	}

	void Curve4::release()
	{
		for(int i = 0; i < curveKeys.size(); i++)
		{
			delete curveKeys[i];
		}
		curveKeys.release();

		rewind();
	}

	CurveKey4* Curve4::getKeyFrame(const int index)
	{
		return curveKeys[index];
	}

	void Curve4::addKeyFrame(CurveKey4* curveKey)
	{
		assert(curveKey);
		if(curveKey)
		{
			curveKeys.push(curveKey);
			normalized = false;
		}
	}

	float Curve4::getLength()
	{
		const int keyCount = getKeyCount();
		if(keyCount <= 1)
			return 0;

		if(!normalized)
			normalize();

		return curveKeys[0]->getTimeSpan(*curveKeys[keyCount - 1]);
	}

	int Curve4::getKeyCount() const
	{
		return curveKeys.size();
	}

	Color Curve4::getValue()
	{
		const int keyCount = getKeyCount();

		if(keyCount <= 0)
			return Color(0, 0, 0, 0);

		if(!normalized)
			normalize();

		CurveKey4* skey  = NULL;
		CurveKey4* ekey  = NULL;
		CurveKey4* sskey = NULL;
		CurveKey4* eekey = NULL;

		for(int i = 0; i < keyCount; i++)
		{
			if(curveKeys[i]->getTime() > currentTime)
			{
				ekey  = curveKeys[i];
				skey  = i > 0 ? curveKeys[i - 1] : ekey;
				sskey = i > 1 ? curveKeys[i - 2] : skey;
				eekey = i < (keyCount - 1) ? curveKeys[i + 1] : ekey;
				break;
			}
		}

		if(!ekey)
		{
			return curveKeys[keyCount - 1]->getColor();
		}
		else if(skey == ekey)
		{
			return curveKeys[0]->getColor();
		}

		const Color& p0 = skey->getColor();
		const Color& p1 = ekey->getColor();
		const float st = skey->getTime();
		const float et = ekey->getTime();
		const float nt = currentTime;
		const float t  = (nt - st) / (et - st);

		switch(skey->getInterpolationType())
		{
		case INTERPOLATION_CONST:
			// 定数
			return p0;

		case INTERPOLATION_LINER:
			// 線形補間
			return p0 + (p1 - p0) * t;

		case INTERPOLATION_HERMITE:
			// エルミートスプライン補間
			{
				const Color& p3 = eekey->getColor();
				Color v0 = p1 - p0;
				Color v1 = p3 - p1;
				return (p0 * 2.0f - p1 * 2.0f + v0 + v1) * (t * t * t) + (p0 * -3.0f + p1 * 3.0f - v0 * 2.0f - v1) * (t * t) + v0 * t + p0;
			}

		case INTERPOLATION_CATMULL:
			// Catmull-Romスプライン補間
			{
				const Color& p2 = sskey->getColor();
				const Color& p3 = eekey->getColor();
				Color v0 = (p1 - p2) * .5f;
				Color v1 = (p3 - p0) * .5f;
				return (p0 * 2.0f - p1 * 2.0f + v0 + v1) * (t * t * t) + (p0 * -3.0f + p1 * 3.0f - v0 * 2.0f - v1) * (t * t) + v0 * t + p0;
			}

		case INTERPOLATION_SLERP:
			// 球面線形補間
			{
				// theatは本来内積で求めるべきだが、クオータニオンではないのでとりあえず仮の値
				const float theta = 3.14159265358979323846f / 2.f;
				return (p0 * sinf(theta * (1 - t)) + p1 * sinf(theta * t)) / sinf(theta);
			}

		default:
			// それ以外
			return Color(0, 0, 0, 0);
		}
	}

	void Curve4::normalize()
	{
		const int keyCount = getKeyCount();
		int gap  = keyCount;
		int done = 0;
		while(gap > 1 || done == 0)
		{
			gap = (gap * 10) / 13;
			if(gap == 0)
				gap = 1;
			done = 1;
			for(int i = 0; i < keyCount - gap; ++i)
			{
				if(curveKeys[i]->getTime() > curveKeys[i + gap]->getTime())
				{
					CurveKey4* curveKey = curveKeys[i + gap];
					curveKeys.set(i + gap, curveKeys[i]);
					curveKeys.set(i, curveKey);
					done = 0;
				}
			}
		}
		normalized = true;
	}
}
