# coding: UTF-8

require 'base64'
require 'time'

require 'mailutils/mail'
require 'mailutils/mail_address'
require 'mailutils/mail_attachment'
require 'mailutils/mime'

#=メールの解析と送信用メールデータを作成するメールプロセッサー
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: mail_processor.rb 516 2012-02-09 07:09:11Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
#
# 汎用ではなく簡易のものです。
#
# - メール種別（Content-Type）の対応状況
# -- サポート対象
# --- テキストメール（text/plain）をサポートします。
# --- HTML メール（multipart/related, multipart/alternative）をサポートします。
# --- 添付ファイル付きメール（multipart/mixed）をサポートします。
# -- 未サポート
# --- 絵文字入りメール／デコレーションメール。
# --- その他、サポート対象以外
#
# - エンコードの対応状況
# -- タイトル／送信元／送信先は、Ｂ符号化およびＱ符号化の両方をサポートします。
# -- 本文は Content-Transfer-Encoding ヘッダーの値が 7bit/8bit/base64/quoted-printable のものをサポートします。
# -- 添付ファイルは Content-Transfer-Encoding ヘッダーの値が base64 をサポートします。
class MailProcessor

	# メールを解析します。
	def MailProcessor.parse(mail, to_encoding = nil)
		i = 0
		buffers = mail.split(/\r+\n|\r|\n/)

		# ヘッダーを解析する
		(header, line) = parse_header(buffers, 0)

		if header['content-type'] != nil then
			# 添付ファイルが無い場合
			if %r!text/plain! =~ header['content-type'] then
				content = MailProcessor.parse_text_plain(header, line, buffers, to_encoding)
				(subject, from_address, to_addresses) = MailProcessor.get_mail_info(header, to_encoding)
				return Mail.new(subject, from_address, to_addresses, content, [])
			# 添付ファイルがある、もしくはHTML メールの場合
			elsif %r!multipart/(mixed|alternative|related);.+boundary="?(.+?)("| |$)! =~ header['content-type'] then
				multipart_type = $1
				boundary = $2
				(main_header, content, attachments) = MailProcessor.parse_multipart(multipart_type, header, line, boundary, buffers, to_encoding)
				(subject, from_address, to_addresses) = MailProcessor.get_mail_info(header, to_encoding)
				return Mail.new(subject, from_address, to_addresses, content, attachments)
			end
		end

		raise ArgumentError.new("サポートしていないメールを引数に指定しました（未対応な Content-Type[#{header['content-type']}] を持つメール）。")
	end

	# 送信用メールデータを作成します。
	def MailProcessor.make(mail, date = nil, to_header_encoding = "ISO-2022-JP", to_content_encoding = "UTF-8", mime = Mime.new)
		if mail.attachments == nil || mail.attachments.size == 0 then
			return MailProcessor.make_text_plain(mail, date, to_header_encoding, to_content_encoding)
		else
			return MailProcessor.make_multipart_mixed(mail, date, to_header_encoding, to_content_encoding, mime)
		end
	end

	def MailProcessor.load_attachment(filename)
		File.open(filename, 'rb') do |file|
			return MailAttachment.new(File.basename(filename), file.read(nil))
		end
	end

	def MailProcessor.decode_header(content, to_encoding = nil)
		if %r!^=\?(.+?)\?([bqBQ])\?(.+)\?=$! =~ content then
			# ＢもしくはＱ符号化されている
			from_encoding = $1
			encoding_type = $2
			encoded_content = $3
			if encoding_type.upcase == "B" then
				decoded_content = Base64::decode64(encoded_content)
			elsif encoding_type.upcase == "Q" then
				decoded_content = encoded_content.unpack("M")[0].encode(to_encoding, from_encoding)
			end

			if to_encoding != nil then
				return decoded_content.encode(to_encoding, from_encoding)
			else
				return decoded_content.encode(from_encoding, from_encoding)
			end
		else
			# 特に符号化されていない
			return content
		end
	end

	def MailProcessor.encode_header(content, to_encoding = "ISO-2022-JP")
		if content.encoding == nil || content.encoding.to_s == "ASCII" then
			return content
		else
			return "=?#{to_encoding}?B?#{Base64::encode64(content.encode(to_encoding)).chomp}?="
		end
	end

	def MailProcessor.parse_header(buffers, start_line)
		header = {}

		i = start_line
		while i < buffers.size
			line = buffers[i].chomp

			if /^([a-zA-Z][a-zA-Z0-9\-]+): *(.+)$/ =~ line then
				header_name = $1.downcase
				header_value = $2
				if header.has_key?(header_name) then
					header[header_name] << header_value
				else
					header[header_name] = header_value
				end
			elsif line == '' then
				break
			else
				header[header_name] << line
			end

			i = i + 1
		end

		return [header, i]
	end

	def MailProcessor.parse_text_plain(header, line, buffers, to_encoding)
		i = line + 1
		content = ''
		while i < buffers.size
			content = content + buffers[i] + "\n"

			i = i + 1
		end

		return MailProcessor.decode_content(header, content, to_encoding)
	end

	def MailProcessor.parse_multipart(multipart_type, header, line, boundary, buffers, to_encoding)
		main_header = nil
		main_content = nil

		result = []
		i = line + 1
		while i < buffers.size
			if buffers[i] == "--#{boundary}--" then
				break
			elsif buffers[i] == "--#{boundary}" then
				if i + 1 == buffers.size || buffers[i + 1] == '' then
					break
				else
					(part_header, part_line) = MailProcessor.parse_header(buffers, i + 1)
					i = part_line + 1
					if %r!^ *([a-z]+/[a-z]+)! =~ part_header['content-type'] then
						type = $1
					end

					content = ''
					while i < buffers.size && /^--#{boundary}/ !~ buffers[i]
						if %r!text/plain! =~ type then
							if part_header['content-type'] != nil && part_header['content-transfer-encoding'] != nil && /quoted-printable/ =~ part_header['content-transfer-encoding'].downcase then
								if %r!charset="?([0-9a-zA-Z_\-]+)"?! =~ part_header['content-type'] then
									from_encoding = $1
									content = content + buffers[i].unpack("M")[0].encode(to_encoding, from_encoding) + "\n"
								else
									content = content + buffers[i].unpack("M")[0].encode(to_encoding) + "\n"
								end
							else
								content = content + buffers[i] + "\n"
							end
						else
							content = content + buffers[i]
						end

						i = i + 1
					end

					case multipart_type
					when 'alternative'
						if %r!text/plain! =~ type then
							main_header = part_header
							main_content = MailProcessor.decode_content(part_header, content, to_encoding)
						end
					when 'mixed'
						if main_content == nil && %r!text/plain! =~ type then
							main_header = part_header
							main_content = MailProcessor.decode_content(part_header, content, to_encoding)
						else
							result.push(MailAttachment.new(MailProcessor.decode_filename(part_header, to_encoding), MailProcessor.decode_filecontents(part_header, content)))
						end
					else
						if part_header['content-type'] != nil && %r!multipart/(.+);.+boundary="?(.+?)("| |$)! =~ part_header['content-type'] then
							multipart_type = $1
							part_boundary = $2
							(multipart_header, multipart_content, attachments) = MailProcessor.parse_multipart(multipart_type, part_header, line, part_boundary, buffers, to_encoding)
							attachments.each do |attachment|
								result.push(attachment)
							end
						end
					end
				end
			else
				i = i + 1
			end
		end

		if main_header != nil && main_content != nil then
			return [main_header, main_content, result]
		else
			raise ArgumentError.new('サポートしていないメールを引数に指定しました（text/plain のコンテンツを持っていないメール）。')
		end
	end

	def MailProcessor.make_text_plain(mail, date, to_header_encoding, to_content_encoding)
			result = ""

			# タイトル
			result << "Subject: #{MailProcessor.encode_header(mail.subject, to_header_encoding)}\n"

			# From
			result << "From: #{MailProcessor.make_address_content(mail.from_address, to_header_encoding)}\n"

			# To
			result << "To: #{MailProcessor.make_addresses_content(mail.to_addresses, to_header_encoding)}\n"

			# 時刻
			if date != nil then
				result << "Date: #{date.rfc2822}\n"
			end

			# 本文
			result << %Q!Content-Type: text/plain; charset="#{to_content_encoding}"\n!
			result << "Content-Transfer-Encoding: base64\n"
			result << "\n"
			result << Base64::encode64(mail.content)

			return result
	end

	def MailProcessor.make_multipart_mixed(mail, date, to_header_encoding, to_content_encoding, mime)
			result = ""

			# タイトル
			result << "Subject: #{MailProcessor.encode_header(mail.subject, to_header_encoding)}\n"

			# From
			result << "From: #{MailProcessor.make_address_content(mail.from_address, to_header_encoding)}\n"

			# To
			result << "To: #{MailProcessor.make_addresses_content(mail.to_addresses, to_header_encoding)}\n"

			# 時刻
			if date != nil then
				result << "Date: #{date.rfc2822}\n"
			end

			# 境界の定義
			# ※バウンダリの決定方針：本文は Base64 でエンコーディングしてしまうため境界にハイフンを入れた時点で問題ない
			boundary = "--Boundary"
			result << %Q!Content-Type: multipart/mixed; boundary="#{boundary}"\n!
			result << "\n"
			result << "--#{boundary}\n"

			# 本文
			result << %Q!Content-Type: text/plain; charset="#{to_content_encoding}"\n!
			result << "Content-Transfer-Encoding: base64\n"
			result << "\n"
			result << "#{Base64::encode64(mail.content)}\n"
			result << "--#{boundary}\n"

			# 添付ファイル
			mail.attachments.each do |attachment|
				result << %Q!Content-Type: #{mime.mime(attachment.filename)}; name="#{MailProcessor.decode_header(attachment.filename, to_header_encoding)}"\n!
				result << "Content-Transfer-Encoding: base64\n"
				result << %Q!Content-Disposition: attachment; filename="#{MailProcessor.decode_header(attachment.filename, to_header_encoding)}"\n!
				result << "\n"
				result << "#{Base64::encode64(attachment.content)}\n"
				result << "--#{boundary}\n"
			end

			return result
		end

	def MailProcessor.get_mail_info(header, to_encoding = nil)
		# Subject
		subject = MailProcessor.decode_header(header['subject'], to_encoding)

		# From
		from_address = MailProcessor.parse_mail_address(header['from'], to_encoding)

		# To
		to_addresses = []
		header['to'].split(',').each do |to|
			to_addresses.push(MailProcessor.parse_mail_address(to, to_encoding))
		end

		return [subject, from_address, to_addresses]
	end

	def MailProcessor.parse_mail_address(buffer, to_encoding = nil)
		if /^ *"?(.*?)"? *<(.+)> *$/ =~ buffer then
			return MailAddress.new($2, MailProcessor.decode_header($1, to_encoding))
		else
			return MailAddress.new(buffer)
		end
	end

	def MailProcessor.decode_content(header, content, to_encoding = nil)
		if %r!charset="?([0-9a-zA-Z_\-]+)"?! =~ header['content-type'] then
			from_encoding = $1
		else
			from_encoding = nil
		end

		if header['content-transfer-encoding'] != nil then
			case header['content-transfer-encoding'].downcase
			when '7bit'
			when '8bit'
			when 'base64'
				content = Base64::decode64(content)
			when 'quoted-printable'
				return content.gsub(/\r+\n/, "\n")
			else
				raise EncodingError.new('7bit/8bit/base64/quoted-printable 以外の Content-Transfer-Encoding を持つメールはサポートしていません。')
			end
		end

		if from_encoding != nil then
			if to_encoding != nil then
				return content.encode(to_encoding, from_encoding).gsub(/\r+\n/, "\n")
			else
				return content.encode(from_encoding, from_encoding).gsub(/\r+\n/, "\n")
			end
		else
			return content.gsub(/\r+\n/, "\n")
		end
	end

	def MailProcessor.decode_filename(header, to_encoding)
		if /name="(.+?)"/ =~ header['content-type'] then
			return MailProcessor.decode_header($1, to_encoding)
		else
			return nil
		end
	end

	def MailProcessor.decode_filecontents(header, contents)
		if header['content-transfer-encoding'] == nil || 'base64' != header['content-transfer-encoding'].downcase then
			raise EncodingError.new("base64 以外の Content-Transfer-Encoding を持つ添付ファイルはサポートしていません。(#{header['content-transfer-encoding']})")
		end

		return Base64::decode64(contents)
	end

	def MailProcessor.make_address_content(address, to_encoding = nil)
		if address.name != nil then
			if to_encoding != nil then
				return %Q!"#{MailProcessor.encode_header(address.name, to_encoding)}" <#{address.address}>!
			else
				return %Q!"#{MailProcessor.encode_header(address.name)}" <#{address.address}>!
			end
		else
			return address.address
		end
	end

	def MailProcessor.make_addresses_content(addresses, to_encoding = nil)
		result = ''

		i = 0
		while i < addresses.size
			if i != 0 then
				result << ', '
			end

			result << MailProcessor.make_address_content(addresses[i])

			i = i + 1
		end

		return result
	end

	def MailProcessor.make_address_list(addresses)
		result = []
		addresses.each do |address|
			result.push(address.address)
		end
		return result
	end

end
