# coding: UTF-8

ROOT_PATH = File.dirname(File.expand_path(__FILE__))

$:.unshift(ROOT_PATH + "/")
$:.unshift(ROOT_PATH + "/../")
$:.unshift(ROOT_PATH + "/../../main/")
$:.unshift(ROOT_PATH + "/../../main/lib")

require 'base64'
require 'net/pop'
require 'net/smtp'

require 'conf/01_system/ruby_version_absorber'
require 'conf/setting'
require 'mailutils/mail_processor'
require 'testutils/mail_testutils'

require 'minitest/unit'

MiniTest::Unit.autorun

#=MailProcessor のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_mailutils.rb 335 2011-10-15 06:38:51Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestMailProcessor < MiniTest::Unit::TestCase

	def test_escaped_prefix_subject
		mail = Mail.new("Re:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Fw:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Fwd:title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Re>title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject

		mail = Mail.new("Re > title", nil, nil, nil, [])
		assert_equal "title", mail.escaped_prefix_subject
	end

	def test_encode_header1
		assert_equal '=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', MailProcessor.encode_header('テスト')
	end

	def test_decode_header2
		assert_equal 'テスト', MailProcessor.decode_header('=?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=', 'UTF-8')
	end

	def test_decode_header3
		assert_equal 'm2w:post:1', MailProcessor.decode_header('=?iso-8859-1?q?m2w:post:1?=', 'UTF-8')
	end

	def test_parse_address_content1
		address = MailProcessor.parse_mail_address('name <abcdef@ghjik.lmnop>')
		assert_equal('name', address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_content2
		address = MailProcessor.parse_mail_address('"name" <abcdef@ghjik.lmnop>')
		assert_equal('name', address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_content3
		address = MailProcessor.parse_mail_address("'name' <abcdef@ghjik.lmnop>")
		assert_equal("'name'", address.name)
		assert_equal('abcdef@ghjik.lmnop', address.address)
	end

	def test_parse_address_conten4
		address = MailProcessor.parse_mail_address("<w-w.w-w.0310@docomo.ne.jp>")
		assert_equal("", address.name)
		assert_equal('w-w.w-w.0310@docomo.ne.jp', address.address)
	end

	def test_make1
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info'),
			[MailAddress.new('test_receiver1@ownway.info')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = ''
		expected << "Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=\n"
		expected << "From: test_sender@ownway.info\n"
		expected << "To: test_receiver1@ownway.info\n"
		expected << "Date: Sun, 02 Jan 2011 03:04:05 +0900\n"
		expected << %Q!Content-Type: text/plain; charset="UTF-8"\n!
		expected << "Content-Transfer-Encoding: base64\n"
		expected << "\n"
		expected << "44OG44K544OI\n"

		assert_equal expected, result
	end

	def test_make2
		result = MailProcessor.make(Mail.new(
			'テスト',
			MailAddress.new('test_sender@ownway.info', 'テスト送信者'),
			[MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')],
			'テスト'), Time.local(2011, 1, 2, 3, 4, 5, 6))

		expected = ''
		expected << "Subject: =?ISO-2022-JP?B?GyRCJUYlOSVIGyhC?=\n"
		expected << %Q!From: "=?ISO-2022-JP?B?GyRCJUYlOSVIQXc/LjxUGyhC?=" <test_sender@ownway.info>\n!
		expected << %Q!To: "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzEbKEI=?=" <test_receiver1@ownway.info>, "=?ISO-2022-JP?B?GyRCJUYlOSVIPHU/LjxUIzIbKEI=?=" <test_receiver2@ownway.info>\n!
		expected << "Date: Sun, 02 Jan 2011 03:04:05 +0900\n"
		expected << %Q!Content-Type: text/plain; charset="UTF-8"\n!
		expected << "Content-Transfer-Encoding: base64\n"
		expected << "\n"
		expected << "44OG44K544OI\n"

		assert_equal expected, result
	end

	def test_parse
		from_address = MailAddress.new('test_sender@ownway.info', 'テスト送信者')
		to_addresses = [MailAddress.new('test_receiver1@ownway.info', 'テスト受信者１'), MailAddress.new('test_receiver2@ownway.info', 'テスト受信者２')]
		attachments = [MailProcessor.load_attachment("#{ROOT_PATH}/data/eyes0367.jpg")]
		result = MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments), Time.local(2011, 1, 2, 3, 4, 5, 6))

		mail = MailProcessor.parse(result, 'UTF-8')
		assert_equal 'テスト', mail.subject
		assert_equal from_address.address, mail.from_address.address
		assert_equal from_address.name, mail.from_address.name
		assert_equal to_addresses.size, mail.to_addresses.size
		assert_equal to_addresses[0].address, mail.to_addresses[0].address
		assert_equal to_addresses[0].name, mail.to_addresses[0].name
		assert_equal to_addresses[1].address, mail.to_addresses[1].address
		assert_equal to_addresses[1].name, mail.to_addresses[1].name
		assert_equal 'テスト', mail.content
		assert_equal attachments.size, mail.attachments.size
		assert_equal attachments[0].filename, mail.attachments[0].filename
		assert_equal attachments[0].content, mail.attachments[0].content
	end

	if REAL_MAIL_TEST then
		def test_pseudo_send_receive
			# 送信用ユーティリティを準備する
			mail_testutils = MailTestUtils.new(
				SMTP_SERVER_ADDRESS,
				SMTP_SERVER_PORT,
				SMTP_SECURITY_TYPE,
				SMTP_SERVER_USER,
				SMTP_SERVER_PASSWORD,
				nil,
				nil,
				nil,
				nil,
				'UTF-8')

			# 受信用ユーティリティー１を準備する
			mail_testutils1 = MailTestUtils.new(
				nil,
				nil,
				nil,
				nil,
				nil,
				POP3_SERVER1_ADDRESS,
				POP3_SERVER1_PORT,
				POP3_SERVER1_USER,
				POP3_SERVER1_PASSWORD,
				'UTF-8')

			# 受信用ユーティリティー２を準備する
			mail_testutils2 = MailTestUtils.new(
				nil,
				nil,
				nil,
				nil,
				nil,
				POP3_SERVER2_ADDRESS,
				POP3_SERVER2_PORT,
				POP3_SERVER2_USER,
				POP3_SERVER2_PASSWORD,
				'UTF-8')

			# メールを送信する
			from_address = MailAddress.new(SMTP_SERVER_MAILADDRESS, 'テスト送信者')
			to_addresses = [MailAddress.new(POP3_SERVER1_MAILADDRESS, 'テスト受信者１'), MailAddress.new(POP3_SERVER2_MAILADDRESS, 'テスト受信者２')]
			attachments = [MailProcessor.load_attachment("#{ROOT_PATH}/data/eyes0367.jpg")]
			mail_testutils.sendmail(
				MailProcessor.make(Mail.new('テスト', from_address, to_addresses, 'テスト', attachments), Time.local(2011, 1, 2, 3, 4, 5, 6))
			)

			# サーバ１でメールを受信する
			mail = mail_testutils1.get_mail()

			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content

			# サーバ２でメールを受信する
			mail = mail_testutils2.get_mail()

			assert_equal 'テスト', mail.subject
			assert_equal from_address.address, mail.from_address.address
			assert_equal from_address.name, mail.from_address.name
			assert_equal to_addresses.size, mail.to_addresses.size
			assert_equal to_addresses[0].address, mail.to_addresses[0].address
			assert_equal to_addresses[0].name, mail.to_addresses[0].name
			assert_equal to_addresses[1].address, mail.to_addresses[1].address
			assert_equal to_addresses[1].name, mail.to_addresses[1].name
			assert_equal 'テスト', mail.content
			assert_equal attachments.size, mail.attachments.size
			assert_equal attachments[0].filename, mail.attachments[0].filename
			assert_equal attachments[0].content, mail.attachments[0].content
		end
	end

	def test_parse_real_mail
expected_content = <<"END_CONTENT"
メール本文１行目
メール本文２行目

空行入れてメール本文４行目
メール本文５行目

空行入れてメール本文７行目
メール本文８行目
END_CONTENT

		expected_attachments = [MailProcessor.load_attachment("#{ROOT_PATH}/data/eyes0367.jpg")]

		Dir::glob(ROOT_PATH + "/data/*.txt") do |filename|
			open(filename, 'rb') do |file|
				mail = MailProcessor.parse(file.read, 'UTF-8')
				assert_equal 'タイトル', mail.subject
				assert_equal 'test_sender@ownway.info', mail.from_address.address
				# assert_equal nil, mail.from_address.name
				assert_equal 1, mail.to_addresses.size
				assert_equal 'test_receiver1@ownway.info', mail.to_addresses[0].address
				# assert_equal nil, mail.to_addresses[0].name
				assert_equal expected_content.chomp(''), mail.content.chomp('')
				assert_equal expected_attachments.size, mail.attachments.size
				i = 0
				expected_attachments.each do |attachment|
					assert_equal attachment.filename, mail.attachments[i].filename
					assert_match attachment.content, mail.attachments[i].content
					i = i + 1
				end
			end
		end
	end

end
