# coding: UTF-8

ROOT_PATH = File.dirname(File.expand_path(__FILE__))

$:.unshift(ROOT_PATH)
$:.unshift(ROOT_PATH + "/../")
$:.unshift(ROOT_PATH + "/../../main/")
$:.unshift(ROOT_PATH + "/../../main/lib")

# mail2weblog のルートパスを設定する
M2W_ROOT_PATH = ROOT_PATH + "/../../main/"

require 'm2w_configurator'
require 'm2w_mail_parser'
require 'm2w_mail_parser_html_text_type'

require 'html_data/expected/expected'

require 'minitest/unit'

MiniTest::Unit.autorun

#=Mail2WeblogMailParser のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_m2w_mail_parser_html_text_type.rb 593 2012-03-13 10:24:42Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestMail2WeblogMailParserHtmlTextType < MiniTest::Unit::TestCase

	def test_parse_content
		parser = Mail2WeblogMailParser.new(
			M2W_SUBJECT_HEADER,
			M2W_SUBJECT_SEPARATOR_REGEX)

		mail_content = <<"END_CONTENT"
<div>
<div>
attr1.value1<br />
attr2.value2<br />
attr3.value3<br />
<br />
<div>本文</div>
</div>
</div>
END_CONTENT

		(header, content, contents) = parser.parse_content(mail_content, {}, Mail2WeblogMailParserHtmlTextType.new)
		assert_equal "value1", header['attr1'], header.to_s
		assert_equal "value2", header['attr2'], header.to_s
		assert_equal "value3", header['attr3'], header.to_s
		assert_equal "<div>本文</div>", content, content
		assert_equal "<div>本文</div>", contents['description'], contents.to_s
	end

	def test_arrange_content
		type = Mail2WeblogMailParserHtmlTextType.new
		mail_content = <<"END_CONTENT".gsub(/\r|\n/, '')
<div>
<div>
attr1.value1<br />
attr2.value2<br />
attr3.value3<br />
<br />
<div>本文</div>
</div>
</div>
END_CONTENT

		expected_content = <<"END_CONTENT".gsub(/\r|\n/, '')
<div>
attr1.value1<br />
attr2.value2<br />
attr3.value3<br />
<br />
<div>本文</div>
</div>
END_CONTENT

		content = type.arrange_content(mail_content)
		assert_equal expected_content, content
	end

	def test_clean_html_line
		type = Mail2WeblogMailParserHtmlTextType.new

		assert_equal "abcdef", type.__clean_html_line("  abcdef  ")
	end

	def test_split
		type = Mail2WeblogMailParserHtmlTextType.new
		mail_content = <<"END_CONTENT"
<div>
<div>
attr1.value1<br />
attr2.value2<br />
attr3.value3<br />
<br />
<div>本文</div>
</div>
</div>
END_CONTENT

		expected = [
			"<div>attr1.value1<br />",
			"attr2.value2<br />",
			"attr3.value3<br />",
			"<br />",
			"<div>本文</div>",
			"</div>",
		]

		buffers = type.split(mail_content)
		assert_equal expected, buffers
	end

	def test_is_space_line
		type = Mail2WeblogMailParserHtmlTextType.new

		assert type.is_space_line('<div></div>')
		assert type.is_space_line('<DIV></DIV>')
		assert type.is_space_line('<div></DIV>')
		assert type.is_space_line('<DIV></div>')
		assert type.is_space_line('<div attr1="value1"></div>')
		assert !type.is_space_line('<div>')
		assert !type.is_space_line('<DIV>')

		assert type.is_space_line('<p></p>')
		assert type.is_space_line('<P></P>')
		assert type.is_space_line('<p></P>')
		assert type.is_space_line('<P></p>')
		assert type.is_space_line('<p attr1="value1"></p>')
		assert type.is_space_line('<p>')
		assert type.is_space_line('<p></p>')

		assert type.is_space_line('<br>')
		assert type.is_space_line('<BR>')

		assert type.is_space_line('<div>&nbsp;</div>')
		assert type.is_space_line('<p>&nbsp;</p>')
		assert type.is_space_line('&nbsp;<p>')
		assert type.is_space_line('&nbsp;<br>')
	end

	def test_parse_header
		type = Mail2WeblogMailParserHtmlTextType.new

		datas = [
			[true , "<div>attr1.value1</div>"],
			[true , "<DIV>attr1.value1</DIV>"],
			[true , "<div>attr1.value1</DIV>"],
			[true , "<DIV>attr1.value1</div>"],
			[true , "<div attr='value'>attr1.value1</div>"],
			[true , "<p>attr1.value1</p>"],
			[true , "<P>attr1.value1</P>"],
			[true , "<p>attr1.value1</P>"],
			[true , "<P>attr1.value1</p>"],
			[true , "<p attr='value'>attr1.value1</p>"],
			[true , "attr1.value1<br />"],
			[true , "attr1.value1<p />"],
			[false, "attr1.value1<div />"],
		]

		datas.each do |data|
			(is_header, key, value) = type.parse_header(data[1], M2W_SUBJECT_SEPARATOR_REGEX)
			assert_equal data[0], is_header, data.to_s
			if data[0] then
				assert_equal "attr1", key, data.to_s
				assert_equal "value1", value, data.to_s
			end
		end
	end

	def test_parse_subject_separator
		type = Mail2WeblogMailParserHtmlTextType.new

		datas = [
			[true , "<div>more....</div>"],
			[true , "<div>more////</div>"],
			[true , "<div>more::::</div>"],
			[false, "<div>more.:::</div>"],
			[true , "<DIV>more....</DIV>"],
			[true , "<div>more....</DIV>"],
			[true , "<DIV>more....</div>"],
			[true , "<div attr='value'>more....</div>"],
			[true , "<p>more....</p>"],
			[true , "<P>more....</P>"],
			[true , "<p>more....</P>"],
			[true , "<P>more....</p>"],
			[true , "<p attr='value'>more....</p>"],
			[true , "more....<br />"],
			[true , "more....<p />"],
			[false, "more....<div />"],
		]

		datas.each do |data|
			(is_subject_separator, separating_contents_name) = type.parse_subject_separation(data[1], M2W_SUBJECT_SEPARATOR_REGEX)
			assert_equal data[0], is_subject_separator, data.to_s
			if data[0] then
				assert_equal "more", separating_contents_name, data.to_s
			end
		end
	end

	def test_parse_real_content
		parser = Mail2WeblogMailParser.new(
			M2W_SUBJECT_HEADER,
			M2W_SUBJECT_SEPARATOR_REGEX)

		Dir::glob(ROOT_PATH + "/html_data/*.txt") do |filename|
			open(filename, 'rb') do |file|
				(header, content, contents) = parser.parse_content(file.read.encode(M2W_SYSTEM_INNER_ENCODING, M2W_SYSTEM_INNER_ENCODING), {}, Mail2WeblogMailParserHtmlTextType.new)

				expected_name = File.basename(filename, '.*')
				assert_equal EXPECTED_PARSE_REAL_CONTENT[expected_name]['header'].size, header.size
				header.each do |key, value|
					assert_equal EXPECTED_PARSE_REAL_CONTENT[expected_name]['header'][key], header[key], header.to_s
				end
				assert_equal EXPECTED_PARSE_REAL_CONTENT[expected_name]['content'], content, content
				assert_equal EXPECTED_PARSE_REAL_CONTENT[expected_name]['contents'].size, contents.size, contents.to_s
				contents.each do |key, value|
					assert_equal EXPECTED_PARSE_REAL_CONTENT[expected_name]['contents'][key], contents[key], contents.to_s
				end
			end
		end
	end

end
