# coding: UTF-8

if $0 == __FILE__ then
	ROOT_PATH = File.dirname(File.expand_path(__FILE__))

	$:.unshift(ROOT_PATH + "/../")
	$:.unshift(ROOT_PATH + "/../../main/")
	$:.unshift(ROOT_PATH + "/../../main/lib/")

	# mail2weblog のルートパスを設定する
	M2W_ROOT_PATH = ROOT_PATH + "/../../main/"

	require 'm2w_configurator'
	require 'minitest/autorun'
end

require 'm2w_plugin_manager'
require 'style/formatter'
require 'plugin/format/lib/format_plugin_utils'

class Example_FormatPlugin
	include FormatPluginUtils

	def plugin_contents(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning", "warning")
			when "error"
				raise FormatPluginError.new("error", "error")
			end
		else
			if content == nil then
				return "example"
			else
				return "example[#{content}]"
			end
		end
	end

	def plugin_contents_inline(param, vars)
		(params, option, content) = get_format_plugin_parameter(param, vars, 0)

		if option['type'] != nil then
			case option['type']
			when "warning"
				raise FormatPluginWarning.new("warning_inline", "warning_inline")
			when "error"
				raise FormatPluginError.new("error_inline", "error_inline")
			end
		else
			if content == nil then
				return "example_inline"
			else
				return "example_inline[#{content}]"
			end
		end
	end

end

#=Formatter のテスト
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: test_formatter_html.rb 857 2012-10-21 12:18:48Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
class TestFormatter4HTML < MiniTest::Unit::TestCase

	def setup
		@formatter = Formatter.new(Mail2WeblogPluginManager.new())
	end

	def test_format
original_content = <<"END_CONTENT"
tit.タイトル &lt;&amp;&gt;<br />
<br />
<p>１行目</p>
<p>２行目</p>
END_CONTENT

content = <<"END_CONTENT"
<p>１行目</p>
<p>２行目</p>
END_CONTENT

		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] tit.タイトル &amp;lt;&amp;amp;&amp;gt;&lt;br /&gt;
&lt;br /&gt;
&lt;p&gt;１行目&lt;/p&gt;
&lt;p&gt;２行目&lt;/p&gt;
 --><p>１行目</p>
<p>２行目</p>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin
		original_content = "<div>{{example}}</div>"
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;{{example}}&lt;/div&gt; --><div>example</div>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_multiline_with_plugin
original_content = <<"END_CONTENT"
<div>{{example
line1
line2}}</div>
END_CONTENT

		content = original_content
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;{{example
line1
line2}}&lt;/div&gt;
 --><div>example[line1
line2]</div>
END_CONTENT

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin_warning
		original_content = "<div>{{example(type@warning)}}</div>"
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;{{example(type@warning)}}&lt;/div&gt; --><div>{{警告: warning}}</div>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 1, warnings.size
	end

	def test_format_with_plugin_error
		original_content = "<div>{{example(type@error)}}</div>"
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;{{example(type@error)}}&lt;/div&gt; --><div>{{エラー: error}}</div>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 1, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin_inline
		original_content = "<div>テスト{{example}}</div>"
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;テスト{{example}}&lt;/div&gt; --><div>テストexample_inline</div>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_multiline_with_plugin_inline
original_content = <<"END_CONTENT"
<div>テスト{{example
line1
line2}}</div>
END_CONTENT

		content = original_content
		contents = {
			'description' => content
		}

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;テスト{{example
line1
line2}}&lt;/div&gt;
 --><div>テストexample_inline[line1
line2]</div>
END_CONTENT

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugin_inline_warning
		original_content = "<div>テスト{{example(type@warning)}}</div>"
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;テスト{{example(type@warning)}}&lt;/div&gt; --><div>テスト{{警告: warning_inline}}</div>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 1, warnings.size
	end

	def test_format_with_plugin_inline_error
		original_content = "<div>テスト{{example(type@error)}}</div>"
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;テスト{{example(type@error)}}&lt;/div&gt; --><div>テスト{{エラー: error_inline}}</div>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 1, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_plugins
original_content = <<"END_CONTENT"
<div>テスト{{example}}</div>
<div>{{example}}</div>
<div>{{example</div>
END_CONTENT
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;テスト{{example}}&lt;/div&gt;
&lt;div&gt;{{example}}&lt;/div&gt;
&lt;div&gt;{{example&lt;/div&gt;
 --><div>テストexample_inline</div>
<div>example</div>
<div>{{example</div>
END_CONTENT

		errors = []
		warnings = []

		result = @formatter.format(request, {}, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_related_attachment
		original_content = %Q!<div>テスト<img src="cid:content_id1" /></div>!
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;テスト&lt;img src=&quot;cid:content_id1&quot; /&gt;&lt;/div&gt; --><!--[related_urls] content_id1=http://abc.def.ghi/jkl.png --><div>テスト<img src="http://abc.def.ghi/jkl.png" /></div>
END_CONTENT

		vars = {
			:related_attachment_urls => {
				'content_id1' => 'http://abc.def.ghi/jkl.png'},
			:related_thumbnail_urls => {},
		}
		errors = []
		warnings = []

		result = @formatter.format(request, vars, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_format_with_related_attachments
		original_content = %Q!<div>テスト<img src="cid:content_id1" /><img src="cid:content_id2" /></div>!
		content = original_content
		contents = {
			'description' => content
		}

		request = Mail2WeblogRequest.new(
			:format => 'html',
			:original_content => original_content,
			:content => content,
			:contents => contents)

expected_description = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] &lt;div&gt;テスト&lt;img src=&quot;cid:content_id1&quot; /&gt;&lt;img src=&quot;cid:content_id2&quot; /&gt;&lt;/div&gt; --><!--[related_urls] content_id1=http://abc.def.ghi/jkl.png,content_id2=http://mno.pqr.stu/vwx.png --><div>テスト<img src="http://abc.def.ghi/jkl.png" /><img src="http://mno.pqr.stu/vwx.png" /></div>
END_CONTENT

		vars = {
			:related_attachment_urls => {
				'content_id1' => 'http://abc.def.ghi/jkl.png',
				'content_id2' => 'http://mno.pqr.stu/vwx.png',
			},
			:related_thumbnail_urls => {},
		}
		errors = []
		warnings = []

		result = @formatter.format(request, vars, errors, warnings)

		assert_equal 1, result.size
		assert_equal expected_description, result['description']
		assert_equal 0, errors.size
		assert_equal 0, warnings.size
	end

	def test_get_uncommentout_content
content = <<"END_CONTENT"
<!--m2w_version:#{M2W_VERSION}--><!--[htmlcontent] tit.タイトル &amp;lt;&amp;amp;&amp;gt;&lt;br /&gt;
&lt;br /&gt;
&lt;p&gt;１行目&lt;/p&gt;
&lt;p&gt;２行目&lt;/p&gt; --><p>１行目</p><p>２行目</p>
END_CONTENT

expected_content = <<"END_CONTENT"
tit.タイトル &lt;&amp;&gt;<br />
<br />
<p>１行目</p>
<p>２行目</p>
END_CONTENT

		(version, content, type) = @formatter.get_uncommentout_content(content)
		assert_equal M2W_VERSION, version
		assert_equal expected_content.chomp, content
		assert_equal 'html', type
	end

	def test_get_uncommentout_related_attachment_urls_content
		content = "<!--[related_urls] content1=abc.txt,content2=def.txt,content3=ghi.txt -->"
		urls = @formatter.get_uncommentout_related_attachment_urls_content(content)

		assert_equal 3, urls.length
		assert_equal 'abc.txt', urls['content1']
		assert_equal 'def.txt', urls['content2']
		assert_equal 'ghi.txt', urls['content3']
	end

end
