# coding: UTF-8

require 'exception/m2w_timeout'
require 'mail_encoder/simple_mail_encoder'
require 'mailutils/mail_processor'

M2W_TRANSFORM_TARGET_COMMAND = [SimplePost_CommandPlugin, Post_CommandPlugin, Get_CommandPlugin, Edit_CommandPlugin, Delete_CommandPlugin]

#=メールを基にブログへのコマンドを発行するブログコマンド
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: m2w_command.rb 774 2012-07-10 08:37:06Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
#
# 発行するコマンドの種類は、メールタイトル（Subject ヘッダー）で判断します。
#
# 想定するメールタイトルのフォーマットは以下の通りです。
#
# header:コマンド名:パラメータ
#
# hedaer は、コンストラクタで指定した任意の文字列です。
# 本クラスは、メールタイトルが上記フォーマットにマッチしないメールを無視します。
#
# 該当するメールであることが分かった場合は、コマンド名に応じた処理を実行します。
# コマンドの詳細は、コマンドを実行する各クラスを参照してください。
#
# コマンドが成功したと認識した場合、本クラスはメールボックスから該当のメールを削除します。
class Mail2WeblogCommand

	attr_accessor :start_time

	def initialize(weblog, mail_parser, mail_sender, plugin_manager, processing_time)
		@weblog = weblog
		@mail_parser = mail_parser
		@mail_sender = mail_sender
		@plugin_manager = plugin_manager
		@processing_time = processing_time

		@start_time = Time.now.to_i
	end

	def run(m)
		if @processing_time >= 0 && Time.now.to_i - @start_time >= @processing_time then
			raise Mail2WeblogTimeout.new("動作制限時間に到達。", "動作制限時間に到達しました。")
		end

		begin
			mail = MailProcessor.parse(m.pop, SimpleMailEncoder.new(M2W_SYSTEM_INNER_ENCODING))

			if mail.from_address.address == M2W_REPLY_ADDRESS then
				M2W_LOGGER.info("応答メールアドレスからメールが送信されてきました。")
				m.delete
				return true
			end

			if mail.is_report then
				M2W_LOGGER.info("メール送信に失敗し、エラーレポートが #{mail.from_address.address} から返ってきました。")
				m.delete
				return true
			end
		rescue => e
			M2W_LOGGER.error("メール解析中に予測不能なエラーが発生しました。")
			M2W_LOGGER.error(e)
			M2W_LOGGER.error(m.pop)
			m.delete
			return true
		end

		begin
			M2W_LOGGER.info("Processing ... subject = #{mail.subject}, from = #{mail.from_address.address}")

			(command, param) = __get_command_and_param(mail)
			if command != nil then
				command.weblog = @weblog if command.respond_to?("weblog=")
				command.mail_parser = @mail_parser if command.respond_to?("mail_parser=")
				command.mail_sender = @mail_sender if command.respond_to?("mail_sender=")
				command.plugin_manager = @plugin_manager if command.respond_to?("plugin_manager=")
				command.mail = mail if command.respond_to?("mail=")
				command.param = param if command.respond_to?("param=")

				if command.run() then
					M2W_LOGGER.info("Successed")
				else
					M2W_LOGGER.info("Failed")
				end
			else
				M2W_LOGGER.error("不明なコマンド(#{command_name})を実行しようとしました。")
				@mail_sender.sendmail_as_command("error", nil, {}, "不明なコマンド(#{command_name})を実行しようとしました。", mail.from_address, [])
			end

			m.delete
		rescue => e
			M2W_LOGGER.error("ブログコマンド実行中に予測不能なエラーが発生しました。")
			M2W_LOGGER.error(e)
			M2W_LOGGER.error(m.pop)
			m.delete
			@mail_sender.sendmail_as_command("error", nil, {}, "ブログコマンド実行中に予測不能なエラーが発生しました。", mail.from_address, [])
		end

		return true
	end

	def __get_command_and_param(mail)
		command = nil
		param = ''

		subject_parsed_result = @mail_parser.parse_subject(mail.escaped_prefix_subject)
		if subject_parsed_result != nil then
			command_name = subject_parsed_result[0]
			param = subject_parsed_result[1]

			command = @plugin_manager.get_command_plugin(command_name)
		else
			command = @plugin_manager.get_command_plugin('simple_post')
		end

		return [command, __transform_plugin_param(command, mail, param)]
	end

	def __transform_plugin_param(command, mail, param)
		M2W_LOGGER.debug("Start  #{self.class.name}#__transform_plugin_param ... command = #{command.class}, param = #{param}")

		M2W_LOGGER.debug("index ... #{M2W_TRANSFORM_TARGET_COMMAND.index(command.class)}")

		if M2W_TRANSFORM_TARGET_COMMAND.index(command.class) != nil then
			M2W_LOGGER.debug("translating ...")

			if mail.has_content then
				(header, content, contents) = @mail_parser.parse_content(mail.content, {}, Mail2WeblogMailParserPlainTextType.new)
				param = header['blogid'] if header.has_key?('blogid')
			elsif mail.has_html_content then
				(html_header, html_content, html_contents) = @mail_parser.parse_content(mail.html_content, {}, Mail2WeblogMailParserHtmlTextType.new)
				param = html_header['blogid'] if html_header.has_key?('blogid')
			end

			M2W_LOGGER.debug("Finish #{self.class.name}#__transform_plugin_param ... param = #{param}")
			return param
		else
			M2W_LOGGER.debug("Finish #{self.class.name}#__transform_plugin_param ... param = #{param}")
			return param
		end
	end

end
