﻿#ifndef   _HUtility_h_
#define   _HUtility_h_


/**
 * @addtogroup HLib
 *  @{
 */


/**
 * HUtility
 * @file
 * @brief HLib utility.
 * @brief ユーティリティと、なんとなく行き場の無いマクロ達。
 * @author aoi_lif
 */




//////////////////////////////////////////////////////////////
// include
//////////////////////////////////////////////////////////////
#include "./HDefine.h"										// HLib define
#include "./HPlatform.h"									// HLib platform
#include "./HType.h"										// HLib type




// start HLib namespace
H_NAMESPACE_START




/**
 * 2進表記(1Byte)
 * @param[in] v1 下位5～8ビット(0か1を4ビットのみ指定する)
 * @param[in] v0 下位1～4ビット(0か1を4ビットのみ指定する)
 * @note 引数を2進で表記する。
 * @note enumの内部等、ほぼ何処でも使用可能。
 * @code
 *   _0b1(1111,1111) == 0xFF
 * @endcode
 * @warning 0と1以外が入ったときも、4ビット以上指定されたときもエラーチェックしていないので注意。
 */
#define _0b1(v1,v0)                     (((((0x##v1)>>12)%2)<<7) +          \
                                         ((((0x##v1)>> 8)%2)<<6) +          \
                                         ((((0x##v1)>> 4)%2)<<5) +          \
                                         ((((0x##v1)>> 0)%2)<<4) +          \
                                         ((((0x##v0)>>12)%2)<<3) +          \
                                         ((((0x##v0)>> 8)%2)<<2) +          \
                                         ((((0x##v0)>> 4)%2)<<1) +          \
                                           ((0x##v0)>> 0)%2     )


/**
 * 2進表記(2Byte)
 * @param[in] v3 下位13～16ビット(0か1を4ビットのみ指定する)
 * @param[in] v2 下位9～12ビット(0か1を4ビットのみ指定する)
 * @param[in] v1 下位5～8ビット(0か1を4ビットのみ指定する)
 * @param[in] v0 下位1～4ビット(0か1を4ビットのみ指定する)
 * @note 詳細は、2進表記(1Byte)を参照して下さい。
 * @code
 *   _0b2(1111,0000,1111,0000) == 0xF0F0
 * @endcode
 */
#define _0b2(v3,v2,v1,v0)               (((((0x##v3)>>12)%2)<<15) +         \
                                         ((((0x##v3)>> 8)%2)<<14) +         \
                                         ((((0x##v3)>> 4)%2)<<13) +         \
                                         ((((0x##v3)>> 0)%2)<<12) +         \
                                         ((((0x##v2)>>12)%2)<<11) +         \
                                         ((((0x##v2)>> 8)%2)<<10) +         \
                                         ((((0x##v2)>> 4)%2)<< 9) +         \
                                         ((((0x##v2)>> 0)%2)<< 8) +         \
                                         ((((0x##v1)>>12)%2)<< 7) +         \
                                         ((((0x##v1)>> 8)%2)<< 6) +         \
                                         ((((0x##v1)>> 4)%2)<< 5) +         \
                                         ((((0x##v1)>> 0)%2)<< 4) +         \
                                         ((((0x##v0)>>12)%2)<< 3) +         \
                                         ((((0x##v0)>> 8)%2)<< 2) +         \
                                         ((((0x##v0)>> 4)%2)<< 1) +         \
                                           ((0x##v0)>> 0)%2      )


/**
 * 2進表記(3Byte)
 * @param[in] v5 下位21～24ビット(0か1を4ビットのみ指定する)
 * @param[in] v4 下位17～20ビット(0か1を4ビットのみ指定する)
 * @param[in] v3 下位13～16ビット(0か1を4ビットのみ指定する)
 * @param[in] v2 下位9～12ビット(0か1を4ビットのみ指定する)
 * @param[in] v1 下位5～8ビット(0か1を4ビットのみ指定する)
 * @param[in] v0 下位1～4ビット(0か1を4ビットのみ指定する)
 * @note 詳細は、2進表記(1Byte)を参照して下さい。
 * @code
 *   _0b3(1111,0000,1111,0000,1111,0000) == 0xF0F0F0
 * @endcode
 */
#define _0b3(v5,v4,v3,v2,v1,v0)         (((((0x##v5)>>12)%2)<<23) +         \
                                         ((((0x##v5)>> 8)%2)<<22) +         \
                                         ((((0x##v5)>> 4)%2)<<21) +         \
                                         ((((0x##v5)>> 0)%2)<<20) +         \
                                         ((((0x##v4)>>12)%2)<<19) +         \
                                         ((((0x##v4)>> 8)%2)<<18) +         \
                                         ((((0x##v4)>> 4)%2)<<17) +         \
                                         ((((0x##v4)>> 0)%2)<<16) +         \
                                         ((((0x##v3)>>12)%2)<<15) +         \
                                         ((((0x##v3)>> 8)%2)<<14) +         \
                                         ((((0x##v3)>> 4)%2)<<13) +         \
                                         ((((0x##v3)>> 0)%2)<<12) +         \
                                         ((((0x##v2)>>12)%2)<<11) +         \
                                         ((((0x##v2)>> 8)%2)<<10) +         \
                                         ((((0x##v2)>> 4)%2)<< 9) +         \
                                         ((((0x##v2)>> 0)%2)<< 8) +         \
                                         ((((0x##v1)>>12)%2)<< 7) +         \
                                         ((((0x##v1)>> 8)%2)<< 6) +         \
                                         ((((0x##v1)>> 4)%2)<< 5) +         \
                                         ((((0x##v1)>> 0)%2)<< 4) +         \
                                         ((((0x##v0)>>12)%2)<< 3) +         \
                                         ((((0x##v0)>> 8)%2)<< 2) +         \
                                         ((((0x##v0)>> 4)%2)<< 1) +         \
                                           ((0x##v0)>> 0)%2      )


/**
 * 2進表記(4Byte)
 * @param[in] v7 下位29～32ビット(0か1を4ビットのみ指定する)
 * @param[in] v6 下位25～28ビット(0か1を4ビットのみ指定する)
 * @param[in] v5 下位21～24ビット(0か1を4ビットのみ指定する)
 * @param[in] v4 下位17～20ビット(0か1を4ビットのみ指定する)
 * @param[in] v3 下位13～16ビット(0か1を4ビットのみ指定する)
 * @param[in] v2 下位9～12ビット(0か1を4ビットのみ指定する)
 * @param[in] v1 下位5～8ビット(0か1を4ビットのみ指定する)
 * @param[in] v0 下位1～4ビット(0か1を4ビットのみ指定する)
 * @note 詳細は、2進表記(1Byte)を参照して下さい。
 * @code
 *   _0b4(1111,0000,1111,0000,1111,0000,1111,0000) == 0xF0F0F0F0
 * @endcode
 */
#define _0b4(v7,v6,v5,v4,v3,v2,v1,v0)   (((((0x##v7)>>12)%2)<<31) +         \
                                         ((((0x##v7)>> 8)%2)<<30) +         \
                                         ((((0x##v7)>> 4)%2)<<29) +         \
                                         ((((0x##v7)>> 0)%2)<<28) +         \
                                         ((((0x##v6)>>12)%2)<<27) +         \
                                         ((((0x##v6)>> 8)%2)<<26) +         \
                                         ((((0x##v6)>> 4)%2)<<25) +         \
                                         ((((0x##v6)>> 0)%2)<<24) +         \
                                         ((((0x##v5)>>12)%2)<<23) +         \
                                         ((((0x##v5)>> 8)%2)<<22) +         \
                                         ((((0x##v5)>> 4)%2)<<21) +         \
                                         ((((0x##v5)>> 0)%2)<<20) +         \
                                         ((((0x##v4)>>12)%2)<<19) +         \
                                         ((((0x##v4)>> 8)%2)<<18) +         \
                                         ((((0x##v4)>> 4)%2)<<17) +         \
                                         ((((0x##v4)>> 0)%2)<<16) +         \
                                         ((((0x##v3)>>12)%2)<<15) +         \
                                         ((((0x##v3)>> 8)%2)<<14) +         \
                                         ((((0x##v3)>> 4)%2)<<13) +         \
                                         ((((0x##v3)>> 0)%2)<<12) +         \
                                         ((((0x##v2)>>12)%2)<<11) +         \
                                         ((((0x##v2)>> 8)%2)<<10) +         \
                                         ((((0x##v2)>> 4)%2)<< 9) +         \
                                         ((((0x##v2)>> 0)%2)<< 8) +         \
                                         ((((0x##v1)>>12)%2)<< 7) +         \
                                         ((((0x##v1)>> 8)%2)<< 6) +         \
                                         ((((0x##v1)>> 4)%2)<< 5) +         \
                                         ((((0x##v1)>> 0)%2)<< 4) +         \
                                         ((((0x##v0)>>12)%2)<< 3) +         \
                                         ((((0x##v0)>> 8)%2)<< 2) +         \
                                         ((((0x##v0)>> 4)%2)<< 1) +         \
                                           ((0x##v0)>> 0)%2      )


/**
 * ファイル名
 */
#define  H_FILE_NAME         __FILE__


/**
 * 行番号
 */
#define  H_LINE              __LINE__


/**
 * 関数名
 */
#if       H_COMPILER_IS_VC
#	// コンパイラはVC
#       define  H_FUNC_NAME         __FUNCTION__
#       define  H_FUNC_NAME_EXT     __FUNCSIG__
#else  // H_COMPILER_IS_VC
#	// コンパイラはVC以外
#   ifdef __cplusplus ? __GNUC_PREREQ (2, 6) : __GNUC_PREREQ (2, 4)
#       define  H_FUNC_NAME         __FUNCTION__
#       define  H_FUNC_NAME_EXT     __PRETTY_FUNCTION__
#   elif defined(HAVE_FUNCNAME__FUNCTION)
#       define  H_FUNC_NAME         __FUNCTION__"()"
#       define  H_FUNC_NAME_EXT     __FUNCTION__"()"
#   elif   defined(HAVE_FUNCNAME__FUNC)
#       define  H_FUNC_NAME         __func__"()"
#       define  H_FUNC_NAME_EXT     __func__"()"
#   else
#       define  H_FUNC_NAME         ((const char *)"")
#       define  H_FUNC_NAME_EXT     ((const char *)"")
#   endif
#endif // H_COMPILER_IS_VC
/*
#if       H_COMPILER_IS_VC
#	// コンパイラはVC
#       define  H_FUNC_NAME         __FUNCTION__
#       define  H_FUNC_NAME_EXT     __FUNCSIG__
#else  // H_COMPILER_IS_VC
#	// コンパイラはVC以外
#   ifdef __cplusplus ? __GNUC_PREREQ (2, 6) : __GNUC_PREREQ (2, 4)
#       define  H_FUNC_NAME         __FUNCTION__
#       define  H_FUNC_NAME_EXT     __PRETTY_FUNCTION__
#   else
#   ifdef HAVE_FUNCNAME__FUNCTION
#       define  H_FUNC_NAME         __FUNCTION__"()"
#       define  H_FUNC_NAME_EXT     __FUNCTION__"()"
#   else
#   ifdef HAVE_FUNCNAME__FUNC
#       define  H_FUNC_NAME         __func__"()"
#       define  H_FUNC_NAME_EXT     __func__"()"
#   else
#       define  H_FUNC_NAME         ((const char *)"")
#       define  H_FUNC_NAME_EXT     ((const char *)"")
#   endif // HAVE_FUNCNAME__FUNC
#   endif // HAVE_FUNCNAME__FUNCTION
#   endif // __cplusplus ? __GNUC_PREREQ (2, 6) : __GNUC_PREREQ (2, 4)
#endif // H_COMPILER_IS_VC
*/
/*
#  ifdef __cplusplus ? __GNUC_PREREQ (2, 6) : __GNUC_PREREQ (2, 4)
#    define   H_FUNC_NAME         __PRETTY_FUNCTION__
#    define   H_FUNC_ARG          ((const char *)"")
#  else
#    ifdef HAVE_FUNCNAME__FUNCTION
#      define  H_FUNC_NAME         __FUNCTION__
#      define  H_FUNC_ARG          "()"
#    else
#      ifdef HAVE_FUNCNAME__FUNC
#        define H_FUNC_NAME         __func__
#        define H_FUNC_ARG          "()"
#      else
#        define H_FUNC_NAME         ((const char *)"")
#        define  H_FUNC_ARG          ((const char *)"")
#      endif
#    endif
#  endif
*/


/**
 * 文字列化内部用(to string internal)
 * @param[in] content 文字列に変換したいもの
 * @note H_TO_STR()内部より呼ばれる
 */
#define H_TO_STR_INTERNAL(content)          #content
/**
 * 文字列化(to string)
 * @param[in] macro_or_string 文字列に変換したいもの
 */
#define H_TO_STR(macro_or_string)           H_TO_STR_INTERNAL(macro_or_string)

/**
 * マクロの結合内部用(concatenate two macro strings internal)
 * @param[in] macroL 左のマクロ
 * @param[in] macroR 右のマクロ
 * @note H_MACRO_CAT()内部より呼ばれる
 */
#define H_MACRO_CAT_INTERNAL(macroL,macroR) macroL ## macroR

/**
 * マクロの結合(concatenate two macro strings)
 * @param[in] macroL 左のマクロ
 * @param[in] macroR 右のマクロ
 */
#define H_MACRO_CAT(macroL,macroR)          H_MACRO_CAT_INTERNAL(macroL,macroR)


/**
 * caseの値を文字列でpStrに入れる
 * @param[in] case_val caseの値
 * @param[out] pStr case_valを文字列に変換して返す
 * @note 例：
 * @code
 *   enum ABC {
 *       A,
 *       B,
 *       C
 *   };
 *   
 *   const char*
 *   EnumToStr(ABC abc)
 *   {
 *       const char* pStr;
 *       switch (abc) {
 *       H_SET_STR_CASE(pStr, A);
 *       H_SET_STR_CASE(pStr, B);
 *       H_SET_STR_CASE(pStr, C);
 *       H_SET_STR_DEFAULT(pStr, abc);
 *       }
 *       return pStr;
 *   }
 *   
 *   main()
 *   {
 *       printf("%s %s %s %s.\n", EnumToStr(A), EnumToStr(B), EnumToStr(C), EnumToStr(999));
 *   }
 * @endcode
 * @note 結果：
 * @code
 *   A B C unknown.
 * @endcode
 */
#define H_SET_STR_CASE(pStr, case_val)							\
			case case_val:										\
			pStr = #case_val;									\
			break
#define H_SET_STR_DEFAULT(pStr, switch_val)						\
			default:											\
			pStr = "unknown";									\
			HDPrintf(H_TO_STR(switch_val)":%d\n", switch_val);	\
			break;



/**
  foreach
 * @param[in] a_type	型
 * @param[in] a_value	変数
 * @param[in] a_itrW	イテレータ
 * @note itrEに行数を付けているのはデバッガで追いやすくするため。
 * @note map<int,int>の様な","を含んだものは使用できないのでtypedefしたものを使用してください。
 * @note 使い方
 * @note	std::vector<int>		vec;
 * @note	foreach (std::vector<int>, vec, itrW) {
 * @note		int&	i = *itrW;
 * @note	}
 */
#define foreach(a_type, a_value, a_itrW) \
	for (a_type::iterator a_itrW=(a_value).begin(), H_MACRO_CAT(itrE_line,H_LINE)=(a_value).end(); (a_itrW)!=H_MACRO_CAT(itrE_line,H_LINE); ++(a_itrW))

/**
  foreach(const版)
 * @param[in] a_type	型
 * @param[in] a_value	変数
 * @param[in] a_itrW	イテレータ
 */
#define const_foreach(a_type, a_value, a_itrW) \
	for (a_type::const_iterator a_itrW=(a_value).begin(), H_MACRO_CAT(itrE_line,H_LINE)=(a_value).end(); (a_itrW)!=H_MACRO_CAT(itrE_line,H_LINE); ++(a_itrW))

/**
  foreach(配列版)
 * @param[in] a_type	型
 * @param[in] a_array	配列
 * @param[in] a_element	配列の要素
 */
#define array_foreach(a_type, a_array, a_element) \
	for (a_type *H_MACRO_CAT(work_line,H_LINE)=&(a_array)[0], a_element=*H_MACRO_CAT(work_line,H_LINE); H_MACRO_CAT(work_line,H_LINE)<&(a_array)[HNumberOf(a_array)]; ++H_MACRO_CAT(work_line,H_LINE),a_element=*H_MACRO_CAT(work_line,H_LINE))



// カウンタースイッチ
#define C_SWITCH( pCounter )											\
			{ s32 *pCSCnt = (pCounter); if( 0 ) {}
#define C_CASE( val )													\
			else if( *pCSCnt == (val) )
#define C_CASE_BETWEEN( valA, valB )									\
			else if( (valA)<*pCSCnt && *pCSCnt<(valB) )
#define C_CASE_BETWEEN_E( valA, valB )									\
			else if( (valA)<=*pCSCnt && *pCSCnt<=(valB) )
#define C_CASE_BOOL( bool )												\
			else if( (bool) )
#define C_CASE_DEFAULT()												\
			else
#define C_SWITCH_END()													\
			}
/*
	counter++;
	C_SWITCH( &counter )
		C_CASE_BETWEEN( 4*ONE_SEC_FRAME, 5*ONE_SEC_FRAME ) {
			// 4秒～5秒の処理
		}
		C_CASE( 5*ONE_SEC_FRAME ) {
			// 5秒の処理
		}
		C_CASE_DEFAULT() {
			// デフォルトの処理
		}
	C_SWITCH_END()
*/


// else if
//  if ( A ) {a;}
//  ef ( B ) {b;}
//  等を行うため
#define ef								else if


/*
#if __GNUC_PREREQ__(2, 96)
 #define H_EXPECT_true(val)     __builtin_expect(!(val), false)
 #define H_EXPECT_false(val)    __builtin_expect(!(val), true)
 #define H_EXPECT(val,exp)      __builtin_expect((val), exp)
 #define ifExpectTrue(val,exp)  if (__builtin_expect(!(val),false))
 #define ifExpectFalse(val,exp) if (__builtin_expect(!(val),true))
 #define ifExpect(val,exp)      if (__builtin_expect((val),exp))
#else
 #define H_EXPECT_true(val)     (!(val))
 #define H_EXPECT_false(val)    (!(val))
 #define H_EXPECT(val,exp)      (val)
 #define ifExpectTrue(val,exp)  if (!(val))
 #define ifExpectFalse(val,exp) if (!(val))
 #define ifExpect(val,exp)      if ((val))
#endif

*/

/*
 // インライン強制
 #define INLINE_FORCE               __attribute__((always_inline))
 void Func(void) INLINE_FORCE;

 // インライン禁止
 #define INLINE_NO               __attribute__((noinline))
 void Func(void) INLINE_NO;
*/


#if     0
 switchを使用するタイプだと、expにコンパイル時に確定できない式(変数等)が入ったときにコンパイルできない
 #define H_COMPILE_TIME_ASSERT(exp) { switch (false) { case false: case (exp):; } }

 typedefを使用したタイプだと、bool以外の式に対応できない
 template<bool> struct compile_time_assert;
 template<> struct compile_time_assert<true> { class failure{}; };
 #define H_COMPILE_TIME_ASSERT(exp)                                     \
     do {                                                               \
         typedef compile_time_assert<exp>::failure HCompileTimeAssert;  \
     }while(0)
#endif

/**
 * expが偽の時アサートを出す。(expが変数の時はコンパイルタイムエラー、変数の時はランタイムアサート)
 * @param[in] exp expression 真か偽の式をいれる。(偽:アサート,真:何もしない)(偽以外は真として処理している)
 * @note expに偽の定数式が入ると「size of array ‘HDAssert’ is negative」コンパイルタイムエラーが出る。
 * @note expに偽の変数式が入ると「HDAssert(exp) is failed」ランタイムアサートが出る。
 * @note コンパイルタイムエラーの例：
 * @code
 *   main()
 *   {
 *       HDAssert(false);
 *   }
 * @endcode
 * @note 結果：
 * @code
 *   main.cpp: In function ‘main()’:
 *   main.cpp:10: error: size of array ‘HDAssert’ is negative
 * @endcode
 * @note ランタイムアサートの例：
 * @code
 *   main()
 *   {
 *       bool    bValue = false;
 *       HDAssert(bValue, "bValue:%d\n", bValue);
 *   }
 * @endcode
 * @note 結果：
 * @code
 *   error! main.cpp:11:
 *    In function main()
 *    HDAssert(bValue) is failed.
 *    bValue:0
 *   abort() is called
 * @endcode
 * @warning HDAssert[(exp)?1:-1]のexpを_expに変更するとコンパイルタイムエラーとならない為、変更不可。
 * @warning gcc拡張を使用しているので注意。
 */
#if       H_DEBUG
	// デバグ
	#if       H_COMPILER_IS_VC
		// コンパイラはVC
		#define HDAssert(_exp, ...)																									\
					do {																											\
						bool			expBool	= !!(_exp);	/* 念の為!!でboolに変換 */												\
						if (!expBool) {																								\
							/* アサート */																							\
							HDPError(																								\
								"HDAssert(" H_TO_STR(_exp) ") is failed.\n " __VA_ARGS__);											\
							abort();																								\
							/* _expが定数式の時はHDAssert[]のサイズをマイナスとしてコンパイルタイムエラーとする */					\
							int _HDAssert;																							\
							_HDAssert = 1/(int)(_exp);																				\
						}																											\
					} while(0)
	#elif     H_COMPILER_IS_GCC
		// コンパイラはGCC
		#define HDAssert(_exp, ...)																									\
					do {																											\
						typeof(_exp)	exp		= (_exp);																			\
						bool			expBool	= !!(exp);	/* 念の為!!でboolに変換 */												\
						if (!expBool) {																								\
							/* アサート */																							\
							HDPError(																								\
								"HDAssert(" H_TO_STR(_exp) ") is failed.\n " __VA_ARGS__);											\
							abort();																								\
							/* expが定数式の時はHDAssert[]のサイズをマイナスとしてコンパイルタイムエラーとする */					\
							int HDAssert[Iif(_exp,1,-1)]; /* _expをexpに変更するとコンパイルタイムエラーとならない為、変更不可 */	\
							HDAssert[0] = 0;																						\
						}																											\
					} while(0)
	#else  // H_COMPILER_IS
		// コンパイラはサポート外
		#error "unsupported compiler"
	#endif // H_COMPILER_IS
#else  // H_DEBUG
	// デバグ以外
	#define HDAssert(_exp, ...)
#endif // H_DEBUG



/**
 * 配列のサイズを返す
 * @param[in] array 配列
 * @return 配列のサイズ
 * @note Get the number of array elements.
 * @note 配列のサイズを取得します。
 * @warning arrayは動的に確保した配列ではいけない。
 * @warning arrayは多次元配列ではいけない。
 * @warning arrayを複数回参照する為、arrayはマクロの副作用が起きないものでなければいけない。
 * @note Count the number of elements in an array.
 * @warning gcc拡張を使用しているので注意。
 * @note Count the number of elements in an array. The array must be defined
 * @note  as such; using this with a dynamically allocated array will give
 * @note  incorrect results.
 */
#if       H_COMPILER_IS_VC
	#define HNumberOf(_array)   (sizeof(_array) / sizeof((_array)[0]))
#else  // H_COMPILER_IS
	#if       H_DEBUG
		// デバグ
		#define HNumberOf(_array)																								\
					(																											\
						{																										\
							/* _arrayが変数の時には((_array)==(const volatile void*)&(_array))がfalseになる為アサートとする */	\
							HDAssert((_array)==(const volatile void*)&(_array));												\
							(sizeof(_array) / sizeof((_array)[0]));																\
						}																										\
					)
	#else  // H_DEBUG
		// デバグ以外
		#define HNumberOf(_array)		(sizeof(_array) / sizeof((_array)[0]))
	#endif // H_DEBUG
#endif // H_COMPILER_IS


/**
 * バイトアライメントをとる(nを超えるtimeの倍数で一番小さいものを返す)
 */
inline u32
HByteAlignmentUp(u32 n, u32 time)
{
//	return (n/time + !!(n%time)) * time;
	return (n+(time-1)) / time * time;
}
inline u64
HByteAlignmentUp(u64 n, u64 time)
{
//	return (n/time + !!(n%time)) * time;
	return (n+(time-1)) / time * time;
}


/**
 * Immediate if(==Ternary Operation==三項演算子)
 * @note 三項演算子を関数型式で扱えるマクロ
 * @note 三項演算子では見づらい所や、なれていない人がいるときに。
 */
#define Iif(expression, truePart, falsePart)				((expression) ? (truePart) : (falsePart))


/**
 * booleanを文字列に変換して返す
 * @param[in] _booleanValue 真偽値
 * @return 真偽値の文字列
 * @note _booleanValue 真偽値
 * @note 例：
 * @code
 *   bool value = true;
 *   printf("value is %s.\n", HBool2Str(value));
 * @endcode
 * @note 結果：
 * @note  value is true.
 */
#if     0
	inline const char* const
	HBool2Str(bool _booleanValue)
	{
		return Iif(_booleanValue, "true", "false");
	}
#else
	#define HBool2Str(_booleanValue)						Iif((_booleanValue), "true", "false")
#endif


#if       H_DEBUG
	// デバグ
	#define HDbgScope(_code)		_code
#else  // H_DEBUG
	// デバグ以外
	/**
	 * デバッグスコープ
	 * デバグに使用する区間を関数形式で指定する。
	 * if H_DEBUG の代わり。
	 * 例
	 * @code
	 * HDbgScope(
	 *     for (int i=0; i<NUM; ++i) {
	 *         HDPrintf("%d\n", i);
	 *     }
	 * );
	 * Func(a, b HDbgScope(,dbgExplanation));
	 * @endcode
	 */
	#define HDbgScope(_code)
#endif // H_DEBUG


/**
 * マクロのwarning回避用
 * @note 下記のVCのワーニング回避
 * @note  testhlib.cpp(142) : warning C4552: '+' : 演算子にプログラム上の作用がありません。作用を伴う演算子を使用してください
 */
inline int
HDReturnS32(int _val)
{
	return _val;
}


/**
 * 浮動小数の値が有効かを返す
 * @param[in] val 浮動小数の値
 * @return valが有効なときtrue
 * @note valがNaN・-Inf・+Infのときfalse
 */
#if       H_COMPILER_IS_VC
	// コンパイラはVC
	#define IsFinite(val)	_finite(val)
#else  // H_COMPILER_IS
	// コンパイラはVC以外
	#define IsFinite(val)	isfinite(val)
#endif // H_COMPILER_IS




// end HLib namespace
H_NAMESPACE_END




/** @} */       // end of HLib group


#endif // _HUtility_h_
