﻿#ifndef   _HDPrint_h_
#define   _HDPrint_h_


/**
 * @addtogroup HLib
 *  @{
 */


/**
 * HDPrint
 * @file
 * @brief HLib debug print
 * @brief デバグプリント。
 * @note デバグ用です。
 * @note バイナリが大きくなるのを無視し、ヘッダファイルのインクルードのみで使用できるようにしています。
 * @author aoi_lif
 */




//////////////////////////////////////////////////////////////
// include
//////////////////////////////////////////////////////////////
#include "./HDefine.h"										// HLib define
#include "./HPlatform.h"									// HLib platform
#include "./HUtility.h"										// HLib utility




// start HLib namespace
H_NAMESPACE_START




//////////////////////////////////////////////////////////////
// define
//////////////////////////////////////////////////////////////
#if       H_DEBUG
	// デバグ
	#define HDPrint_USE_FLAG						(1)				///< HDPrintを使用するかのフラグ
	#define HDPrint_SAVE_USE_FLAG					(0)				///< HDPrintセーブ機能を使用するかのフラグ
	#define HDPrint_COLOR_ESCAPE_SEQUENCE_USE_FLAG	(0)				///< HDPrintカラー変更エスケープシーケンスを使用するかのフラグ
#else  // H_DEBUG
	// デバグ以外
	#define HDPrint_USE_FLAG						(1)				///< HDPrintを使用するかのフラグ
	#define HDPrint_SAVE_USE_FLAG					(0)				///< HDPrintセーブ機能を使用するかのフラグ
	#define HDPrint_COLOR_ESCAPE_SEQUENCE_USE_FLAG	(0)				///< HDPrintカラー変更エスケープシーケンスを使用するかのフラグ
#endif // H_DEBUG

#if		1
	#define HDPrint_PRINT_FUNC(str)					printf(str)		///< printf関数(出力した文字数を返す)(可変引数未対応で問題なし)
	#define HDPrint_T_PRINT_FUNC(str)				_tprintf(str)	///< printf関数(出力した文字数を返す)(可変引数未対応で問題なし)
#else
	#define HDPrint_PRINT_FUNC(str)					OutputDebugStringA(str),0	///< printf関数(出力した文字数を返す)(可変引数未対応で問題なし)
	#define HDPrint_T_PRINT_FUNC(str)				OutputDebugString(str),0	///< printf関数(出力した文字数を返す)(可変引数未対応で問題なし)
#endif
#define HDPrint_FFLUSH_FUNC()						fflush(stdout)	///< fflush関数
#define HDPrint_BREAK_FUNC()						DebugBreak()	///< break関数


#if       HDPrint_SAVE_USE_FLAG
	// HDPrintセーブ機能を使用する
	#define HDPrint_OUTPUT_TEXT_NAME				"./HDPrint.log"	///< debug print セーブファイル名
#else  // HDPrint_SAVE_USE_FLAG
	// HDPrintセーブ機能を使用しない
	#define HDPrint_OUTPUT_TEXT_NAME				NULL			///< debug print セーブファイル名
#endif // HDPrint_SAVE_USE_FLAG


// escape sequence
#if       HDPrint_COLOR_ESCAPE_SEQUENCE_USE_FLAG      // HDPrintカラー変更エスケープシーケンスを使用するかのフラグ
	#define ESC_DEF									"\x1b[0m"		///< 通常に戻すエスケープシーケンス		// default
	#define ESC_BLK									"\x1b[30m"		///< 黒色にするエスケープシーケンス		// black
	#define ESC_RED									"\x1b[31m"		///< 赤色にするエスケープシーケンス		// red
	#define ESC_GRE									"\x1b[32m"		///< 緑色にするエスケープシーケンス		// green
	#define ESC_YEL									"\x1b[33m"		///< 黄色にするエスケープシーケンス		// yellow
	#define ESC_BLU									"\x1b[34m"		///< 青色にするエスケープシーケンス		// blue
	#define ESC_PUR									"\x1b[35m"		///< 紫色にするエスケープシーケンス		// purple
	#define ESC_CYA									"\x1b[36m"		///< 水色にするエスケープシーケンス		// cyan
	#define ESC_WHI									"\x1b[37m"		///< 明白色にするエスケープシーケンス	// light gray
#else  // HDPrint_COLOR_ESCAPE_SEQUENCE_USE_FLAG   // HDPrintカラー変更エスケープシーケンスを使用するかのフラグ
	#define ESC_DEF									""				///< 通常に戻すエスケープシーケンス		// default
	#define ESC_BLK									""				///< 黒色にするエスケープシーケンス		// black
	#define ESC_RED									""				///< 赤色にするエスケープシーケンス		// red
	#define ESC_GRE									""				///< 緑色にするエスケープシーケンス		// green
	#define ESC_YEL									""				///< 黄色にするエスケープシーケンス		// yellow
	#define ESC_BLU									""				///< 青色にするエスケープシーケンス		// blue
	#define ESC_PUR									""				///< 紫色にするエスケープシーケンス		// purple
	#define ESC_CYA									""				///< 水色にするエスケープシーケンス		// cyan
	#define ESC_WHI									""				///< 明白色にするエスケープシーケンス	// light gray
#endif // HDPrint_COLOR_ESCAPE_SEQUENCE_USE_FLAG   // HDPrintカラー変更エスケープシーケンスを使用するかのフラグ


#if       H_COMPILER_IS_VC
	/**
	 * デバグプリント プリント
	 * @note デバグ用のprintf()
	 */
	#define HDPrint(_str)		HDPrintInternal(HDPrint_OUTPUT_TEXT_NAME, _str)

	 /**
	 * デバグプリント プリント(可変引数対応)
	 * @note デバグ用のprintf()
	 */
	#define HDPrintf(...)		HDPrintfInternal(HDPrint_OUTPUT_TEXT_NAME, __VA_ARGS__)

	/**
	 * デバグプリント プリント(フラッシュストリーム付)
	 * @note デバグ用のprintf()
	 */
	#define HDPFlush(...)																										\
				HDPrintf(__VA_ARGS__),																							\
				HDPrint_FFLUSH_FUNC()

	/**
	 * デバグプリント エラー(可変引数対応)
	 * @note デバグ用のエラー表示
	 */
	#define HDPError(...)																										\
				HDReturnS32(																									\
					HDPrintf(ESC_RED"error! %s:%d:\n In function %s\n ", H_FILE_NAME, H_LINE, H_FUNC_NAME) +					\
					HDPrintf(__VA_ARGS__) +																						\
					HDPColor(DEF) )

	/**
	 * デバグプリント ワーニング(可変引数対応)
	 * @note デバグ用のワーニング表示
	 */
	#define HDPWarning(...)																										\
				HDReturnS32(																									\
					HDPrintf(ESC_YEL"warning! %s:%d:\n In function %s\n ", H_FILE_NAME, H_LINE, H_FUNC_NAME) +					\
					HDPrintf(__VA_ARGS__) +																						\
					HDPColor(DEF) )

	/**
	 * デバグプリント 現在行表示(可変引数対応)
	 * @note デバグ用の現在行表示(可変引数対応)
	 */
	#define HDPLinef(...)																										\
				HDReturnS32(																									\
					HDPrintf("%s:%d:\n In function %s\n ", H_FILE_NAME, H_LINE, H_FUNC_NAME) +									\
					HDPrintf(__VA_ARGS__) )

	/**
	 * デバグプリント 現在行表示
	 * @note デバグ用の現在行表示
	 */
	#define HDPLine()																											\
				HDPrintf("%s:%d:\n In function %s\n ", H_FILE_NAME, H_LINE, H_FUNC_NAME)

	/**
	 * デバグプリント カラー変更
	 * @note デバグ用のカラー変更
	 */
	#define HDPColor(col)																										\
				HDPrint(H_MACRO_CAT(ESC_,col))

#if     HDPrint_USE_FLAG                // HDPrintを使用するかのフラグ
			// HDPrintを使用する
			#include <stdio.h>
			#include <stdarg.h>									// [va_start,va_end]
			#include <string.h>
#include <windows.h>
#include <tchar.h>
			#include "./HOverloadSecure.h"						// HLib overload secure
			#if       HDPrint_SAVE_USE_FLAG						// HDPrintセーブ機能を使用するかのフラグ
				#include "./HDFWrite.h"							// HLib debug fwrite
			#endif // HDPrint_SAVE_USE_FLAG
			inline int
			HDPrintInternal(const char* const _outputTextName, const char* _str)
			{
				// HDPrintセーブ
				#if       HDPrint_SAVE_USE_FLAG					// HDPrintセーブ機能を使用するかのフラグ
					if (_outputTextName != NULL) {
						HDFWrite(_outputTextName, _str, strlen(_str));
					}
				#else  // HDPrint_SAVE_USE_FLAG
					(void)_outputTextName;
				#endif // HDPrint_SAVE_USE_FLAG
				// print
				return HDPrint_PRINT_FUNC(_str);
			}
			inline int
			HDPrintfInternal(const char* const _outputTextName, const char* _format, ...)
			{
				// 可変引数の解釈
				const size_t		STR_LEN = 1024*16;
				char				str[STR_LEN];
				va_list				ap;
				va_start(ap, _format);
				HVSNPrintf(str, STR_LEN, STR_LEN, _format, ap);
				va_end(ap);
				// print
				return HDPrintInternal(_outputTextName, str);
			}
			inline int
			HDPrintInternal(const char* const _outputTextName, const TCHAR* _str)
			{
				// HDPrintセーブ
				#if       HDPrint_SAVE_USE_FLAG					// HDPrintセーブ機能を使用するかのフラグ
					if (_outputTextName != NULL) {
//						HDFWrite(_outputTextName, _str, strlen(_str));
						HDFWrite(_outputTextName, _str, _tcslen(_str));
					}
				#else  // HDPrint_SAVE_USE_FLAG
					(void)_outputTextName;
				#endif // HDPrint_SAVE_USE_FLAG
				// print
				return HDPrint_T_PRINT_FUNC(_str);
			}
			inline int
			HDPrintfInternal(const char* const _outputTextName, const TCHAR* _format, ...)
			{
				// 可変引数の解釈
				const size_t		STR_LEN = 1024*16;
				TCHAR				str[STR_LEN];
				va_list				ap;
				va_start(ap, _format);
				HVSNTPrintf(str, STR_LEN, STR_LEN, _format, ap);
				va_end(ap);
				// print
				return HDPrintInternal(_outputTextName, str);
			}
	#else   // HDPrint_USE_FLAG
		// HDPrintを使用しない
		/**
		 * デバグプリント 内部用プリント
		 * @note デバグ用の内部用プリント
		 */
		#define HDPrintfInternal(...)	
	#endif  // HDPrint_USE_FLAG

	#define H_USE_HDPrintToBeAbolished						(1)					//< HDPrintToBeAbolishedを使用するか
	#if       H_USE_HDPrintToBeAbolished
		/**
		 * @def HDPrintToBeAbolished(whenStr, alternateFunctionStr)
		 * 廃止予定関数の表示
		 * @brief 廃止予定関数が呼ばれた最初の1回にwarning表示を行う
		 * 関数のはじめに呼び出す
		 * 引数を内部でconst char*の変数に渡しなおしているのは、引数にNULLを指定したときのwarning(reading through null pointer)回避のため
		 * @code
		 * void OldFunc(void) {
		 *     HDPrintToBeAbolished("on monday", "NewFunc()");
		 * }
		 * 
		 * warning! test.cpp:10: In function void OldFunc(void)
		 *  This function will not be offered on monday.
		 *  Please use NewFunc().
		 * @endcode
		 */
		#define HDPrintToBeAbolished(_whenStr, _alternateFunctionStr)															\
			do {																												\
				static bool oneTime = false;																					\
				if (oneTime == false) {																							\
					oneTime = true;																								\
					const char* whenStr = (_whenStr);																			\
					const char* alternateFunctionStr = (_alternateFunctionStr);													\
					if (alternateFunctionStr == NULL) {																			\
						HDPWarning(																								\
							"This function will not be offered%s%s.\n",															\
							Iif((whenStr==NULL), "", " "),																		\
							Iif((whenStr==NULL), "", whenStr) );																\
					} else {																									\
						HDPWarning(																								\
							"This function will not be offered%s%s.\n"															\
							" Please use %s.\n",																				\
							Iif((whenStr==NULL), "", " "),																		\
							Iif((whenStr==NULL), "", whenStr),																	\
							alternateFunctionStr );																				\
					}																											\
				}																												\
			}while(0)
	#else  // H_USE_HDPrintToBeAbolished
		#define HDPrintToBeAbolished(whenStr, alternateFunctionStr)		
	#endif // H_USE_HDPrintToBeAbolished
#else  // H_COMPILER_IS_VC
	/**
	 * デバグプリント プリント
	 * @note デバグ用のprintf()
	 */
	#define HDPrint(_str)        HDPrintfInternal(0,     NULL,        0,        NULL,       NULL, HDPrint_OUTPUT_TEXT_NAME,       _str)

	 /**
	 * デバグプリント プリント(可変引数対応)
	 * @note デバグ用のprintf()
	 */
	#define HDPrintf(...)       HDPrintfInternal(0,     NULL,        0,        NULL,       NULL, HDPrint_OUTPUT_TEXT_NAME, __VA_ARGS__)

	/**
	 * デバグプリント プリント(フラッシュストリーム付)
	 * @note デバグ用のprintf()
	 */
	#define HDPFlush(_str)                                                                                                      \
				(                                                                                                               \
					{                                                                                                           \
						int len = HDPrintfInternal(0,     NULL,        0,        NULL, HDPrint_OUTPUT_TEXT_NAME, _str);         \
						HDPrint_FFLUSH_FUNC();                                                                                  \
						len;                                                                                                    \
					}                                                                                                           \
				)

	/**
	 * デバグプリント エラー(可変引数対応)
	 * @note デバグ用のエラー表示
	 */
	#define HDPError(...)       HDPrintfInternal(1, H_FILE_NAME, H_LINE, H_FUNC_NAME, HDPrint_OUTPUT_TEXT_NAME, __VA_ARGS__)

	/**
	 * デバグプリント ワーニング(可変引数対応)
	 * @note デバグ用のワーニング表示
	 */
	#define HDPWarning(...)     HDPrintfInternal(2, H_FILE_NAME, H_LINE, H_FUNC_NAME, HDPrint_OUTPUT_TEXT_NAME, __VA_ARGS__)

	/**
	 * デバグプリント 現在行表示(可変引数対応)
	 * @note デバグ用の現在行表示(可変引数対応)
	 */
	#define HDPLinef(...)       HDPrintfInternal(3, H_FILE_NAME, H_LINE, H_FUNC_NAME, HDPrint_OUTPUT_TEXT_NAME, __VA_ARGS__)

	/**
	 * デバグプリント 現在行表示
	 * @note デバグ用の現在行表示
	 */
	#define HDPLine()           HDPrintfInternal(4, H_FILE_NAME, H_LINE, H_FUNC_NAME, HDPrint_OUTPUT_TEXT_NAME,          "")

	/**
	 * デバグプリント カラー変更
	 * @note デバグ用のカラー変更
	 */
	#define HDPColor(col)       HDPrintfInternal(0,     NULL,        0,        NULL, HDPrint_OUTPUT_TEXT_NAME, H_MACRO_CAT(ESC_,col))
	#if     HDPrint_USE_FLAG                // HDPrintを使用するかのフラグ
			// HDPrintを使用する
			#include <stdio.h>
			#include <stdarg.h>                                 // [va_start,va_end]
			#include <string.h>
			#include "./HOverloadSecure.h"                      // HLib overload secure
			#if     HDPrint_SAVE_USE_FLAG                       // HDPrintセーブ機能を使用するかのフラグ
				#include "./HDFWrite.h"                         // HLib debug fwrite
			#endif  // HDPrint_SAVE_USE_FLAG                    // HDPrintセーブ機能を使用するかのフラグ
			inline int
			HDPrintfInternal(int type, const char* fileName, unsigned int lineNum, const char* funcName, const char* const outputTextName, const char* format, ...)
			{
				int                 allLen = 0;
				bool                line2 = true;
				bool                returnColor = false;
				// 1行目
				switch (type) {
				case 0:     // printf
					break;
				case 1:     // error
					allLen += HDPrintf(ESC_RED"error! %s:%d:\n In function %s\n ", fileName, lineNum, funcName);
					returnColor = true;
					break;
				case 2:     // warning
					allLen += HDPrintf(ESC_YEL"warning! %s:%d:\n In function %s\n ", fileName, lineNum, funcName);
					returnColor = true;
					break;
				case 3:     // linef
					allLen += HDPrintf("%s:%d:\n In function %s\n ", fileName, lineNum, funcName);
					break;
				case 4:     // line
					allLen += HDPrintf("%s:%d:\n In function %s\n", fileName, lineNum, funcName);
					line2 = false;
					break;
				}
				// 2行目
				if (line2 == true) {
					const size_t        STR_LEN = 1024;
					char                str[STR_LEN];
					va_list             ap;
					va_start(ap, format);
					HVSNPrintf(str, STR_LEN, STR_LEN, format, ap);
					va_end(ap);
					allLen += HDPrint_PRINTF_FUNC(str);

					#if     HDPrint_SAVE_USE_FLAG                       // HDPrintセーブ機能を使用するかのフラグ
						if (outputTextName != NULL) {
							HDFWrite(outputTextName, str, strlen(str));
						}
					#else   // HDPrint_SAVE_USE_FLAG                    // HDPrintセーブ機能を使用するかのフラグ
						(void)outputTextName;
					#endif  // HDPrint_SAVE_USE_FLAG                    // HDPrintセーブ機能を使用するかのフラグ
				}
				// 色を戻す
				if (returnColor == true) {
					allLen += HDPColor(DEF);
				}
				return allLen;
			}
	#else   // HDPrint_USE_FLAG
		// HDPrintを使用しない
		/**
		 * デバグプリント 内部用プリント
		 * @note デバグ用の内部用プリント
		 */
		#define HDPrintfInternal(...)    
	#endif  // HDPrint_USE_FLAG

	#if     1
		/**
		 * @def HDPrintToBeAbolished(whenStr, alternateFunctionStr)
		 * 廃止予定関数の表示
		 * @brief 廃止予定関数が呼ばれた最初の1回にwarning表示を行う
		 * 関数のはじめに呼び出す
		 * 引数を内部でconst char*の変数に渡しなおしているのは、引数にNULLを指定したときのwarning(reading through null pointer)回避のため
		 * @code
		 * void OldFunc(void) {
		 *     HDPrintToBeAbolished("on monday", "NewFunc()");
		 * }
		 * 
		 * warning! test.cpp:10: In function void OldFunc(void)
		 *  This function will not be offered on monday.
		 *  Please use NewFunc().
		 * @endcode
		 */
		#define HDPrintToBeAbolished(whenStr, alternateFunctionStr)                                             \
			do {                                                                                                \
				static bool oneTime = false;                                                                    \
				if (oneTime == false) {                                                                         \
					oneTime = true;                                                                             \
					const char* _whenStr = (whenStr);                                                           \
					const char* _alternateFunctionStr = (alternateFunctionStr);                                 \
					char str[512];                                                                              \
					if (_whenStr != NULL) {                                                                     \
						sprintf(str, "This function will not be offered %s.\n", _whenStr);                      \
					} else {                                                                                    \
						sprintf(str, "This function will not be offered.\n");                                   \
					}                                                                                           \
					if (_alternateFunctionStr != NULL) {                                                        \
						char str1[512];                                                                         \
						sprintf(str1, " Please use %s.\n", _alternateFunctionStr);                              \
						strcat(str, str1);                                                                      \
					}                                                                                           \
					HDPrint_PRINTF_FUNC(ESC_YEL"warning! " H_FILE_NAME ":%d:\n In function %s\n ", H_LINE, H_FUNC_NAME_EXT);    \
					HDPrint_PRINTF_FUNC(str);                                                                   \
					HDPrint_PRINTF_FUNC(ESC_DEF);                                                               \
				}                                                                                               \
			}while(0)
	#else
		#define HDPrintToBeAbolished(whenStr, alternateFunctionStr)     
	#endif
#endif // H_COMPILER_IS_VC



// end HLib namespace
H_NAMESPACE_END


/** @} */       // end of HLib group

#endif // _HDPrint_h_
