/*
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package net.sf.l2j.gameserver;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import javolution.util.FastList;
import javolution.util.FastMap;
import net.sf.l2j.Config;
import net.sf.l2j.L2DatabaseFactory;
import net.sf.l2j.gameserver.datatables.MapRegionTable;
import net.sf.l2j.gameserver.datatables.NpcTable;
import net.sf.l2j.gameserver.datatables.SkillTable;
import net.sf.l2j.gameserver.datatables.SpawnTable;
import net.sf.l2j.gameserver.instancemanager.CastleManager;
import net.sf.l2j.gameserver.model.AutoChatHandler;
import net.sf.l2j.gameserver.model.AutoSpawnHandler;
import net.sf.l2j.gameserver.model.L2Spawn;
import net.sf.l2j.gameserver.model.L2World;
import net.sf.l2j.gameserver.model.AutoSpawnHandler.AutoSpawnInstance;
import net.sf.l2j.gameserver.model.actor.instance.L2NpcInstance;
import net.sf.l2j.gameserver.model.actor.instance.L2PcInstance;
import net.sf.l2j.gameserver.model.entity.Castle;
import net.sf.l2j.gameserver.network.SystemMessageId;
import net.sf.l2j.gameserver.network.serverpackets.SSQInfo;
import net.sf.l2j.gameserver.network.serverpackets.SystemMessage;
import net.sf.l2j.gameserver.templates.L2NpcTemplate;
import net.sf.l2j.gameserver.templates.StatsSet;
import net.sf.l2j.gameserver.util.Broadcast;

/**
 *  Seven Signs Engine
 *
 * 
 *  
 *
 *  @author Tempy
 */
public class SevenSigns
{
    protected static final Logger _log = Logger.getLogger(SevenSigns.class.getName());
	private static SevenSigns _instance;

	// Basic Seven Signs Constants \\
	public static final String SEVEN_SIGNS_DATA_FILE = "config/signs.properties";
	public static final String SEVEN_SIGNS_HTML_PATH = "data/html/seven_signs/";

	public static final int CABAL_NULL = 0;
	public static final int CABAL_DUSK = 1;
	public static final int CABAL_DAWN = 2;

	public static final int SEAL_NULL = 0;
	public static final int SEAL_AVARICE = 1;
	public static final int SEAL_GNOSIS = 2;
	public static final int SEAL_STRIFE = 3;

	public static final int PERIOD_COMP_RECRUITING = 0;
	public static final int PERIOD_COMPETITION = 1;
	public static final int PERIOD_COMP_RESULTS = 2;
	public static final int PERIOD_SEAL_VALIDATION = 3;

	public static final int PERIOD_START_HOUR = 18;
	public static final int PERIOD_START_MINS = 00;
    public static final int PERIOD_START_DAY = Calendar.MONDAY;

    // The quest event and seal validation periods last for approximately one week
    // with a 15 minutes "interval" period sandwiched between them.
    public static final int PERIOD_MINOR_LENGTH = 900000;
    public static final int PERIOD_MAJOR_LENGTH = 604800000 - PERIOD_MINOR_LENGTH;

	public static final int ANCIENT_ADENA_ID = 5575;
	public static final int RECORD_SEVEN_SIGNS_ID = 5707;
    public static final int CERTIFICATE_OF_APPROVAL_ID = 6388;
	public static final int RECORD_SEVEN_SIGNS_COST = 500;
    public static final int ADENA_JOIN_DAWN_COST = 50000;

    // NPC Related Constants \\
	public static final int ORATOR_NPC_ID = 31094;
	public static final int PREACHER_NPC_ID = 31093;
    public static final int MAMMON_MERCHANT_ID = 31113;
    public static final int MAMMON_BLACKSMITH_ID = 31126;
    public static final int MAMMON_MARKETEER_ID = 31092;
    public static final int SPIRIT_IN_ID = 31111;
    public static final int SPIRIT_OUT_ID = 31112;
    public static final int LILITH_NPC_ID = 25283;
    public static final int ANAKIM_NPC_ID = 25286;
    public static final int CREST_OF_DAWN_ID = 31170;
    public static final int CREST_OF_DUSK_ID = 31171;
	// Seal Stone Related Constants \\
	public static final int SEAL_STONE_BLUE_ID = 6360;
	public static final int SEAL_STONE_GREEN_ID = 6361;
	public static final int SEAL_STONE_RED_ID = 6362;

	public static final int SEAL_STONE_BLUE_VALUE = 3;
	public static final int SEAL_STONE_GREEN_VALUE = 5;
	public static final int SEAL_STONE_RED_VALUE = 10;

	public static final int BLUE_CONTRIB_POINTS = 3;
	public static final int GREEN_CONTRIB_POINTS = 5;
	public static final int RED_CONTRIB_POINTS = 10;

	private final Calendar _calendar = Calendar.getInstance();
	private        Calendar _savedCalendar;	//XXX:[JOJO]

    protected int _activePeriod;
    protected int _currentCycle;
    protected double _dawnStoneScore;
    protected double _duskStoneScore;
    protected int _dawnFestivalScore;
    protected int _duskFestivalScore;
    protected int _compWinner;
    protected int _previousWinner;

	private Map<Integer, StatsSet> _signsPlayerData;
    // [L2J_JP ADD START]
    private boolean _initializedNecCatSpawns = false;
    private List<L2NpcInstance> _signsNecCatMobInsts;
    // [L2J_JP ADD END]

	private Map<Integer, Integer> _signsSealOwners;
	private Map<Integer, Integer> _signsDuskSealTotals;
	private Map<Integer, Integer> _signsDawnSealTotals;

    private static AutoSpawnInstance _merchantSpawn;
    private static AutoSpawnInstance _blacksmithSpawn;
    private static AutoSpawnInstance _spiritInSpawn;
    private static AutoSpawnInstance _spiritOutSpawn;
    private static AutoSpawnInstance _lilithSpawn;
    private static AutoSpawnInstance _anakimSpawn;
    private static AutoSpawnInstance _crestofdawnspawn;
    private static AutoSpawnInstance _crestofduskspawn;
    private static Map<Integer, AutoSpawnInstance> _oratorSpawns;
    private static Map<Integer, AutoSpawnInstance> _preacherSpawns;
    private static Map<Integer, AutoSpawnInstance> _marketeerSpawns;

	public SevenSigns()
	{
		_signsPlayerData = new FastMap<Integer, StatsSet>();
		_signsSealOwners = new FastMap<Integer, Integer>();
		_signsDuskSealTotals = new FastMap<Integer, Integer>();
		_signsDawnSealTotals = new FastMap<Integer, Integer>();
        // [L2J_JP ADD]
        _signsNecCatMobInsts = new FastList<L2NpcInstance>();

		try
		{
			restoreSevenSignsData();
		}
		catch (Exception e) {
			_log.severe("SevenSigns: Failed to load configuration: " + e);
		}

		_log.info("SevenSigns: Currently in the " + getCurrentPeriodName() + " period!");
		initializeSeals();

		if (isSealValidationPeriod())
			if (getCabalHighestScore() == CABAL_NULL)
	            _log.info("SevenSigns: The competition ended with a tie last week.");
			else
	            _log.info("SevenSigns: The " + getCabalName(getCabalHighestScore()) + " were victorious last week.");
		else
			if (getCabalHighestScore() == CABAL_NULL)
				_log.info("SevenSigns: The competition, if the current trend continues, will end in a tie this week.");
			else
				_log.info("SevenSigns: The " + getCabalName(getCabalHighestScore()) + " are in the lead this week.");

		// [L2J_JP ADD]
		spawnNecropolisAndCatacomb();

		synchronized (this)
		{
			//XXX:[JOJO]
			// Schedule a time for the next period change.
			long milliToChange;
			if (_calendar.after(_savedCalendar)) {
				// jPWPXɃT[o_EĂƂ̑΍.
				// Oݒ肵J_[߂Ă܂ĂA̎Ȃ
				milliToChange = 1000;
			} else {
				// ]̏
				setCalendarForNextPeriodChange();
				milliToChange = getMilliToPeriodChange();
			}
			_savedCalendar = null;
			_log.info("SevenSigns: Next period begins at " + StrCalender());
			_log.info("SevenSigns: Next period begins in " + StrMillTime(milliToChange));
			SevenSignsPeriodChange sspc = new SevenSignsPeriodChange();
			ThreadPoolManager.getInstance().scheduleGeneral(sspc, milliToChange);
			//XXX:[JOJO]
		}
	}

	//XXX:[JOJO]
	protected String StrMillTime(double milliToChange)
	{
		double numSecs = (milliToChange / 1000) % 60;
		double countDown = ((milliToChange / 1000) - numSecs) / 60;
		int numMins = (int)Math.floor(countDown % 60);
		countDown = (countDown - numMins) / 60; 
		int numHours = (int)Math.floor(countDown % 24);
		int numDays = (int)Math.floor((countDown - numHours) / 24);
		return numDays + " " + numHours + " " + numMins + "";
	}

	protected String StrCalender(Calendar calendar)
	{
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss (E)");
		return dateFormat.format(calendar.getTime().getTime());
	}
	protected String StrCalender()
	{
		return StrCalender(_calendar);
	}
	//XXX:[JOJO]
	
    /**
     * Registers all random spawns and auto-chats for Seven Signs NPCs,
     * along with spawns for the Preachers of Doom and Orators of Revelations
     * at the beginning of the Seal Validation period.
     *
     */
	public void spawnSevenSignsNPC()
	{
		_merchantSpawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(MAMMON_MERCHANT_ID, false);
		_blacksmithSpawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(MAMMON_BLACKSMITH_ID, false);
		_marketeerSpawns = AutoSpawnHandler.getInstance().getAutoSpawnInstances(MAMMON_MARKETEER_ID);
		_spiritInSpawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(SPIRIT_IN_ID, false);
		_spiritOutSpawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(SPIRIT_OUT_ID, false);
		_lilithSpawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(LILITH_NPC_ID, false);
		_anakimSpawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(ANAKIM_NPC_ID, false);
		_crestofdawnspawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(CREST_OF_DAWN_ID, false);
		_crestofduskspawn = AutoSpawnHandler.getInstance().getAutoSpawnInstance(CREST_OF_DUSK_ID, false);
		_oratorSpawns = AutoSpawnHandler.getInstance().getAutoSpawnInstances(ORATOR_NPC_ID);
		_preacherSpawns = AutoSpawnHandler.getInstance().getAutoSpawnInstances(PREACHER_NPC_ID);

		if (isSealValidationPeriod() || isCompResultsPeriod())
		{
			for (AutoSpawnInstance spawnInst: _marketeerSpawns.values())
				AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, true);

			if (getSealOwner(SEAL_GNOSIS) == getCabalHighestScore() && getSealOwner(SEAL_GNOSIS) != CABAL_NULL)
			{
		        if (!Config.ANNOUNCE_MAMMON_SPAWN)
		            _blacksmithSpawn.setBroadcast(false);

		        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_blacksmithSpawn.getObjectId(), true).isSpawnActive())
		        	AutoSpawnHandler.getInstance().setSpawnActive(_blacksmithSpawn, true);

				for (AutoSpawnInstance spawnInst: _oratorSpawns.values())
					if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(spawnInst.getObjectId(), true).isSpawnActive())
						AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, true);

				for (AutoSpawnInstance spawnInst: _preacherSpawns.values())
					if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(spawnInst.getObjectId(), true).isSpawnActive())
						AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, true);

				if (!AutoChatHandler.getInstance().getAutoChatInstance(PREACHER_NPC_ID, false).isActive() && !AutoChatHandler.getInstance().getAutoChatInstance(ORATOR_NPC_ID, false).isActive())
					AutoChatHandler.getInstance().setAutoChatActive(true);
			}
	        else
	        {
				AutoSpawnHandler.getInstance().setSpawnActive(_blacksmithSpawn, false);

				for (AutoSpawnInstance spawnInst: _oratorSpawns.values())
					AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, false);

				for (AutoSpawnInstance spawnInst: _preacherSpawns.values())
					AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, false);

				AutoChatHandler.getInstance().setAutoChatActive(false);
	        }

	        if (getSealOwner(SEAL_AVARICE) == getCabalHighestScore() && getSealOwner(SEAL_AVARICE) != CABAL_NULL)
	        {
		        if (!Config.ANNOUNCE_MAMMON_SPAWN)
		            _merchantSpawn.setBroadcast(false);

		        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_merchantSpawn.getObjectId(), true).isSpawnActive())
		        	AutoSpawnHandler.getInstance().setSpawnActive(_merchantSpawn, true);

		        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_spiritInSpawn.getObjectId(), true).isSpawnActive())
		        	AutoSpawnHandler.getInstance().setSpawnActive(_spiritInSpawn, true);

		        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_spiritOutSpawn.getObjectId(), true).isSpawnActive())
		        	AutoSpawnHandler.getInstance().setSpawnActive(_spiritOutSpawn, true);

				switch (getCabalHighestScore())
		        {
		        	case CABAL_DAWN:
				        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_lilithSpawn.getObjectId(), true).isSpawnActive())
				        	AutoSpawnHandler.getInstance().setSpawnActive(_lilithSpawn, true);

		        		AutoSpawnHandler.getInstance().setSpawnActive(_anakimSpawn, false);
				        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_crestofdawnspawn.getObjectId(), true).isSpawnActive())
				        	AutoSpawnHandler.getInstance().setSpawnActive(_crestofdawnspawn, true);

		        		AutoSpawnHandler.getInstance().setSpawnActive(_crestofduskspawn, false);
		        		break;

		        	case CABAL_DUSK:
				        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_anakimSpawn.getObjectId(), true).isSpawnActive())
				        	AutoSpawnHandler.getInstance().setSpawnActive(_anakimSpawn, true);

		        		AutoSpawnHandler.getInstance().setSpawnActive(_lilithSpawn, false);
				        if (!AutoSpawnHandler.getInstance().getAutoSpawnInstance(_crestofduskspawn.getObjectId(), true).isSpawnActive())
				        	AutoSpawnHandler.getInstance().setSpawnActive(_crestofduskspawn, true);

		        		AutoSpawnHandler.getInstance().setSpawnActive(_crestofdawnspawn, false);
		        		break;
		        }
	        }
	        else
	        {
				AutoSpawnHandler.getInstance().setSpawnActive(_merchantSpawn, false);
				AutoSpawnHandler.getInstance().setSpawnActive(_lilithSpawn, false);
				AutoSpawnHandler.getInstance().setSpawnActive(_anakimSpawn, false);
				AutoSpawnHandler.getInstance().setSpawnActive(_crestofdawnspawn, false);
				AutoSpawnHandler.getInstance().setSpawnActive(_crestofduskspawn, false);
				AutoSpawnHandler.getInstance().setSpawnActive(_spiritInSpawn, false);
				AutoSpawnHandler.getInstance().setSpawnActive(_spiritOutSpawn, false);
	        }
		}
		else
		{
			AutoSpawnHandler.getInstance().setSpawnActive(_merchantSpawn, false);
			AutoSpawnHandler.getInstance().setSpawnActive(_blacksmithSpawn, false);
			AutoSpawnHandler.getInstance().setSpawnActive(_lilithSpawn, false);
			AutoSpawnHandler.getInstance().setSpawnActive(_anakimSpawn, false);
			AutoSpawnHandler.getInstance().setSpawnActive(_crestofdawnspawn, false);
			AutoSpawnHandler.getInstance().setSpawnActive(_crestofduskspawn, false);
			AutoSpawnHandler.getInstance().setSpawnActive(_spiritInSpawn, false);
			AutoSpawnHandler.getInstance().setSpawnActive(_spiritOutSpawn, false);

			for (AutoSpawnInstance spawnInst: _oratorSpawns.values())
				AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, false);

			for (AutoSpawnInstance spawnInst: _preacherSpawns.values())
				AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, false);

			for (AutoSpawnInstance spawnInst: _marketeerSpawns.values())
				// [L2J_JP EDIT SANDMAN -> marketeer is always active]
				AutoSpawnHandler.getInstance().setSpawnActive(spawnInst, true);

			AutoChatHandler.getInstance().setAutoChatActive(false);
		}
	}

	public static SevenSigns getInstance()
    {
		if (_instance == null)
			_instance = new SevenSigns();

		return _instance;
	}

	public static int calcContributionScore(int blueCount, int greenCount, int redCount)
	{
		int contrib = blueCount * BLUE_CONTRIB_POINTS;
		contrib += greenCount * GREEN_CONTRIB_POINTS;
		contrib += redCount * RED_CONTRIB_POINTS;

		return contrib;
	}

	public static int calcAncientAdenaReward(int blueCount, int greenCount, int redCount)
	{
		int reward = blueCount * SEAL_STONE_BLUE_VALUE;
		reward += greenCount * SEAL_STONE_GREEN_VALUE;
		reward += redCount * SEAL_STONE_RED_VALUE;

		return reward;
	}

	public static final String getCabalShortName(int cabal)
	{
		switch (cabal)
		{
			case CABAL_DAWN:
				return "dawn";
			case CABAL_DUSK:
				return "dusk";
		}

		return "No Cabal";
	}

	public static final String getCabalName(int cabal)
	{
		switch (cabal)
		{
			case CABAL_DAWN:
				return "Lords of Dawn";
			case CABAL_DUSK:
				return "Revolutionaries of Dusk";
		}

		return "No Cabal";
	}

	public static final String getSealName(int seal, boolean shortName)
	{
		String sealName = (!shortName) ? "Seal of " : "";

		switch (seal)
		{
			case SEAL_AVARICE:
				sealName += "Avarice";
				break;
			case SEAL_GNOSIS:
				sealName += "Gnosis";
				break;
			case SEAL_STRIFE:
				sealName += "Strife";
				break;
		}

		return sealName;
	}

	public final int getCurrentCycle()
    {
		return _currentCycle;
	}

	public final int getCurrentPeriod()
    {
		return _activePeriod;
	}

	private final int getDaysToPeriodChange()
    {
		int numDays = _calendar.get(Calendar.DAY_OF_WEEK) - PERIOD_START_DAY;

		if (numDays < 0)
			return 0 - numDays;

        return 7 - numDays;
	}

    public final long getMilliToPeriodChange()
    {
        long currTimeMillis = System.currentTimeMillis();
        long changeTimeMillis = _calendar.getTimeInMillis();

        return (changeTimeMillis - currTimeMillis);
    }

    protected void setCalendarForNextPeriodChange()
    {
        // Calculate the number of days until the next period
        // A period starts at 18:00 pm (local time), like on official servers.
        switch (getCurrentPeriod())
        {
            case PERIOD_SEAL_VALIDATION:
            case PERIOD_COMPETITION:
                int daysToChange = getDaysToPeriodChange();

                if (daysToChange == 7)
                	if (_calendar.get(Calendar.HOUR_OF_DAY) < PERIOD_START_HOUR)
                		daysToChange = 0;
                	else if (_calendar.get(Calendar.HOUR_OF_DAY) == PERIOD_START_HOUR && _calendar.get(Calendar.MINUTE) < PERIOD_START_MINS)
                		daysToChange = 0;

                // Otherwise...
                if (daysToChange > 0)
                    _calendar.add(Calendar.DATE, daysToChange);

                _calendar.set(Calendar.HOUR_OF_DAY, PERIOD_START_HOUR);
                _calendar.set(Calendar.MINUTE, PERIOD_START_MINS);
                break;
            case PERIOD_COMP_RECRUITING:
            case PERIOD_COMP_RESULTS:
                _calendar.add(Calendar.MILLISECOND, PERIOD_MINOR_LENGTH);
                break;
        }
        //XXX:[JOJO]
        _calendar.set(Calendar.SECOND, 0);
		saveSevenSignsCalendar();	// _calendar ۑ
		//XXX:[JOJO]
    }

	public final String getCurrentPeriodName()
    {
		String periodName = null;

		switch (_activePeriod)
		{
			case PERIOD_COMP_RECRUITING:
				periodName = ""; //"Quest Event Initialization";
				break;
			case PERIOD_COMPETITION:
				periodName = "";//"Competition (Quest Event)";
				break;
			case PERIOD_COMP_RESULTS:
				periodName = "Wv";//"Quest Event Results";
				break;
			case PERIOD_SEAL_VALIDATION:
				periodName = "L";//"Seal Validation";
				break;
		}

		return periodName;
	}

	public final boolean isSealValidationPeriod()
	{
		return (_activePeriod == PERIOD_SEAL_VALIDATION);
	}

	public final boolean isCompResultsPeriod()
	{
		return (_activePeriod == PERIOD_COMP_RESULTS);
	}
	
	/**
	 * returns true if the given date is in Seal Validation or in Quest Event Results period 
	 * @param date
	 */
	public boolean isDateInSealValidPeriod(Calendar date)
	{
        long nextPeriodChange = getMilliToPeriodChange();
        long nextQuestStart = 0;
        long nextValidStart = 0;
        long tillDate = date.getTimeInMillis() - Calendar.getInstance().getTimeInMillis();
        while ((2 * PERIOD_MAJOR_LENGTH + 2 * PERIOD_MINOR_LENGTH) < tillDate)
        	tillDate -= (2 * PERIOD_MAJOR_LENGTH + 2 * PERIOD_MINOR_LENGTH);
        while (tillDate < 0)
        	tillDate += (2 * PERIOD_MAJOR_LENGTH + 2 * PERIOD_MINOR_LENGTH);
        
		switch (getCurrentPeriod())
		{
			case PERIOD_COMP_RECRUITING:
				nextValidStart = nextPeriodChange + PERIOD_MAJOR_LENGTH;
				nextQuestStart = nextValidStart + PERIOD_MAJOR_LENGTH + PERIOD_MINOR_LENGTH;
				break;
			case PERIOD_COMPETITION:
				nextValidStart = nextPeriodChange;
				nextQuestStart = nextPeriodChange + PERIOD_MAJOR_LENGTH + PERIOD_MINOR_LENGTH;
				break;
			case PERIOD_COMP_RESULTS:
				nextQuestStart = nextPeriodChange + PERIOD_MAJOR_LENGTH;
				nextValidStart = nextQuestStart + PERIOD_MAJOR_LENGTH + PERIOD_MINOR_LENGTH;
				break;
			case PERIOD_SEAL_VALIDATION:
				nextQuestStart = nextPeriodChange;
				nextValidStart = nextPeriodChange + PERIOD_MAJOR_LENGTH + PERIOD_MINOR_LENGTH;				
				break;
		}
		
		if ((nextQuestStart < tillDate && tillDate < nextValidStart) ||
				(nextValidStart < nextQuestStart && (tillDate < nextValidStart || nextQuestStart < tillDate)))
			return false;
		return true;
	}

	public final int getCurrentScore(int cabal)
	{
        double totalStoneScore = _dawnStoneScore + _duskStoneScore;

		switch (cabal)
		{
			case CABAL_NULL:
				return 0;
			case CABAL_DAWN:
				return Math.round((float)(_dawnStoneScore / ((float)totalStoneScore == 0 ? 1 : totalStoneScore)) * 500) + _dawnFestivalScore;
			case CABAL_DUSK:
				return Math.round((float)(_duskStoneScore / ((float)totalStoneScore == 0 ? 1 : totalStoneScore)) * 500) + _duskFestivalScore;
		}

		return 0;
	}

	public final double getCurrentStoneScore(int cabal)
	{
		switch (cabal)
		{
			case CABAL_NULL:
				return 0;
			case CABAL_DAWN:
				return _dawnStoneScore;
			case CABAL_DUSK:
				return _duskStoneScore;
		}

		return 0;
	}

	public final int getCurrentFestivalScore(int cabal)
	{
		switch (cabal)
		{
			case CABAL_NULL:
				return 0;
			case CABAL_DAWN:
				return _dawnFestivalScore;
			case CABAL_DUSK:
				return _duskFestivalScore;
		}

		return 0;
	}

	public final int getCabalHighestScore()
	{
		if (getCurrentScore(CABAL_DUSK) == getCurrentScore(CABAL_DAWN))
			return CABAL_NULL;
		else if (getCurrentScore(CABAL_DUSK) > getCurrentScore(CABAL_DAWN))
			return CABAL_DUSK;
		else
			return CABAL_DAWN;
	}

	public final int getSealOwner(int seal)
	{
		return _signsSealOwners.get(seal);
	}

	public final int getSealProportion(int seal, int cabal)
	{
		if (cabal == CABAL_NULL)
			return 0;
		else if (cabal == CABAL_DUSK)
			return _signsDuskSealTotals.get(seal);
		else
			return _signsDawnSealTotals.get(seal);
	}

	public final int getTotalMembers(int cabal)
	{
		int cabalMembers = 0;
		String cabalName = getCabalShortName(cabal);

		for (StatsSet sevenDat : _signsPlayerData.values())
			if (sevenDat.getString("cabal").equals(cabalName))
				cabalMembers++;

		return cabalMembers;
	}

    public final StatsSet getPlayerData(L2PcInstance player)
    {
        if (!hasRegisteredBefore(player))
            return null;

        return _signsPlayerData.get(player.getObjectId());
    }

	public int getPlayerStoneContrib(L2PcInstance player)
	{
		if (!hasRegisteredBefore(player))
			return 0;

		int stoneCount = 0;

		StatsSet currPlayer = getPlayerData(player);

		stoneCount += currPlayer.getInteger("red_stones");
		stoneCount += currPlayer.getInteger("green_stones");
		stoneCount += currPlayer.getInteger("blue_stones");

		return stoneCount;
	}

	public int getPlayerContribScore(L2PcInstance player)
	{
		if (!hasRegisteredBefore(player))
			return 0;

		StatsSet currPlayer = getPlayerData(player);

		return currPlayer.getInteger("contribution_score");
	}

	public int getPlayerAdenaCollect(L2PcInstance player)
	{
		if (!hasRegisteredBefore(player))
			return 0;

		return _signsPlayerData.get(player.getObjectId()).getInteger("ancient_adena_amount");
	}

	public int getPlayerSeal(L2PcInstance player)
	{
		if (!hasRegisteredBefore(player))
			return SEAL_NULL;

		return getPlayerData(player).getInteger("seal");
	}

	public int getPlayerCabal(L2PcInstance player)
	{
		if (!hasRegisteredBefore(player))
			return CABAL_NULL;

		String playerCabal = getPlayerData(player).getString("cabal");

		if (playerCabal.equalsIgnoreCase("dawn"))
			return CABAL_DAWN;
		else if (playerCabal.equalsIgnoreCase("dusk"))
			return CABAL_DUSK;
		else
			return CABAL_NULL;
	}

    /**
     * Restores all Seven Signs data and settings, usually called at server startup.
     *
     * @throws Exception
     */
    protected void restoreSevenSignsData()
	{
    	Connection con = null;
    	PreparedStatement statement = null;
    	ResultSet rset = null;

    	try
    	{
	    	con = L2DatabaseFactory.getInstance().getConnection();
		    statement = con.prepareStatement("SELECT charId, cabal, seal, red_stones, green_stones, blue_stones, " +
		    	"ancient_adena_amount, contribution_score FROM seven_signs");
		    rset = statement.executeQuery();

		    while (rset.next())
		    {
		    	int charObjId = rset.getInt("charId");

		    	StatsSet sevenDat = new StatsSet();
				sevenDat.set("charId", charObjId);
				sevenDat.set("cabal", rset.getString("cabal"));
				sevenDat.set("seal", rset.getInt("seal"));
				sevenDat.set("red_stones", rset.getInt("red_stones"));
				sevenDat.set("green_stones", rset.getInt("green_stones"));
				sevenDat.set("blue_stones", rset.getInt("blue_stones"));
				sevenDat.set("ancient_adena_amount", rset.getDouble("ancient_adena_amount"));
				sevenDat.set("contribution_score", rset.getDouble("contribution_score"));

                if (Config.DEBUG)
					_log.info("SevenSigns: Loaded data from DB for char ID " + charObjId + " (" + sevenDat.getString("cabal") + ")");

				_signsPlayerData.put(charObjId, sevenDat);
		    }

		    rset.close();
		    statement.close();

		    statement = con.prepareStatement("SELECT * FROM seven_signs_status WHERE id=0");
	        rset = statement.executeQuery();

	        while (rset.next())
	        {
	            _currentCycle = rset.getInt("current_cycle");
	            _activePeriod = rset.getInt("active_period");
	            _previousWinner = rset.getInt("previous_winner");

	            _dawnStoneScore = rset.getDouble("dawn_stone_score");
	            _dawnFestivalScore = rset.getInt("dawn_festival_score");
	            _duskStoneScore = rset.getDouble("dusk_stone_score");
	            _duskFestivalScore = rset.getInt("dusk_festival_score");

	            _signsSealOwners.put(SEAL_AVARICE, rset.getInt("avarice_owner"));
	            _signsSealOwners.put(SEAL_GNOSIS, rset.getInt("gnosis_owner"));
	            _signsSealOwners.put(SEAL_STRIFE, rset.getInt("strife_owner"));

	            _signsDawnSealTotals.put(SEAL_AVARICE, rset.getInt("avarice_dawn_score"));
	            _signsDawnSealTotals.put(SEAL_GNOSIS, rset.getInt("gnosis_dawn_score"));
	            _signsDawnSealTotals.put(SEAL_STRIFE, rset.getInt("strife_dawn_score"));
	            _signsDuskSealTotals.put(SEAL_AVARICE, rset.getInt("avarice_dusk_score"));
	            _signsDuskSealTotals.put(SEAL_GNOSIS, rset.getInt("gnosis_dusk_score"));
	            _signsDuskSealTotals.put(SEAL_STRIFE, rset.getInt("strife_dusk_score"));

				//XXX:[JOJO]
	            _savedCalendar = Calendar.getInstance();
	            _savedCalendar.setTime(rset.getTimestamp("calendar", _savedCalendar));
				_log.info("SevenSigns: restore period begins at " + StrCalender(_savedCalendar));
	           	//XXX:[JOJO]
	        }

	        rset.close();
	        statement.close();

	        statement = con.prepareStatement("UPDATE seven_signs_status SET date=? WHERE id=0");
	        statement.setInt(1, Calendar.getInstance().get(Calendar.DAY_OF_WEEK));
	        statement.execute();

	        statement.close();
	        con.close();
    	}
    	catch (SQLException e)
    	{
    		_log.severe("SevenSigns: Unable to load Seven Signs data from database: " + e);
    	}
    	finally
    	{
    		try
    		{
    		    rset.close();
    		    statement.close();
    		    con.close();
    		}
    		catch (Exception e) {}
    	}

		// Festival data is loaded now after the Seven Signs engine data.
	}

	/**
     * Saves all Seven Signs data, both to the database and properties file (if updateSettings = True).
     * Often called to preserve data integrity and synchronization with DB, in case of errors.
     * <BR>
     * If player != null, just that player's data is updated in the database, otherwise all player's data is
     * sequentially updated.
     *
	 * @param player
	 * @param updateSettings
	 * @throws Exception
	 */
	public void saveSevenSignsData(L2PcInstance player, boolean updateSettings)
	{
		Connection con = null;
		PreparedStatement statement = null;

        if (Config.DEBUG)
            _log.info("SevenSigns: Saving data to disk.");

		try
		{
			con = L2DatabaseFactory.getInstance().getConnection();

			for (StatsSet sevenDat : _signsPlayerData.values())
		    {
		    	if (player != null)
		    		if (sevenDat.getInteger("charId") != player.getObjectId())
		    			continue;

		    	statement = con.prepareStatement(
		    		"UPDATE seven_signs SET cabal=?, seal=?, red_stones=?, " +
		    		"green_stones=?, blue_stones=?, " +
		    		"ancient_adena_amount=?, contribution_score=? " +
		    		"WHERE charId=?");
		    	statement.setString(1, sevenDat.getString("cabal"));
		    	statement.setInt(2, sevenDat.getInteger("seal"));
		    	statement.setInt(3, sevenDat.getInteger("red_stones"));
		    	statement.setInt(4, sevenDat.getInteger("green_stones"));
		    	statement.setInt(5, sevenDat.getInteger("blue_stones"));
		    	statement.setDouble(6, sevenDat.getDouble("ancient_adena_amount"));
		    	statement.setDouble(7, sevenDat.getDouble("contribution_score"));
		    	statement.setInt(8, sevenDat.getInteger("charId"));
		    	statement.execute();

		    	statement.close();

				if (Config.DEBUG)
					_log.info("SevenSigns: Updated data in database for char ID " + sevenDat.getInteger("charId") + " (" + sevenDat.getString("cabal") + ")");
		    }

	        if (updateSettings)
	        {
	        	String sqlQuery = "UPDATE seven_signs_status SET current_cycle=?, active_period=?, previous_winner=?, " +
	               "dawn_stone_score=?, dawn_festival_score=?, dusk_stone_score=?, dusk_festival_score=?, " +
	               "avarice_owner=?, gnosis_owner=?, strife_owner=?, avarice_dawn_score=?, gnosis_dawn_score=?, " +
	               "strife_dawn_score=?, avarice_dusk_score=?, gnosis_dusk_score=?, strife_dusk_score=?, " +
	               "festival_cycle=?, ";

	            for (int i = 0; i < (SevenSignsFestival.FESTIVAL_COUNT); i++)
	                sqlQuery += "accumulated_bonus" + String.valueOf(i) + "=?, ";

                sqlQuery += "dateupdate=CURRENT_TIMESTAMP, " + "date=? WHERE id=0";

	            statement = con.prepareStatement(sqlQuery);
	            statement.setInt(1, _currentCycle);
	            statement.setInt(2, _activePeriod);
	            statement.setInt(3, _previousWinner);
	            statement.setDouble(4, _dawnStoneScore);
	            statement.setInt(5, _dawnFestivalScore);
	            statement.setDouble(6, _duskStoneScore);
	            statement.setInt(7, _duskFestivalScore);
	            statement.setInt(8, _signsSealOwners.get(SEAL_AVARICE));
	            statement.setInt(9, _signsSealOwners.get(SEAL_GNOSIS));
	            statement.setInt(10, _signsSealOwners.get(SEAL_STRIFE));
	            statement.setInt(11, _signsDawnSealTotals.get(SEAL_AVARICE));
	            statement.setInt(12, _signsDawnSealTotals.get(SEAL_GNOSIS));
	            statement.setInt(13, _signsDawnSealTotals.get(SEAL_STRIFE));
	            statement.setInt(14, _signsDuskSealTotals.get(SEAL_AVARICE));
	            statement.setInt(15, _signsDuskSealTotals.get(SEAL_GNOSIS));
	            statement.setInt(16, _signsDuskSealTotals.get(SEAL_STRIFE));
	            statement.setInt(17, SevenSignsFestival.getInstance().getCurrentFestivalCycle());

	            for (int i = 0; i < SevenSignsFestival.FESTIVAL_COUNT; i++)
	                statement.setInt(18 + i, SevenSignsFestival.getInstance().getAccumulatedBonus(i));

	            statement.setInt(18 + SevenSignsFestival.FESTIVAL_COUNT, Calendar.getInstance().get(Calendar.DAY_OF_WEEK));
	            statement.execute();

	            statement.close();
		        con.close();

	            if (Config.DEBUG)
	                _log.info("SevenSigns: Updated data in database.");

	        }
		}
		catch (SQLException e)
		{
			_log.severe("SevenSigns: Unable to save data to database: " + e);
		}
		finally
        {
    		try	{
    		    statement.close();
    		    con.close();
    		}
    		catch (Exception e) {}
		}
	}

	/**
	 * TODO:[JOJO] Save _calendar to disk
	 */
    public void saveSevenSignsCalendar()
	{
    	Connection con = null;
		PreparedStatement statement = null;

		if (Config.DEBUG)
			System.out.println("SevenSigns: Saving calendar to disk.");

		try
		{
			con = L2DatabaseFactory.getInstance().getConnection();
			
			String sqlQuery = "UPDATE seven_signs_status SET active_period=?, date=?, dateupdate=CURRENT_TIMESTAMP, calendar=? WHERE id=0";
			statement = con.prepareStatement(sqlQuery);

			statement.setInt(1, _activePeriod);
			statement.setInt(2, Calendar.getInstance().get(Calendar.DAY_OF_WEEK));
			statement.setTimestamp(3, new java.sql.Timestamp(_calendar.getTime().getTime()));

			statement.execute();
			statement.close();
			con.close();

			if (Config.DEBUG)
			    _log.info("SevenSigns: Updated calendar in database.");
		}
		catch (SQLException e)
		{
			_log.severe("SevenSigns: Unable to save calendar to database: " + e);
		}
		finally	
		{
			try	{
    			statement.close();
    		    con.close();
    		}
    		catch (Exception e) {}
		}
	}

    /**
     * Used to reset the cabal details of all players, and update the database.<BR>
     * Primarily used when beginning a new cycle, and should otherwise never be called.
     */
    protected void resetPlayerData()
	{
		if (Config.DEBUG)
			_log.info("SevenSigns: Resetting player data for new event period.");

		// Reset each player's contribution data as well as seal and cabal.
		for (StatsSet sevenDat : _signsPlayerData.values())
		{
		    int charObjId = sevenDat.getInteger("charId");

			// Reset the player's cabal and seal information
			sevenDat.set("cabal", "");
			sevenDat.set("seal", SEAL_NULL);
            sevenDat.set("contribution_score", 0);

            _signsPlayerData.put(charObjId, sevenDat);
		}
	}

	/**
     * Tests whether the specified player has joined a cabal in the past.
     *
	 * @param player
	 * @return boolean hasRegistered
	 */
	private boolean hasRegisteredBefore(L2PcInstance player)
    {
		return _signsPlayerData.containsKey(player.getObjectId());
	}


	/**
     * Used to specify cabal-related details for the specified player. This method
     * checks to see if the player has registered before and will update the database
     * if necessary.
     * <BR>
     * Returns the cabal ID the player has joined.
     *
	 * @param player
	 * @param chosenCabal
	 * @param chosenSeal
	 * @return int cabal
	 */
	public int setPlayerInfo(L2PcInstance player, int chosenCabal, int chosenSeal)
	{
		int charObjId = player.getObjectId();
		Connection con = null;
		PreparedStatement statement = null;
		StatsSet currPlayerData = getPlayerData(player);

		if (currPlayerData != null)
		{
			// If the seal validation period has passed,
			// cabal information was removed and so "re-register" player
			currPlayerData.set("cabal", getCabalShortName(chosenCabal));
			currPlayerData.set("seal", chosenSeal);

			_signsPlayerData.put(charObjId, currPlayerData);
		}
		else
		{
			currPlayerData = new StatsSet();
			currPlayerData.set("charId", charObjId);
			currPlayerData.set("cabal", getCabalShortName(chosenCabal));
			currPlayerData.set("seal", chosenSeal);
			currPlayerData.set("red_stones", 0);
			currPlayerData.set("green_stones", 0);
			currPlayerData.set("blue_stones", 0);
			currPlayerData.set("ancient_adena_amount", 0);
			currPlayerData.set("contribution_score", 0);

			_signsPlayerData.put(charObjId, currPlayerData);

			// Update data in database, as we have a new player signing up.
			try
			{
				con = L2DatabaseFactory.getInstance().getConnection();
				statement = con.prepareStatement(
		    		"INSERT INTO seven_signs (charId, cabal, seal) VALUES (?,?,?)");
				statement.setInt(1, charObjId);
				statement.setString(2, getCabalShortName(chosenCabal));
				statement.setInt(3, chosenSeal);
				statement.execute();

				statement.close();
				con.close();

				if (Config.DEBUG)
					_log.info("SevenSigns: Inserted data in DB for char ID " + currPlayerData.getInteger("charId") + " (" + currPlayerData.getString("cabal") + ")");
			}
			catch (SQLException e)
			{
				_log.severe("SevenSigns: Failed to save data: " + e);
			}
			finally
			{
	    		try
                {
	    		    statement.close();
	    		    con.close();
	    		}
	    		catch (Exception e) {}
			}
		}

		// Increasing Seal total score for the player chosen Seal.
		if (currPlayerData.getString("cabal") == "dawn")
			_signsDawnSealTotals.put(chosenSeal, _signsDawnSealTotals.get(chosenSeal) + 1);
		else
			_signsDuskSealTotals.put(chosenSeal, _signsDuskSealTotals.get(chosenSeal) + 1);

        saveSevenSignsData(player, true);

		if (Config.DEBUG)
			_log.info("SevenSigns: " + player.getName() + " has joined the " + getCabalName(chosenCabal) + " for the " + getSealName(chosenSeal, false) + "!");

		return chosenCabal;
	}

	/**
     * Returns the amount of ancient adena the specified player can claim, if any.<BR>
     * If removeReward = True, all the ancient adena owed to them is removed, then
     * DB is updated.
     *
	 * @param player
	 * @param removeReward
	 * @return int rewardAmount
	 */
	public int getAncientAdenaReward(L2PcInstance player, boolean removeReward)
    {
		StatsSet currPlayer = getPlayerData(player);
		int rewardAmount = currPlayer.getInteger("ancient_adena_amount");

		currPlayer.set("red_stones", 0);
        currPlayer.set("green_stones", 0);
        currPlayer.set("blue_stones", 0);
        currPlayer.set("ancient_adena_amount", 0);

		if (removeReward)
		{
			_signsPlayerData.put(player.getObjectId(), currPlayer);
			saveSevenSignsData(player, true);
		}

		return rewardAmount;
	}

    /**
     * Used to add the specified player's seal stone contribution points
     * to the current total for their cabal. Returns the point score the
     * contribution was worth.
     *
     * Each stone count <B>must be</B> broken down and specified by the stone's color.
     *
	 * @param player
	 * @param blueCount
	 * @param greenCount
	 * @param redCount
	 * @return int contribScore
	 */
	public int addPlayerStoneContrib(L2PcInstance player, int blueCount, int greenCount, int redCount)
	{
		StatsSet currPlayer = getPlayerData(player);

		int contribScore = calcContributionScore(blueCount, greenCount, redCount);
		int totalAncientAdena = currPlayer.getInteger("ancient_adena_amount") + calcAncientAdenaReward(blueCount, greenCount, redCount);
		int totalContribScore = currPlayer.getInteger("contribution_score") + contribScore;

		if (totalContribScore > Config.ALT_MAXIMUM_PLAYER_CONTRIB)
			return -1;

		currPlayer.set("red_stones", currPlayer.getInteger("red_stones") + redCount);
		currPlayer.set("green_stones", currPlayer.getInteger("green_stones") + greenCount);
		currPlayer.set("blue_stones", currPlayer.getInteger("blue_stones") + blueCount);
		currPlayer.set("ancient_adena_amount", totalAncientAdena);
		currPlayer.set("contribution_score", totalContribScore);
		_signsPlayerData.put(player.getObjectId(), currPlayer);

        switch (getPlayerCabal(player))
        {
            case CABAL_DAWN:
                _dawnStoneScore += contribScore;
                break;
            case CABAL_DUSK:
                _duskStoneScore += contribScore;
                break;
        }

		saveSevenSignsData(player, true);

		if (Config.DEBUG)
			_log.info("SevenSigns: " + player.getName() + " contributed " + contribScore + " seal stone points to their cabal.");

		return contribScore;
	}

	/**
     * Adds the specified number of festival points to the specified cabal.
     * Remember, the same number of points are <B>deducted from the rival cabal</B>
     * to maintain proportionality.
     *
	 * @param cabal
	 * @param amount
	 */
	public void addFestivalScore(int cabal, int amount)
    {
		if (cabal == CABAL_DUSK) {
			_duskFestivalScore += amount;

            // To prevent negative scores!
            if (_dawnFestivalScore >= amount)
                _dawnFestivalScore -= amount;
        }
		else {
			_dawnFestivalScore += amount;

            if (_duskFestivalScore >= amount)
                _duskFestivalScore -= amount;
        }
	}

	/**
     * Send info on the current Seven Signs period to the specified player.
     *
	 * @param player
	 */
	public void sendCurrentPeriodMsg(L2PcInstance player)
	{
        SystemMessage sm = null;

    	switch (getCurrentPeriod())
        {
            case PERIOD_COMP_RECRUITING:
                sm = new SystemMessage(SystemMessageId.PREPARATIONS_PERIOD_BEGUN);
                break;
            case PERIOD_COMPETITION:
                sm = new SystemMessage(SystemMessageId.COMPETITION_PERIOD_BEGUN);
                break;
            case PERIOD_COMP_RESULTS:
                sm = new SystemMessage(SystemMessageId.RESULTS_PERIOD_BEGUN);
                break;
            case PERIOD_SEAL_VALIDATION:
                sm = new SystemMessage(SystemMessageId.VALIDATION_PERIOD_BEGUN);
                break;
        }

   		player.sendPacket(sm);
	}

	/**
     * Sends the built-in system message specified by sysMsgId to all online players.
     *
	 * @param sysMsgId
	 */
	public void sendMessageToAll(SystemMessageId sysMsgId)
	{
        SystemMessage sm = new SystemMessage(sysMsgId);
        Broadcast.toAllOnlinePlayers(sm);
	}

    /**
     * Used to initialize the seals for each cabal. (Used at startup or at beginning of a new cycle).
     * This method should  be called after <B>resetSeals()</B> and <B>calcNewSealOwners()</B> on a new cycle.
     */
    protected void initializeSeals()
	{
		for (Integer currSeal : _signsSealOwners.keySet())
		{
			int sealOwner = _signsSealOwners.get(currSeal);

			if (sealOwner != CABAL_NULL)
				if (isSealValidationPeriod())
					_log.info("SevenSigns: The " + getCabalName(sealOwner) + " have won the " + getSealName(currSeal, false) + ".");
				else
					_log.info("SevenSigns: The " + getSealName(currSeal, false) + " is currently owned by " + getCabalName(sealOwner) + ".");
			else
				_log.info("SevenSigns: The " + getSealName(currSeal, false) + " remains unclaimed.");
		}
	}

    /**
     * Only really used at the beginning of a new cycle, this method resets all seal-related data.
     */
    protected void resetSeals()
	{
		_signsDawnSealTotals.put(SEAL_AVARICE, 0);
		_signsDawnSealTotals.put(SEAL_GNOSIS, 0);
		_signsDawnSealTotals.put(SEAL_STRIFE, 0);
		_signsDuskSealTotals.put(SEAL_AVARICE, 0);
		_signsDuskSealTotals.put(SEAL_GNOSIS, 0);
		_signsDuskSealTotals.put(SEAL_STRIFE, 0);
	}

    /**
     * Calculates the ownership of the three Seals of the Seven Signs,
     * based on various criterion.
     * <BR><BR>
     * Should only ever called at the beginning of a new cycle.
     */
    protected void calcNewSealOwners()
	{
		if (Config.DEBUG)
		{
			_log.info("SevenSigns: (Avarice) Dawn = " + _signsDawnSealTotals.get(SEAL_AVARICE) + ", Dusk = " + _signsDuskSealTotals.get(SEAL_AVARICE));
			_log.info("SevenSigns: (Gnosis) Dawn = " + _signsDawnSealTotals.get(SEAL_GNOSIS) + ", Dusk = " + _signsDuskSealTotals.get(SEAL_GNOSIS));
			_log.info("SevenSigns: (Strife) Dawn = " + _signsDawnSealTotals.get(SEAL_STRIFE) + ", Dusk = " + _signsDuskSealTotals.get(SEAL_STRIFE));
		}

		for (Integer currSeal : _signsDawnSealTotals.keySet())
		{
			int prevSealOwner = _signsSealOwners.get(currSeal);
			int newSealOwner = CABAL_NULL;
			int dawnProportion = getSealProportion(currSeal, CABAL_DAWN);
			int totalDawnMembers = getTotalMembers(CABAL_DAWN) == 0 ? 1 : getTotalMembers(CABAL_DAWN);
            int dawnPercent = Math.round(((float)dawnProportion / (float)totalDawnMembers) * 100);
			int duskProportion = getSealProportion(currSeal, CABAL_DUSK);
			int totalDuskMembers = getTotalMembers(CABAL_DUSK) == 0 ? 1 : getTotalMembers(CABAL_DUSK);
            int duskPercent = Math.round(((float)duskProportion / (float)totalDuskMembers) * 100);

			/*
			 * - If a Seal was already closed or owned by the opponent and the new winner wants
			 *   to assume ownership of the Seal, 35% or more of the members of the Cabal must
			 *   have chosen the Seal. If they chose less than 35%, they cannot own the Seal.
			 *
			 * - If the Seal was owned by the winner in the previous Seven Signs, they can retain
			 *   that seal if 10% or more members have chosen it. If they want to possess a new Seal,
			 *   at least 35% of the members of the Cabal must have chosen the new Seal.
			 */
			switch (prevSealOwner)
			{
				case CABAL_NULL:
					switch (getCabalHighestScore())
					{
						case CABAL_NULL:
							newSealOwner = CABAL_NULL;
							break;
						case CABAL_DAWN:
							if (dawnPercent >= 35)
								newSealOwner = CABAL_DAWN;
							else
								newSealOwner = CABAL_NULL;
							break;
						case CABAL_DUSK:
							if (duskPercent >= 35)
								newSealOwner = CABAL_DUSK;
							else
								newSealOwner = CABAL_NULL;
							break;
					}
					break;
				case CABAL_DAWN:
					switch (getCabalHighestScore())
					{
						case CABAL_NULL:
							if (dawnPercent >= 10)
								newSealOwner = CABAL_DAWN;
							else
								newSealOwner = CABAL_NULL;
							break;
						case CABAL_DAWN:
							if (dawnPercent >= 10)
								newSealOwner = CABAL_DAWN;
							else
								newSealOwner = CABAL_NULL;
							break;
						case CABAL_DUSK:
							if (duskPercent >= 35)
								newSealOwner = CABAL_DUSK;
							else if (dawnPercent >= 10)
								newSealOwner = CABAL_DAWN;
							else
								newSealOwner = CABAL_NULL;
							break;
					}
					break;
				case CABAL_DUSK:
					switch (getCabalHighestScore())
					{
						case CABAL_NULL:
							if (duskPercent >= 10)
								newSealOwner = CABAL_DUSK;
							else
								newSealOwner = CABAL_NULL;
							break;
						case CABAL_DAWN:
							if (dawnPercent >= 35)
								newSealOwner = CABAL_DAWN;
							else if (duskPercent >= 10)
								newSealOwner = CABAL_DUSK;
							else
								newSealOwner = CABAL_NULL;
							break;
						case CABAL_DUSK:
							if (duskPercent >= 10)
								newSealOwner = CABAL_DUSK;
							else
								newSealOwner = CABAL_NULL;
							break;
					}
					break;
			}

			_signsSealOwners.put(currSeal, newSealOwner);

			// Alert all online players to new seal status.
			switch (currSeal)
			{
				case SEAL_AVARICE:
					if (newSealOwner == CABAL_DAWN)
						sendMessageToAll(SystemMessageId.DAWN_OBTAINED_AVARICE);
					else if (newSealOwner == CABAL_DUSK)
						sendMessageToAll(SystemMessageId.DUSK_OBTAINED_AVARICE);
					break;
				case SEAL_GNOSIS:
					if (newSealOwner == CABAL_DAWN)
						sendMessageToAll(SystemMessageId.DAWN_OBTAINED_GNOSIS);
					else if (newSealOwner == CABAL_DUSK)
						sendMessageToAll(SystemMessageId.DUSK_OBTAINED_GNOSIS);
					break;
				case SEAL_STRIFE:
					if (newSealOwner == CABAL_DAWN)
						sendMessageToAll(SystemMessageId.DAWN_OBTAINED_STRIFE);
					else if (newSealOwner == CABAL_DUSK)
						sendMessageToAll(SystemMessageId.DUSK_OBTAINED_STRIFE);

					CastleManager.getInstance().validateTaxes(newSealOwner);
					break;
			}
		}
	}

    /**
     * This method is called to remove all players from catacombs and
     * necropolises, who belong to the losing cabal.
     * <BR><BR>
     * Should only ever called at the beginning of Seal Validation.
     */
    protected void teleLosingCabalFromDungeons(String compWinner)
    {
    	Collection<L2PcInstance> pls = L2World.getInstance().getAllPlayers().values();
    	//synchronized (L2World.getInstance().getAllPlayers())
    	{
    		for (L2PcInstance onlinePlayer : pls)
    		{
    			StatsSet currPlayer = getPlayerData(onlinePlayer);

    			if (isSealValidationPeriod() || isCompResultsPeriod())
    			{
    				if (!onlinePlayer.isGM() && onlinePlayer.isIn7sDungeon() && !currPlayer.getString("cabal").equals(compWinner))
    				{
    					onlinePlayer.teleToLocation(MapRegionTable.TeleportWhereType.Town);
    					onlinePlayer.setIsIn7sDungeon(false);
    					onlinePlayer.sendMessage("You have been teleported to the nearest town due to the beginning of the Seal Validation period.");
    				}
    			}
    			else
    			{
    				if (!onlinePlayer.isGM() && onlinePlayer.isIn7sDungeon() && !currPlayer.getString("cabal").equals(""))
    				{
    					onlinePlayer.teleToLocation(MapRegionTable.TeleportWhereType.Town);
    					onlinePlayer.setIsIn7sDungeon(false);
    					onlinePlayer.sendMessage("You have been teleported to the nearest town because you have not signed for any cabal.");
    				}
    			}
    		}
    	}
    }

	/**
     * The primary controller of period change of the Seven Signs system.
     * This runs all related tasks depending on the period that is about to begin.
     *
	 * @author Tempy
	 */
	protected class SevenSignsPeriodChange implements Runnable
	{
        public void run()
        {
            /*
             * Remember the period check here refers to the period just ENDED!
             */
            final int periodEnded = getCurrentPeriod();
            _activePeriod++;

	        switch (periodEnded)
	        {
	            case PERIOD_COMP_RECRUITING: // Initialization

                    // Start the Festival of Darkness cycle.
                    SevenSignsFestival.getInstance().startFestivalManager();

                    // Send message that Competition has begun.
                    sendMessageToAll(SystemMessageId.QUEST_EVENT_PERIOD_BEGUN);
	                break;
	            case PERIOD_COMPETITION: // Results Calculation

	                // Send message that Competition has ended.
	                sendMessageToAll(SystemMessageId.QUEST_EVENT_PERIOD_ENDED);

	                int compWinner = getCabalHighestScore();

	                // Schedule a stop of the festival engine.
	                SevenSignsFestival.getInstance().getFestivalManagerSchedule().cancel(false);

                    calcNewSealOwners();

                    switch (compWinner)
                    {
                    	case CABAL_DAWN:
                    		sendMessageToAll(SystemMessageId.DAWN_WON);
                    		break;
                    	case CABAL_DUSK:
                    		sendMessageToAll(SystemMessageId.DUSK_WON);
                    		break;
                    }

                    _previousWinner = compWinner;
	                break;
	            case PERIOD_COMP_RESULTS: // Seal Validation

                    // Perform initial Seal Validation set up.
                    initializeSeals();
                    //Buff/Debuff members of the event when Seal of Strife captured. 
                    GiveCPMult(getSealOwner(SEAL_STRIFE));
                    // Send message that Seal Validation has begun.
                    sendMessageToAll(SystemMessageId.SEAL_VALIDATION_PERIOD_BEGUN);

                    _log.info("SevenSigns: The " + getCabalName(_previousWinner) + " have won the competition with " + getCurrentScore(_previousWinner) + " points!");
	                break;
	            case PERIOD_SEAL_VALIDATION: // Reset for New Cycle

	            	SevenSignsFestival.getInstance().rewardHighestRanked(); // reward highest ranking members from cycle

	                // Ensure a cycle restart when this period ends.
                    _activePeriod = PERIOD_COMP_RECRUITING;

	                // Send message that Seal Validation has ended.
	                sendMessageToAll(SystemMessageId.SEAL_VALIDATION_PERIOD_ENDED);
	                //Clear Seal of Strife influence.
	                RemoveCPMult();
	                // Reset all data
	                resetPlayerData();
	                resetSeals();

                    // Reset all Festival-related data and remove any unused blood offerings.
                    // NOTE: A full update of Festival data in the database is also performed.
                    SevenSignsFestival.getInstance().resetFestivalData(false);

	                _dawnStoneScore = 0;
	                _duskStoneScore = 0;

	                _dawnFestivalScore = 0;
	                _duskFestivalScore = 0;

	                _currentCycle++;
	                break;
	        }

	        // [L2J_JP ADD START]
	        removeNecropolisAndCatacomb();
	        spawnNecropolisAndCatacomb();
	        // [L2J_JP ADD END]

	        // Make sure all Seven Signs data is saved for future use.
	        saveSevenSignsData(null, true);

            teleLosingCabalFromDungeons(getCabalShortName(getCabalHighestScore()));

            SSQInfo ss = new SSQInfo();

            Broadcast.toAllOnlinePlayers(ss);
            spawnSevenSignsNPC();

            _log.info("SevenSigns: The " + getCurrentPeriodName() + " period has begun!");

            setCalendarForNextPeriodChange();
            
            // make sure that all the scheduled siege dates are in the Seal Validation period
            List<Castle> castles = CastleManager.getInstance().getCastles();
            for (Castle castle : castles)
            {
            	castle.getSiege().correctSiegeDateTime();
            }

			//XXX:[JOJO]
	        long milliToChange = getMilliToPeriodChange();		
			_log.info("SevenSigns: Next period begins at " + StrCalender());
			_log.info("SevenSigns: Next period begins in " + StrMillTime(milliToChange));
	        SevenSignsPeriodChange sspc = new SevenSignsPeriodChange();
 	        ThreadPoolManager.getInstance().scheduleGeneral(sspc, milliToChange);
			//XXX:[JOJO]
	    }
 	}
	public boolean CheckIsDawnPostingTicket(int itemId){
		//TODO I think it should be some kind of a list in the datapack for compare;
		if (itemId > 6114 && itemId < 6175) return true;
		if (itemId > 6801 && itemId < 6812) return true;
		if (itemId > 7997 && itemId < 8008) return true;
		if (itemId > 7940 && itemId < 7951) return true;
		if (itemId > 6294 && itemId < 6307) return true;
		if (itemId > 6831 && itemId < 6834) return true;
		if (itemId > 8027 && itemId < 8030) return true;
		if (itemId > 7970 && itemId < 7973) return true;
		return false;	
	}
	public boolean CheckIsRookiePostingTicket(int itemId){
		//TODO I think it should be some kind of a list in the datapack for compare;
		if (itemId > 6174 && itemId < 6295) return true;
		if (itemId > 6811 && itemId < 6832) return true;
		if (itemId > 7950 && itemId < 7971) return true;
		if (itemId > 8007 && itemId < 8028) return true;
		return false;	
		}
	public void GiveCPMult(int StrifeOwner){
		//Gives "Victor of War" passive skill to all online characters with Cabal, which controls Seal of Strife 
	for (L2PcInstance character : L2World.getInstance().getAllPlayers().values()){
    	if (getPlayerCabal(character) != SevenSigns.CABAL_NULL)
    		if (getPlayerCabal(character) == StrifeOwner)
    			character.addSkill(SkillTable.getInstance().getInfo(5074,1));
    		else
    			//Gives "The Vanquished of War" passive skill to all online characters with Cabal, which does not control Seal of Strife
    			character.addSkill(SkillTable.getInstance().getInfo(5075,1));		
		}			
	}
	public void RemoveCPMult(){
		for (L2PcInstance character : L2World.getInstance().getAllPlayers().values()){
			//Remove SevenSigns' buffs/debuffs.
        	character.removeSkill(SkillTable.getInstance().getInfo(5074,1));
        	character.removeSkill(SkillTable.getInstance().getInfo(5075,1));   			
		}
	}
	public boolean CheckSummonConditions(L2PcInstance activeChar){
	if (activeChar == null) return true;
	//Golems cannot be summoned by Dusk when the Seal of Strife is controlled by the Dawn
	if (isSealValidationPeriod())
		if (getSealOwner(SEAL_STRIFE) == CABAL_DAWN)
			if (getPlayerCabal(activeChar) == CABAL_DUSK)
			{
				activeChar.sendMessage("You cannot summon Siege Golem or Cannon while Seal of Strife posessed by Lords of Dawn.");
				return true;	
			}
					
	return false;	
	}

	// [L2J_JP ADD START]
	/**
     * load Necropolis And Catacomb mobs
     * Which at the dawn or dusk the winner is judged,
     * and the monster is loaded into necropolis and the catacomb. 
     */
    public void spawnNecropolisAndCatacomb()
    {
    	if(_initializedNecCatSpawns){
    		return;
    	}

    	int _Winner;
		switch (_activePeriod) {
		case PERIOD_COMP_RECRUITING:
			_Winner = CABAL_NULL;
			break;
		case PERIOD_COMPETITION:
			_Winner = CABAL_NULL;
			break;
		case PERIOD_COMP_RESULTS:
			_Winner = CABAL_NULL;
			break;
		case PERIOD_SEAL_VALIDATION:
			if (getCabalHighestScore() == CABAL_DAWN)
			{
				_Winner = CABAL_DAWN;
			}
			else
			{
				_Winner = CABAL_DUSK;
			}
			break;
		default:
			_Winner = CABAL_NULL;
			break;				
		}

        java.sql.Connection con = null;
		_signsNecCatMobInsts.clear();

        try
        {
            con = L2DatabaseFactory.getInstance().getConnection();
            PreparedStatement statement = con.prepareStatement("SELECT id, count, npc_templateid, locx, locy, locz, heading, respawn_delay, loc_id FROM seven_signs_spawnlist Where winner = ? ORDER BY id");
            statement.setInt(1, _Winner);
            ResultSet rset = statement.executeQuery();

            L2Spawn spawnDat;
            L2NpcTemplate template1;

            while (rset.next())
            {
                template1 = NpcTable.getInstance().getTemplate(rset.getInt("npc_templateid"));
                if (template1 != null)
                {
                	spawnDat = new L2Spawn(template1);
                	spawnDat.setId(rset.getInt("id"));
                	spawnDat.setAmount(rset.getInt("count"));
                	spawnDat.setLocx(rset.getInt("locx"));
                	spawnDat.setLocy(rset.getInt("locy"));
                	spawnDat.setLocz(rset.getInt("locz"));
                	spawnDat.setHeading(rset.getInt("heading"));
                	spawnDat.setRespawnDelay(rset.getInt("respawn_delay"));
                	int loc_id = rset.getInt("loc_id");
                	spawnDat.setLocation(loc_id);
                	
                    SpawnTable.getInstance().addNewSpawn(spawnDat, false);
                    _signsNecCatMobInsts.add(spawnDat.doSpawn());
                    spawnDat.startRespawn();
                }
                else {
                    _log.warning("SpawnNecropolisAndCatacomb: Data missing in NPC table for ID: " + rset.getInt("npc_templateid") + ".");
                }
            }

            rset.close();
            statement.close();
            _log.info("spawnNecropolisAndCatacomb: Loaded " + _signsNecCatMobInsts.size() + " Npc Spawn Locations.");
        }
        catch (Exception e)
        {
            // problem with initializing spawn, go to next one
            _log.warning("SpawnNecropolisAndCatacomb: Spawn could not be initialized: " + e);
        }
        finally
        {
            try { con.close(); } catch (Exception e) {}
        }
        SetNecCatSpawnd(true);

    }

    /**
	 * remove necropolis and the catacomb mob  
	 */
	public void removeNecropolisAndCatacomb()
	{
		for (L2NpcInstance npcInst : _signsNecCatMobInsts)
		{
			try
            {
                npcInst.getSpawn().stopRespawn();
                npcInst.deleteMe();
            }
            catch(Exception e)
            {
                _log.warning(e.getMessage());
            }
		}
		_signsNecCatMobInsts.clear();
        SetNecCatSpawnd(false);
	}
    
    protected void SetNecCatSpawnd(boolean stat)
    {
        this._initializedNecCatSpawns = stat;
    }
	// [L2J_JP ADD END]
}
