﻿module SDL_smpeg;
/*
  SMPEG - SDL MPEG Player Library
  Copyright (C) 1999  Loki Entertainment Software

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free
  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* This is the C interface to the SMPEG library */

// convert to D by shinichiro.h

private import SDL;
private import SDL_smpeg_filter;

extern (C) {

	const int SMPEG_MAJOR_VERSION = 0;
	const int SMPEG_MINOR_VERSION = 4;
	const int SMPEG_PATCHLEVEL = 3;

	struct SMPEG_version {
		Uint8 major;
		Uint8 minor;
		Uint8 patch;
	}

/* This macro can be used to fill a version structure with the compile-time
 * version of the SDL library.
 */
	void SMPEG_VERSION(out SMPEG_version X) {
		X.major = SMPEG_MAJOR_VERSION;
		X.minor = SMPEG_MINOR_VERSION;
		X.patch = SMPEG_PATCHLEVEL;
	}

/* This is the actual SMPEG object */
	struct SMPEG { int dummy; }
	/*
		空のオブジェクトを前方宣言すると落ちる(DMD0.79)
		ので、ダミーの変数いれときますです。
	*/

/* Used to get information about the SMPEG object */
	struct SMPEG_Info {
		int has_audio;
		int has_video;
		int width;
		int height;
		int current_frame;
		double current_fps;
		char audio_string[80];
		int  audio_current_frame;
		Uint32 current_offset;
		Uint32 total_size;
		double current_time;
		double total_time;
	}

/* Possible MPEG status codes */
	enum {
		SMPEG_ERROR = -1,
		SMPEG_STOPPED,
		SMPEG_PLAYING
	}
	alias int SMPEGstatus;


/* Matches the declaration of SDL_UpdateRect() */
	typedef void(*SMPEG_DisplayCallback)(SDL_Surface* dst, int x, int y,
										 uint w, uint h);
//	typedef void delegate (SDL_Surface* dst, int x, int y, uint w, uint h) SMPEG_DisplayCallback;

/* Create a new SMPEG object from an MPEG file.
   On return, if 'info' is not NULL, it will be filled with information 
   about the MPEG object.
   This function returns a new SMPEG object.  Use SMPEG_error() to find out
   whether or not there was a problem building the MPEG stream.
   The sdl_audio parameter indicates if SMPEG should initialize the SDL audio
   subsystem. If not, you will have to use the SMPEG_playaudio() function below
   to extract the decoded data.
*/
	SMPEG* SMPEG_new(char *file, SMPEG_Info* info, int sdl_audio);

/* The same as above for a file descriptor */
	SMPEG* SMPEG_new_descr(int file, SMPEG_Info* info, int sdl_audio);

/*
  The same as above but for a raw chunk of data.  SMPEG makes a copy of the
  data, so the application is free to delete after a successful call to this
  function.
*/
	SMPEG* SMPEG_new_data(void *data, int size, SMPEG_Info* info, int sdl_audio);

/* The same for a generic SDL_RWops structure. */
	SMPEG* SMPEG_new_rwops(SDL_RWops *src, SMPEG_Info* info, int sdl_audio);

/* Get current information about an SMPEG object */
	void SMPEG_getinfo( SMPEG* mpeg, SMPEG_Info* info );

/* Enable or disable audio playback in MPEG stream */
	void SMPEG_enableaudio( SMPEG* mpeg, int enable );

/* Enable or disable video playback in MPEG stream */
	void SMPEG_enablevideo( SMPEG* mpeg, int enable );

/* Delete an SMPEG object */
	void SMPEG_delete( SMPEG* mpeg );

/* Get the current status of an SMPEG object */
	SMPEGstatus SMPEG_status( SMPEG* mpeg );

/* Set the audio volume of an MPEG stream, in the range 0-100 */
	void SMPEG_setvolume( SMPEG* mpeg, int volume );

/* Set the destination surface for MPEG video playback
   'surfLock' is a mutex used to synchronize access to 'dst', and can be NULL.
   'callback' is a function called when an area of 'dst' needs to be updated.
   If 'callback' is NULL, the default function (SDL_UpdateRect) will be used.
*/
	static void SMPEG_setdisplay(SMPEG* mpeg, SDL_Surface* dst, SDL_mutex* surfLock,
						  SMPEG_DisplayCallback callback);

/* Set or clear looping play on an SMPEG object */
	void SMPEG_loop( SMPEG* mpeg, int repeat );

/* Scale pixel display on an SMPEG object */
	void SMPEG_scaleXY( SMPEG* mpeg, int width, int height );
	void SMPEG_scale( SMPEG* mpeg, int scale );
/* */
	void SMPEG_double(SMPEG* mpeg, bit on) {
		SMPEG_scale(mpeg, (on) ? 2 : 1);
	}

/* Move the video display area within the destination surface */
	void SMPEG_move( SMPEG* mpeg, int x, int y );

/* Set the region of the video to be shown */
	void SMPEG_setdisplayregion(SMPEG* mpeg, int x, int y, int w, int h);

/* Play an SMPEG object */
	void SMPEG_play( SMPEG* mpeg );

/* Pause/Resume playback of an SMPEG object */
	void SMPEG_pause( SMPEG* mpeg );

/* Stop playback of an SMPEG object */
	void SMPEG_stop( SMPEG* mpeg );

/* Rewind the play position of an SMPEG object to the beginning of the MPEG */
	void SMPEG_rewind( SMPEG* mpeg );

/* Seek 'bytes' bytes in the MPEG stream */
	void SMPEG_seek( SMPEG* mpeg, int bytes);

/* Skip 'seconds' seconds in the MPEG stream */
	void SMPEG_skip( SMPEG* mpeg, float seconds );

/* Render a particular frame in the MPEG video
   API CHANGE: This function no longer takes a target surface and position.
   Use SMPEG_setdisplay() and SMPEG_move() to set this information.
*/
	void SMPEG_renderFrame( SMPEG* mpeg, int framenum );

/* Render the last frame of an MPEG video */
	void SMPEG_renderFinal( SMPEG* mpeg, SDL_Surface* dst, int x, int y );

/* Set video filter */
	SMPEG_Filter * SMPEG_filter( SMPEG* mpeg, SMPEG_Filter * filter );

/* Return NULL if there is no error in the MPEG stream, or an error message
   if there was a fatal error in the MPEG stream for the SMPEG object.
*/
	char *SMPEG_error( SMPEG* mpeg );

/* Exported callback function for audio playback.
   The function takes a buffer and the amount of data to fill, and returns
   the amount of data in bytes that was actually written.  This will be the
   amount requested unless the MPEG audio has finished.
*/
	int SMPEG_playAudio( SMPEG *mpeg, Uint8 *stream, int len );

/* Wrapper for SMPEG_playAudio() that can be passed to SDL and SDL_mixer
 */
	void SMPEG_playAudioSDL( void *mpeg, Uint8 *stream, int len );

/* Get the best SDL audio spec for the audio stream */
	int SMPEG_wantedSpec( SMPEG *mpeg, SDL_AudioSpec *wanted );

/* Inform SMPEG of the actual SDL audio spec used for sound playback */
	void SMPEG_actualSpec( SMPEG *mpeg, SDL_AudioSpec *spec );

}
