/**
 * @file core.h
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * Core module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#ifndef CORE_H
#define CORE_H

#define LABEL_COUNT     (100)
#define LABEL_LENGTH    (10)
#define FOR_MAX_DEPTH   (25)
#define GOSUB_MAX_DEPTH (25)
#define PROGRAM_SIZE    (10000)

typedef enum {
  TKNONE    = 0,
  DELIMITER = 1,
  VARIABLE  = 2,
  NUMBER    = 3,
  COMMAND   = 4,
  STRING    = 5,
  QUOTE     = 6,
} token_type_t;

typedef enum {
  STNONE    = 0,
  PRINT     = 1,
  INPUT     = 2,
  IF        = 3,
  THEN      = 4,
  FOR       = 5,
  NEXT      = 6,
  TO        = 7,
  GOTO      = 8,
  EOL       = 9,
  FINISHED  = 10,
  GOSUB     = 11,
  RETURN    = 12,
  REM       = 13,
  READ      = 14,
  WRITE     = 15,
  END       = 16,
} statement_t;

typedef struct {
  char name[LABEL_LENGTH];
  char *p;  /* points to place to go in source file*/
} label_t;

typedef struct {
  int var; /* counter variable */
  int target;  /* target value */
  char *loc;
} for_stack_t;

typedef struct {
  char *program;
  char *pointer;    /* holds expression to be analyzed */
  label_t label[LABEL_COUNT];
} location_t;

typedef struct {
  char text[80];
  token_type_t type;
  statement_t statement;
} token_t;

typedef struct {
  location_t location;
  token_t token;

  int variables[26];
  for_stack_t fstack[FOR_MAX_DEPTH]; /* stack for FOR/NEXT loop */
  char *gstack[GOSUB_MAX_DEPTH]; /* stack for GOSUB */
  int ftos;  /* index to top of FOR stack */
  int gtos;  /* index to top of GOSUB stack */
} core_t;

int core_setup(core_t *core, char *program);
int core_find_eol(core_t *core);
int core_putback(core_t *core);
char *core_find_label(core_t *core, char *s);
token_type_t core_get_token(core_t *core);

#endif

