/**
 * @file ntbasic.c
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * NT-Basic interface module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntlibc.h"
#include "error.h"
#include "ntbasic.h"
#include "statement.h"
#include "expression.h"
#include "core.h"
#include "hal.h"

/* Assign a variable a value. */
static void assignment(core_t *core)
{
  int var, value;

  /* get the variable name */
  core_get_token(core);
  if (!ntlibc_isalpha(*core->token.text)) {
    error_message(core, NotAVariable);
    return;
  }

  var = ntlibc_toupper(*core->token.text) - 'A';

  /* get the equals sign */
  core_get_token(core);
  if (*core->token.text != '=') {
    error_message(core, EqualsSignExpected);
    return;
  }

  /* get the value to assign to var */
  expression_entry(core, &value);

  /* assign the value */
  core->variables[var] = value;
}

int ntbasic_setup(core_t *core, char *program)
{
  hal_init();
  core_setup(core, program);
  return 0;
}

int ntbasic_execute(core_t *core)
{
  core->token.type = core_get_token(core);
  /* check for assignment statement */
  if(core->token.type == VARIABLE) {
    core_putback(core); /* return the var to the input stream */
    assignment(core); /* must be assignment statement */
  } else {
    /* is command */
    switch (core->token.statement) {
      case PRINT:
        statement_print(core);
        break;
      case GOTO:
        statement_goto(core);
        break;
      case IF:
        statement_if(core);
        break;
      case FOR:
        statement_for(core);
        break;
      case NEXT:
        statement_next(core);
        break;
      case INPUT:
        statement_input(core);
        break;
      case GOSUB:
        statement_gosub(core);
        break;
      case RETURN:
        statement_return(core);
        break;
      case REM:
        statement_rem(core);
        break;
      case READ:
        statement_read(core);
        break;
      case WRITE:
        statement_write(core);
        break;
      case END:
        return 0;
      case STNONE:
        // Nothing to do.
        break;
      case THEN:
        // Nothing to do.
        break;
      case TO:
        // Nothing to do.
        break;
      case EOL:
        // Nothing to do.
        break;
      case FINISHED:
        // Nothing to do.
        break;
    }
  }
  return (core->token.statement != FINISHED);
}

