/**
 * @file error.c
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * Error message module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "error.h"
#include "hal.h"

static const char *e[]= {
  "syntax error\n",
  "unbalanced parentheses\n",
  "no expression present\n",
  "equals sign expected\n",
  "not a variable\n",
  "Label table full\n",
  "duplicate label\n",
  "undefined label\n",
  "THEN expected\n",
  "TO expected\n",
  "too many nested FOR loops\n",
  "NEXT without FOR\n",
  "too many nested GOSUBs\n",
  "RETURN without GOSUB\n"
};

static void error_puts(char *s)
{
  char *p = s;
  while (*p) {
    hal_putc(*p);
    p++;
  }
}

static int print_integer(int number)
{
    const int base = 10;
    int work = number;
    int bnum = 1;
    int i;
    int length = 10;
    int active = 0;

    if (number == 0) {
      hal_putc('0');
      return 0;
    }

    for (i = 0; i < length - 1; i++) {
        bnum = bnum * base;
    }

    for (i = 0; i < length; i++) {
        int r = work / bnum;
        if (r != 0) {
          if (active == 0) {
            active = 1;
          }
        }
        if (active) {
          hal_putc("0123456789"[r % 10]);
        }
        work = work - (r * bnum);
        bnum = bnum / base;
    }

    return 0;
}

void error_message(core_t *core, ntbasic_error_t error)
{
  char *p = core->location.program;
  int line = 1;
  error_puts("Error: ");
  error_puts((char *)e[(int)error]);
  while (*p) {
    if (*p == '\n') {
      line++;
    }
    if (p == core->location.pointer) {
      break;
    }
    p++;
  }
  error_puts("Line ");
  print_integer(line);
  hal_halt();
}

