/**
 * @file core.c
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * Core module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntlibc.h"
#include "core.h"
#include "error.h"

/**
 * @brief Command keyword lookup table.
 */
typedef struct {
  char command[20];
  statement_t statement;
} command_t;

static const command_t table[] = {
  /* Commands must be entered lowercase */
  {"print", PRINT},
  {"input", INPUT},
  {"if", IF},
  {"then", THEN},
  {"goto", GOTO},
  {"for", FOR},
  {"next", NEXT},
  {"to", TO},
  {"gosub", GOSUB},
  {"return", RETURN},
  {"rem", REM},
  {"read", READ},
  {"write", WRITE},
  {"end", END},
  {"", END}, /* mark end of table */
};

/**
 * @brief Check the white space.
 * @param c Input character.
 *
 * @retval 1 White space.
 * @retval 0 Non white space.
 */
static int iswhite(char c)
{
    if ((c == ' ') || (c == '\t')) {
      return 1;
    }
    return 0;
}

/**
 * @brief Check the delimiter.
 * @param c Input character.
 *
 * @retval 1 Delimiter.
 * @retval 0 Non delimiter.
 */
static int isdelim(char c)
{
    if (ntlibc_strchr(" ;,+-<>/*%^=()", c) || (c == 9) || (c == '\r') || (c == '\n') || (c == 0)) {
      return 1;
    }
    return 0;
}

/* Look up a a token's internal representation in the
   token table.
   */
static int look_up(char *s)
{
  register int i;
  char *p;

  /* convert to lowercase */
  p = s;
  while (*p) {
    *p = ntlibc_tolower(*p);
    p++;
  }

  /* see if token is in table */
  for (i = 0; *table[i].command; i++) {
    if (!ntlibc_strcmp(table[i].command, s)) {
      return table[i].statement;
    }
  }
  return 0; /* unknown command */
}

/**
 * @brief Return index of next free position in label array.
 *
 * @param core core object.
 * @param s Label name.
 *
 * @retval 0>= Next free position.
 * @retval -1 The array is full.
 * @retval -2 Duplicate label is found.
 */
static int get_label_position(core_t *core, char *s)
{
  register int i;

  for (i = 0; i < LABEL_COUNT; ++i) {
    if (core->location.label[i].name[0] == 0) {
      return i;
    }
    if (!ntlibc_strcmp(core->location.label[i].name, s)) {
      /*
       * Duplicate label is found.
       */
      return -2;
    }
  }

  /*
   * The array is full.
   */
  return -1;
}

/**
 * @brief Scan all labels in the core list.
 */
static void scan_labels(core_t *core)
{
  register int t;
  int addr;
  char *prog_top;

  /*
   * Initialize the array that holds the labels.
   * By convention, a null label name indicates that
   * array position is unused.
   */
  for (t = 0; t < LABEL_COUNT; ++t) {
    core->location.label[t].name[0] = '\0';
  }

  /*
   * save pointer to top of core
   */
  prog_top = core->location.pointer;

  /* if the first token in the line is a label */
  core_get_token(core);
  if (core->token.type == NUMBER) {
    ntlibc_strcpy(core->location.label[0].name, core->token.text);
    core->location.label[0].p = core->location.pointer;
  }

  core_find_eol(core);
  do {
    core_get_token(core);
    if (core->token.type == NUMBER) {
      addr = get_label_position(core, core->token.text);
      if ((addr == -1) || (addr == -2)) {
        if (addr == -1) {
          error_message(core, LabelTableFull);
        } else {
          error_message(core, DuplicateLabel);
        }
      }
      ntlibc_strcpy(core->location.label[addr].name, core->token.text);
      core->location.label[addr].p = core->location.pointer;  /* current point in core */
    }
    /* if not on a blank line, find next line */
    if (core->token.statement != EOL) {
      core_find_eol(core);
    }
  } while (core->token.statement != FINISHED);
  core->location.pointer = prog_top;    /* restore to original */
}

static int check_size(char *core)
{
  int cnt = 0;
  char *p = core;
  while (*p) {
    cnt++;
    if (PROGRAM_SIZE < cnt) {
      return 1;
    }
  }
  return 0;
}

int core_setup(core_t *core, char *program)
{
  int i;
  if (!check_size(program)) {
    return -1;
  }
  for (i = 0; i < (int)sizeof(core->variables); i++) {
    core->variables[i] = 0;
  }
  core->location.program = program;
  core->location.pointer = program;
  core->ftos = 0;       /* initialize the FOR stack index */
  core->gtos = 0;       /* initialize the GOSUB stack index */
  scan_labels(core);    /* find the labels in the core */
  return 0;
}

/* Find the start of the next line. */
int core_find_eol(core_t *core)
{
  int cnt = 0;
  while ((*core->location.pointer != '\n') && (*core->location.pointer != '\0')) {
    ++core->location.pointer;
    cnt++;
  }
  if (*core->location.pointer) {
    core->location.pointer++;
    cnt++;
  }
  return cnt;
}

/* Return a token to input stream. */
int core_putback(core_t *core)
{
  int cnt = 0;
  char *t;

  t = core->token.text;
  for (; *t; t++) {
    core->location.pointer--;
    cnt++;
  }
  return cnt;
}

/**
 * @brief Find location of given label.
 *
 * @retval NULL Label is not found.
 * @retval !NULL A pointer to the position of the label.
 */
char *core_find_label(core_t *core, char *s)
{
  register int t;

  for (t = 0; t < LABEL_COUNT; ++t) {
    if (!ntlibc_strcmp(core->location.label[t].name, s)) {
      return core->location.label[t].p;
    }
  }

  return '\0';
}

/* Get a token. */
token_type_t core_get_token(core_t *core)
{
  register char *temp;

  core->token.type = TKNONE;
  core->token.statement = STNONE;
  temp = core->token.text;

  if (*(core->location.pointer) == '\0') {
    /*
     * end of file
     */
    *(core->token.text) = 0;
    core->token.statement = FINISHED;
    core->token.type = DELIMITER;
    return DELIMITER;
  }

  while (iswhite(*(core->location.pointer))) {
    /*
     * skip over white space
     */
    ++core->location.pointer;
  }

  if (*(core->location.pointer) == '\n') {
    /* lf */
    ++core->location.pointer;
    core->token.statement = EOL;
    *(core->token.text) = '\n';
    core->token.text[1] = 0;
    core->token.type = DELIMITER;
    return DELIMITER;
  }

  if (*(core->location.pointer) == '\r') {
    /* crlf */
    ++core->location.pointer;
    ++core->location.pointer;
    core->token.statement = EOL;
    *(core->token.text) = '\r';
    core->token.text[1] = '\n';
    core->token.text[2] = 0;
    core->token.type = DELIMITER;
    return DELIMITER;
  }

  if (ntlibc_iscntrl(*(core->location.pointer))) {
    /*
     * end of file
     */
    *(core->token.text) = 0;
    core->token.statement = FINISHED;
    core->token.type = DELIMITER;
    return DELIMITER;
  }

  if (ntlibc_strchr("+-*^/%=;(),><", *(core->location.pointer))) {
    /*
     * delimiter
     */
    *temp = *(core->location.pointer);
    core->location.pointer++; /* advance to next position */
    temp++;
    *temp = 0;
    core->token.type = DELIMITER;
    return DELIMITER;
  }

  if (*(core->location.pointer) == '"') {
    /*
     * quoted string
     */
    core->location.pointer++;
    while ((*(core->location.pointer) != '"') && (*(core->location.pointer) != '\r') && (*(core->location.pointer) != '\n')) {
      *temp++ = *core->location.pointer++;
    }
    if ((*(core->location.pointer) == '\r') || (*(core->location.pointer) == '\n')) {
      error_message(core, UnbalancedParentheses);
    }
    core->location.pointer++;
    *temp=0;
    core->token.type = QUOTE;
    return QUOTE;
  }

  if (ntlibc_isdigit(*core->location.pointer)) {
    /*
     * number
     */
    while (!isdelim(*core->location.pointer)) {
      *temp++ = *core->location.pointer++;
    }
    *temp = '\0';
    core->token.type = NUMBER;
    return NUMBER;
  }

  if (ntlibc_isalpha(*core->location.pointer)) {
    /*
     * var or command
     */
    while (!isdelim(*core->location.pointer)) {
      *temp++ = *core->location.pointer++;
    }
    core->token.type = STRING;
  }

  *temp = '\0';

  /*
   * see if a string is a command or a variable
   */
  if (core->token.type == STRING) {
    /*
     * convert to internal rep
     */
    core->token.statement = look_up(core->token.text);
    if (!core->token.statement) {
      core->token.type = VARIABLE;
    } else {
      /*
       * is a command
       */
      core->token.type = COMMAND;
    }
  }
  return core->token.type;
}

