/**
 * @file expression.c
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * Core expression module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntlibc.h"
#include "expression.h"
#include "error.h"
#include "core.h"

static void expression_level2(core_t *core, int *result);
static void expression_level3(core_t *core, int *result);
static void expression_level4(core_t *core, int *result);
static void expression_level5(core_t *core, int *result);
static void expression_level6(core_t *core, int *result);

/**
 * @brief Find the value of a variable.
 *
 * @param core The core object.
 * @param s The text string.
 *
 * @return The value of a variable.
 */
static int find_var(core_t *core, char *s)
{
  if (!ntlibc_isalpha(*s)) {
    /*
     * Not a variable
     */
    error_message(core, NotAVariable);
    return 0;
  }
  return core->variables[ntlibc_toupper(*core->token.text) - 'A'];
}

/**
 * @brief Find value of number or variable.
 *
 * @param core The core object.
 * @param result The result.
 */
static void primitive(core_t *core, int *result)
{
  switch (core->token.type) {
    case VARIABLE:
      *result = find_var(core, core->token.text);
      core_get_token(core);
      return;
    case NUMBER:
      *result = ntlibc_atoi(core->token.text);
      core_get_token(core);
      return;
    default:
      error_message(core, SyntaxError);
  }
}

/**
 * @brief Perform the specified arithmetic.
 *
 * @param o The operator.
 * @param r 1st value. (Result)
 * @param h 2nd value. (Hold)
 */
static void arith(char o, int *r, int *h)
{
  register int t, ex;

  switch (o) {
    case '-':
      *r = *r - *h;
      break;
    case '+':
      *r = *r + *h;
      break;
    case '*':
      *r = *r * *h;
      break;
    case '/':
      *r = (*r) / (*h);
      break;
    case '%':
      t = (*r) / (*h);
      *r = *r - (t * (*h));
      break;
    case '^':
      ex = *r;
      if (*h == 0) {
        *r = 1;
        break;
      }
      for (t = *h - 1; t > 0; --t) {
        *r = (*r) * ex;
      }
      break;
  }
}

/**
 * @brief Reverse the sign.
 *
 * @param o The operator.
 * @param r Result.
 */
static void unary(char o, int *r)
{
  if (o == '-') {
    *r = -(*r);
  }
}

/**
 * @brief Entry point into parser.
 *
 * @param core The core object.
 * @param result Result.
 */
void expression_entry(core_t *core, int *result)
{
  core_get_token(core);
  if (!*core->token.text) {
    error_message(core, NoExpressionPresent);
    return;
  }
  expression_level2(core, result);
  core_putback(core); /* return last token read to input stream */
}

/**
 * @brief Add or subtract two terms.
 *
 * @param core core object.
 * @param result The result.
 */
static void expression_level2(core_t *core, int *result)
{
  register char op;
  int hold;

  expression_level3(core, result);
  while (((op = *core->token.text) == '+') || (op == '-')) {
    core_get_token(core);
    expression_level3(core, &hold);
    arith(op, result, &hold);
  }
}

/**
 * @brief Multiply or divide two factors.
 *
 * @param core core object.
 * @param result The result.
 */
static void expression_level3(core_t *core, int *result)
{
  register char op;
  int hold;

  expression_level4(core, result);
  while (((op = *core->token.text) == '*') || (op == '/') || (op == '%')) {
    core_get_token(core);
    expression_level4(core, &hold);
    arith(op, result, &hold);
  }
}

/**
 * @brief Expression integer exponent.
 *
 * @param core core object.
 * @param result The result.
 */
static void expression_level4(core_t *core, int *result)
{
  int hold;

  expression_level5(core, result);
  if (*core->token.text == '^') {
    core_get_token(core);
    expression_level4(core, &hold);
    arith('^', result, &hold);
  }
}

/**
 * @brief Is a unary + or -.
 *
 * @param core core object.
 * @param result The result.
 */
static void expression_level5(core_t *core, int *result)
{
  register char op;

  op = 0;
  if ((core->token.type == DELIMITER) && ((*core->token.text == '+') || (*core->token.text == '-'))) {
    op = *core->token.text;
    core_get_token(core);
  }
  expression_level6(core, result);
  if (op) {
    unary(op, result);
  }
}

/**
 * @brief Expression parenthesized expression.
 *
 * @param core core object.
 * @param result The result.
 */
static void expression_level6(core_t *core, int *result)
{
  if ((*(core->token.text) == '(') && (core->token.type == DELIMITER)) {
    core_get_token(core);
    expression_level2(core, result);
    if (*(core->token.text) != ')') {
      error_message(core, UnbalancedParentheses);
    }
    core_get_token(core);
  } else {
    primitive(core, result);
  }
}

