﻿#ifndef KFX_KFILE_INCLUDED
#define KFX_KFILE_INCLUDED
#include<string>
namespace KFX{


enum FileEvent{
	FE_ERROR,
	FE_OPENCOMPLETED,
	FE_READCOMPLETED,
	FE_WRITECOMPLRETED,
};

///ファイル関係のなんかしらのイベントを受け取る基底クラス
class IFileListener{
	public:
		virtual void Notify(FileEvent event,int error);
};

enum TaskResult{
	TR_SUCCEEDED,
	TR_FAILED
};

///今は特になにもしません。
///いずれ実装予定のワーカースレッドが、ITaskのExecuteメソッドを呼び出す
class ITask{
public:
	virtual TaskResult Execute()=0;
};

///@note リソースリーダーインターフェイス
///リソースを読み込むインターフェイスを提供(生ファイルだろうとZipアーカイブだろうと何だろうと実装側が面倒見る)
///ファイルオープン関数は、このオブジェクトを返す。
///返した時点で中身が入っているかどうかは場合による
///ロードには、非同期ロードと同期ロードがあるり、ブロックロードはロード完了まで待つが、
///非同期ロードでロードすると、ファイルオブジェクトは即時で帰ってくる。このとき、中身が
///ロード完了しているかどうかは保障されない。このため使用可能フラグIsAvailable関数を実装する必要がある
///非同期処理の場合、読み書きが完了するまでは次のリクエストは無視するように作らなければならない
class IResourceReader : public ITask{
	public:

		///ファイルサイズを返す
		virtual long long FileSize()const=0;

		///ファイルの読み込みリクエストを行う
		///@param address 読み込んだファイルの内容が格納されるアドレス(呼び出し側が責任を持つ)
		///@param reqSize 読み取りたいバイトサイズ
		///@param outSize 実際に読み取ったサイズ
		///@retval true エラーなしに読み込み成功(完了したとは限らない)
		///@retval false 何かしらのエラーが起きた可能性がある(エラー処理は実装側でやるなり、呼び出し側でやるなり)
		virtual bool Request(void* address,long long reqSize,size_t& actualSize)=0;

		///Implement ITask
		virtual TaskResult Execute()=0;

		///このファイルオブジェクトが利用可能になっているかどうか
		///@retval true 利用可能
		///@retval false 利用不可能
		///@note オープン直後でオープン処理が終わっていなければFalseだし、
		///直前の読み書き処理が終わっていなければFalseになる
		virtual bool IsAvailable() const=0;
};

///いろいろな種類のリソース(主にファイル)から、値をIFileとして返します。
///この関数から吐き出されたIFileはまだ使用可能状態とは限らない
///このインターフェイスから生成されるファイルオブジェクトはリードオンリーである
///書き込みするオブジェクトは別に、セーバー、ロガーを用意する予定
class KResourceReaderProvider{
	public:
		static KResourceReaderProvider& Instance();
		///@param fileName ファイル名(ファイル識別文字列→厳密に言うとファイル名と異なる可能性もあるため、このようなあいまいな言い方をしている)
		///@param async 非同期操作フラグ
		///@param listener ファイル操作コールバッククラス
		///@remarks asyncフラグをtrueにすると、実際の読み込みはワーカースレッドに登録される
		///かつ、WindowsにおいてはOVERLAPPED読み込みが採用される・・・と思う
		virtual IResourceReader* Create(const wchar_t* fileName,bool async=false,IFileListener* listener=NULL)=0;

};//

}//end of namespace KFX
#endif