﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga.StandardPlugins.WordArrangement
{
    [Serializable]
    [DisplayName("線形")]
    [Description("単語内要素を直線上に配置します。")]
    public class LinearWordArrangement : IWordArrangement
    {
        public LinearWordArrangement() { }

        public LinearWordArrangement(
            bool isRightToLeftMainText,
            bool isRightToLeftRubyText,
            float distanceToRuby,
            float mainTextSpacing,
            float rubyTextSpacing,
            bool isPerCapita,
            bool hasPadding)
        {
            try
            {
                IsRightToLeftMainText = isRightToLeftMainText;
                IsRightToLeftRubyText = isRightToLeftRubyText;
                DistanceToRuby = distanceToRuby;
                MainTextSpacing = mainTextSpacing;
                RubyTextSpacing = rubyTextSpacing;
                IsPerCapita = isPerCapita;
                HasPadding = hasPadding;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static CatalogItem<IWordArrangement> DefaultLinearWordArrangement =
            new CatalogItem<IWordArrangement>(
                "線形 デフォルト",
                new LinearWordArrangement(
                    false,
                    false,
                    -28,
                    0,
                    0,
                    true,
                    false));

        [Category("単語内要素の配置")]
        [DisplayName("本文を右から並べる")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool IsRightToLeftMainText { get; set; }
        [Category("単語内要素の配置")]
        [DisplayName("ルビを右から並べる")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool IsRightToLeftRubyText { get; set; }
        [CategoryAttribute("単語内要素の配置")]
        [DisplayName("ルビまでの距離")]
        public virtual float DistanceToRuby { get; set; }
        [CategoryAttribute("単語内要素の配置")]
        [DisplayName("本文の文字間隔")]
        public virtual float MainTextSpacing { get; set; }
        [CategoryAttribute("単語内要素の配置")]
        [DisplayName("ルビの文字間隔")]
        public virtual float RubyTextSpacing { get; set; }
        [CategoryAttribute("単語内要素の配置")]
        [DisplayName("均等割り付け")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool IsPerCapita { get; set; }
        [CategoryAttribute("単語内要素の配置")]
        [DisplayName("ルビに合わせてパディングを付ける")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool HasPadding { get; set; }

        public float Measure(KsgWord word)
        {
            try
            {
                float sinAngle, cosAngle;
                sinAngle = (float)Math.Sin(0 * Math.PI / 180);
                cosAngle = (float)Math.Cos(0 * Math.PI / 180);

                float mainTextWidth = 0, rubyTextWidth = 0;
                foreach (KsgCharacter character in word.MainTextCharacters)
                {
                    Corners corners = character.Corners;
                    float width, height, hypotenuse, sinTheta, cosTheta;
                    width = corners.BottomRight.X - corners.BottomLeft.X;
                    height = corners.BottomRight.Y - corners.BottomLeft.Y;
                    hypotenuse = (float)Math.Sqrt(Math.Pow(width, 2) + Math.Pow(height, 2));
                    sinTheta = height / hypotenuse;
                    cosTheta = width / hypotenuse;
                    //cos(a - b) = cos(a) * cos(b) + sin(a) * sin(b)
                    mainTextWidth += hypotenuse * (cosTheta * cosAngle + sinTheta * sinAngle) + MainTextSpacing;
                }
                foreach (KsgCharacter character in word.RubyTextCharacters)
                {
                    Corners corners = character.Corners;
                    float width, height, hypotenuse, sinTheta, cosTheta;
                    width = corners.BottomRight.X - corners.BottomLeft.X;
                    height = corners.BottomRight.Y - corners.BottomLeft.Y;
                    hypotenuse = (float)Math.Sqrt(Math.Pow(width, 2) + Math.Pow(height, 2));
                    sinTheta = height / hypotenuse;
                    cosTheta = width / hypotenuse;
                    //cos(a - b) = cos(a) * cos(b) + sin(a) * sin(b)
                    rubyTextWidth += hypotenuse * (cosTheta * cosAngle + sinTheta * sinAngle) + RubyTextSpacing;
                }
                if (HasPadding && mainTextWidth < rubyTextWidth)
                {
                    return rubyTextWidth;
                }
                else
                {
                    return mainTextWidth;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return 0;
            }
        }

        public void Arrange(KsgWord word, PointF basePoint, float baseAngle)
        {
            try
            {
                PointF originalBasePoint = basePoint;
                float sinAngle, cosAngle;
                sinAngle = (float)Math.Sin(baseAngle * Math.PI / 180);
                cosAngle = (float)Math.Cos(baseAngle * Math.PI / 180);

                float mainTextWidth = 0, rubyTextWidth = 0;
                foreach (KsgCharacter character in word.MainTextCharacters)
                {
                    if (mainTextWidth > 0)
                    {
                        mainTextWidth += MainTextSpacing;
                    }
                    {
                        Corners corners = character.Corners;
                        float width, height, hypotenuse, sinTheta, cosTheta;
                        width = corners.BottomRight.X - corners.BottomLeft.X;
                        height = corners.BottomRight.Y - corners.BottomLeft.Y;
                        hypotenuse = (float)Math.Sqrt(Math.Pow(width, 2) + Math.Pow(height, 2));
                        sinTheta = height / hypotenuse;
                        cosTheta = width / hypotenuse;
                        //cos(a - b) = cos(a) * cos(b) + sin(a) * sin(b)
                        mainTextWidth += hypotenuse * (cosTheta * cosAngle + sinTheta * sinAngle);
                    }
                }
                foreach (KsgCharacter character in word.RubyTextCharacters)
                {
                    if (rubyTextWidth > 0)
                    {
                        rubyTextWidth += RubyTextSpacing;
                    }
                    {
                        Corners corners = character.Corners;
                        float width, height, hypotenuse, sinTheta, cosTheta;
                        width = corners.BottomRight.X - corners.BottomLeft.X;
                        height = corners.BottomRight.Y - corners.BottomLeft.Y;
                        hypotenuse = (float)Math.Sqrt(Math.Pow(width, 2) + Math.Pow(height, 2));
                        sinTheta = height / hypotenuse;
                        cosTheta = width / hypotenuse;
                        //cos(a - b) = cos(a) * cos(b) + sin(a) * sin(b)
                        rubyTextWidth += hypotenuse * (cosTheta * cosAngle + sinTheta * sinAngle);
                    }
                }
                if (mainTextWidth >= rubyTextWidth)
                {
                    PointF point = basePoint;
                    if (IsRightToLeftMainText)
                    {
                        for (int i = word.MainTextCharacters.Count - 1; i >= 0; i--)
                        {
                            ArrangeCharacter(word.MainTextCharacters[i], ref point, cosAngle, sinAngle, MainTextSpacing);
                        }
                    }
                    else
                    {
                        for (int i = 0; i < word.MainTextCharacters.Count; i++)
                        {
                            ArrangeCharacter(word.MainTextCharacters[i], ref point, cosAngle, sinAngle, MainTextSpacing);
                        }
                    }
                    if (word.RubyTextCharacters.Count > 0)
                    {
                        point = basePoint;
                        point.Y += DistanceToRuby;
                        float rubyTextSpacingPlus;
                        if (IsPerCapita)
                        {
                            rubyTextSpacingPlus =
                                (mainTextWidth - rubyTextWidth) / word.RubyTextCharacters.Count;
                            point.X += rubyTextSpacingPlus / 2;
                        }
                        else
                        {
                            rubyTextSpacingPlus = 0;
                            point.X += (mainTextWidth - rubyTextWidth) / 2;
                        }
                        if (IsRightToLeftRubyText)
                        {
                            for (int i = word.RubyTextCharacters.Count - 1; i >= 0; i--)
                            {
                                ArrangeCharacter(word.RubyTextCharacters[i], ref point, cosAngle, sinAngle, RubyTextSpacing);
                            }
                        }
                        else
                        {
                            for (int i = 0; i < word.RubyTextCharacters.Count; i++)
                            {
                                ArrangeCharacter(word.RubyTextCharacters[i], ref point, cosAngle, sinAngle, RubyTextSpacing);
                            }
                        }
                    }
                    basePoint.X += mainTextWidth * cosAngle;
                    basePoint.Y -= mainTextWidth * sinAngle;
                }
                else
                {
                    PointF point = basePoint;
                    float mainTextSpacingPlus;
                    if (HasPadding)
                    {
                        if (IsPerCapita)
                        {
                            mainTextSpacingPlus =
                                (rubyTextWidth - mainTextWidth) / word.MainTextCharacters.Count;
                            point.X += mainTextSpacingPlus / 2;
                        }
                        else
                        {
                            mainTextSpacingPlus = 0;
                            point.X += (rubyTextWidth - mainTextWidth) / 2;
                        }
                    }
                    else
                    {
                        mainTextSpacingPlus = 0;
                    }
                    if (IsRightToLeftMainText)
                    {
                        for (int i = word.MainTextCharacters.Count - 1; i >= 0; i--)
                        {
                            ArrangeCharacter(word.MainTextCharacters[i], ref point, cosAngle, sinAngle, MainTextSpacing);
                        }
                    }
                    else
                    {
                        for (int i = 0; i < word.MainTextCharacters.Count; i++)
                        {
                            ArrangeCharacter(word.MainTextCharacters[i], ref point, cosAngle, sinAngle, MainTextSpacing);
                        }
                    }
                    point = basePoint;
                    point.Y += DistanceToRuby;
                    if (IsRightToLeftRubyText)
                    {
                        for (int i = word.RubyTextCharacters.Count - 1; i >= 0; i--)
                        {
                            ArrangeCharacter(word.RubyTextCharacters[i], ref point, cosAngle, sinAngle, RubyTextSpacing);
                        }
                    }
                    else
                    {
                        for (int i = 0; i < word.RubyTextCharacters.Count; i++)
                        {
                            ArrangeCharacter(word.RubyTextCharacters[i], ref point, cosAngle, sinAngle, RubyTextSpacing);
                        }
                    }
                    if (HasPadding)
                    {
                        basePoint.X += rubyTextWidth * cosAngle;
                        basePoint.Y -= rubyTextWidth * sinAngle;
                    }
                    else
                    {
                        basePoint.X += mainTextWidth * cosAngle;
                        basePoint.Y -= mainTextWidth * sinAngle;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private static void ArrangeCharacter(KsgCharacter character, ref PointF point, float cos, float sin, float spacing)
        {
            try
            {
                Corners corners = character.Corners;
                character.Position = new PointF(
                    point.X + corners.TopLeft.X - corners.BottomLeft.X,
                    point.Y + corners.TopLeft.Y - corners.BottomLeft.Y);
                point = character.Corners.BottomRight;
                point.X += spacing * cos;
                point.Y -= spacing * sin;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public object Clone()
        {
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(LinearWordArrangement));
                StringWriter writer = new StringWriter();
                serializer.Serialize(writer, this);
                StringReader reader = new StringReader(writer.ToString());
                object obj = serializer.Deserialize(reader);
                reader.Close();
                writer.Close();
                return obj;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }
    }
}
