﻿using System;
using System.ComponentModel;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Text;
using System.Xml.Serialization;
using Kasuga.Ass;

namespace Kasuga.StandardPlugins.Formats
{
    [Serializable]
    [DisplayName("精密固定")]
    [Description("高品質なワイプしない字幕を提供します。")]
    public class PreciseFixedFormat : FormatForVSF239, IFormat, IFormatForAss
    {
        public PreciseFixedFormat() { }

        public PreciseFixedFormat(
            string fontFamilyName,
            float fontSize,
            bool isBold,
            bool isItalic,
            bool hasUnderline,
            bool hasStrikeout,
            byte gdiCharSet,
            bool isGdiVerticalFont,
            float scaleX,
            float scaleY,
            float borderWidthX,
            float borderWidthY,
            float shadowDepthX,
            float shadowDepthY,
            float rotateX,
            float rotateY,
            float rotateZ,
            bool useOrigination,
            PointF origination,
            float shearX,
            float shearY,
            float blur,
            Color foreColor,
            Color borderColor,
            Color shadowColor)
        {
            try
            {
                FontFamilyName = fontFamilyName;
                FontSize = fontSize;
                IsBold = isBold;
                IsItalic = isItalic;
                HasUnderline = hasUnderline;
                HasStrikeout = hasStrikeout;
                GdiCharSet = gdiCharSet;
                IsGdiVerticalFont = isGdiVerticalFont;
                ScaleX = scaleX;
                ScaleY = scaleY;
                BorderWidthX = borderWidthX;
                BorderWidthY = borderWidthY;
                ShadowDepthX = shadowDepthX;
                ShadowDepthY = shadowDepthY;
                RotateX = rotateX;
                RotateY = rotateY;
                RotateZ = rotateZ;
                UseOrigination = useOrigination;
                Origination = origination;
                ShearX = shearX;
                ShearY = shearY;
                Blur = blur;
                ForeColor = foreColor;
                BorderColor = borderColor;
                ShadowColor = shadowColor;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static CatalogItem<IFormat> MainTextDefaultPreciseFixedFormat =
            new CatalogItem<IFormat>(
                "精密固定 本文デフォルト",
                new PreciseFixedFormat(
                    "ＭＳ Ｐゴシック",
                    48,
                    false,
                    false,
                    false,
                    false,
                    1,
                    false,
                    1,
                    1,
                    4,
                    4,
                    4,
                    4,
                    0,
                    0,
                    0,
                    false,
                    new PointF(0, 0),
                    0,
                    0,
                    0,
                    Color.Blue,
                    Color.White,
                    Color.FromArgb(byte.MaxValue / 2, Color.Black)));

        public static CatalogItem<IFormat> RubyTextDefaultPreciseFixedFormat =
            new CatalogItem<IFormat>(
                "精密固定 ルビデフォルト",
                new PreciseFixedFormat(
                    "ＭＳ Ｐゴシック",
                    22,
                    false,
                    false,
                    false,
                    false,
                    1,
                    false,
                    1,
                    1,
                    2.5F,
                    2.5F,
                    4,
                    4,
                    0,
                    0,
                    0,
                    false,
                    new PointF(0, 0),
                    0,
                    0,
                    0,
                    Color.Blue,
                    Color.White,
                    Color.FromArgb(byte.MaxValue / 2, Color.Black)));

        [Category("書式")]
        [DisplayName("フォント")]
        [TypeConverter(typeof(FontConverter))]
        [XmlIgnore]
        public override Font Font
        {
            get
            {
                try
                {
                    return base.Font;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    base.Font = value;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Category("書式")]
        [DisplayName("X方向の縮尺")]
        public override float ScaleX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縮尺")]
        public override float ScaleY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の縁取りの太さ")]
        public override float BorderWidthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縁取りの太さ")]
        public override float BorderWidthY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の影の深さ")]
        public override float ShadowDepthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の影の深さ")]
        public override float ShadowDepthY { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点を指定する")]
        public override bool UseOrigination { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点")]
        [TypeConverter(typeof(PointFConverter))]
        public override PointF Origination { get; set; }
        [Category("書式")]
        [DisplayName("X方向の回転角度")]
        public override float RotateX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の回転角度")]
        public override float RotateY { get; set; }
        [Category("書式")]
        [DisplayName("Z方向の回転角度")]
        public override float RotateZ { get; set; }
        [Category("書式")]
        [DisplayName("X方向の剪断変形")]
        public override float ShearX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の剪断変形")]
        public override float ShearY { get; set; }
        [Category("書式")]
        [DisplayName("縁取りのぼかしの強さ")]
        public override float Blur { get; set; }
        [Browsable(false)]
        [XmlIgnore]
        public override Color BeforeForeColor { get; set; }
        [Browsable(false)]
        [XmlIgnore]
        public override Color AfterForeColor { get; set; }
        [Browsable(false)]
        [XmlIgnore]
        public override Color BeforeBorderColor { get; set; }
        [Browsable(false)]
        [XmlIgnore]
        public override Color AfterBorderColor { get; set; }
        [Browsable(false)]
        [XmlIgnore]
        public override Color BeforeShadowColor { get; set; }
        [Browsable(false)]
        [XmlIgnore]
        public override Color AfterShadowColor { get; set; }

        [Category("書式")]
        [DisplayName("内側の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color ForeColor
        {
            get
            {
                try
                {
                    return AfterForeColor;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return Color.Empty;
                }
            }
            set
            {
                try
                {
                    BeforeForeColor = value;
                    AfterForeColor = value;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Category("書式")]
        [DisplayName("縁取りの色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color BorderColor
        {
            get
            {
                try
                {
                    return AfterBorderColor;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return Color.Empty;
                }
            }
            set
            {
                try
                {
                    BeforeBorderColor = value;
                    AfterBorderColor = value;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Category("書式")]
        [DisplayName("影の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color ShadowColor
        {
            get
            {
                try
                {
                    return AfterShadowColor;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return Color.Empty;
                }
            }
            set
            {
                try
                {
                    BeforeShadowColor = value;
                    AfterShadowColor = value;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public override string BeforeForeColorString
        {
            get
            {
                return base.BeforeForeColorString;
            }
            set
            {
                base.BeforeForeColorString = value;
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public override string AfterForeColorString
        {
            get
            {
                return base.AfterForeColorString;
            }
            set
            {
                base.AfterForeColorString = value;
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public override string BeforeBorderColorString
        {
            get
            {
                return base.BeforeBorderColorString;
            }
            set
            {
                base.BeforeBorderColorString = value;
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public override string AfterBorderColorString
        {
            get
            {
                return base.AfterBorderColorString;
            }
            set
            {
                base.AfterBorderColorString = value;
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public override string BeforeShadowColorString
        {
            get
            {
                return base.BeforeShadowColorString;
            }
            set
            {
                base.BeforeShadowColorString = value;
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public override string AfterShadowColorString
        {
            get
            {
                return base.AfterShadowColorString;
            }
            set
            {
                base.AfterShadowColorString = value;
            }
        }

        [Browsable(false)]
        public virtual string ForeColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(ForeColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    ForeColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public virtual string BorderColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(BorderColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    BorderColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public virtual string ShadowColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(ShadowColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    ShadowColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public void DrawForPreview(
            SortedDictionary<int, Bitmap> bitmaps,
            KsgCharacter character,
            bool isWiped)
        {
            try
            {
                if (!char.IsWhiteSpace(character.Char))
                {
                    PreciseDrawingWay.Draw(this, bitmaps, character, isWiped);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void ExportToAss(
            List<AssStyle> styles,
            List<AssEvent> events,
            KsgCharacter character)
        {
            try
            {
                if (char.IsWhiteSpace(character.Char))
                {
                    return;
                }

                AssStyle foreStyle, borderStyle, shadowStyle;
                foreStyle = PreciseDrawingWay.CreateStyle(this, styles, true, FillRegionKind.Fore);
                borderStyle = PreciseDrawingWay.CreateStyle(this, styles, true, FillRegionKind.Border);
                shadowStyle = PreciseDrawingWay.CreateStyle(this, styles, true, FillRegionKind.Shadow);

                string commonTags, shadowCommonTags;
                PreciseDrawingWay.GetPositionTag(this, character, out commonTags, out shadowCommonTags);
                string extendTags = GetExtendTags();
                commonTags += extendTags;
                shadowCommonTags += extendTags;

                PreciseDrawingWay.AddDialogues(
                    events,
                    character.ViewStart,
                    character.ViewEnd,
                    foreStyle,
                    borderStyle,
                    shadowStyle,
                    commonTags,
                    commonTags,
                    shadowCommonTags,
                    character.Char);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public object Clone()
        {
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(PreciseFixedFormat));
                StringWriter writer = new StringWriter();
                serializer.Serialize(writer, this);
                StringReader reader = new StringReader(writer.ToString());
                object obj = serializer.Deserialize(reader);
                reader.Close();
                writer.Close();
                return obj;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }
    }
}
