﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Reflection;
using System.Text;
using Kasuga.Ass;

namespace Kasuga.StandardPlugins.Formats
{
    public class LightDrawingWay
    {
        protected const int LAYER = 100;

        public static void Draw(
            FormatBase format,
            SortedDictionary<int, Bitmap> bitmaps,
            KsgCharacter character,
            bool isWiped)
        {
            try
            {
                if (char.IsWhiteSpace(character.Char))
                {
                    return;
                }
                else
                {
                    Size resolution = character.ParentSubtitle.Resolution;
                    if (!bitmaps.ContainsKey(LAYER))
                    {
                        bitmaps.Add(LAYER, new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb));
                    }

                    //shadow layer
                    if (format.ShadowDepthX != 0 || format.ShadowDepthY != 0)
                    {
                        using (GraphicsPath widenPath = format.GetWidenPath(character))
                        {
                            using (Bitmap bitmap = new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb))
                            {
                                using (Graphics graphics = Graphics.FromImage(bitmap))
                                {
                                    using (Brush brush = isWiped ?
                                        new SolidBrush(format.AfterShadowColor) :
                                        new SolidBrush(format.BeforeShadowColor))
                                    {
                                        using (Matrix matrix = new Matrix())
                                        {
                                            matrix.Translate(format.ShadowDepthX, format.ShadowDepthY);
                                            widenPath.Transform(matrix);
                                            matrix.Dispose();
                                        }
                                        graphics.FillPath(brush, widenPath);
                                        brush.Dispose();
                                    }
                                    graphics.Dispose();
                                }
                                format.Rasterize(bitmap, widenPath);
                                using (Graphics graphics = Graphics.FromImage(bitmaps[LAYER]))
                                {
                                    graphics.DrawImage(bitmap, new Point(0, 0));
                                    graphics.Dispose();
                                }
                                bitmap.Dispose();
                            }
                            widenPath.Dispose();
                        }
                    }

                    //border layer
                    if (format.BorderWidthX > 0 || format.BorderWidthY > 0)
                    {
                        using (GraphicsPath forePath = format.GetForePath(character))
                        {
                            using (GraphicsPath widenPath = format.GetWidenPath(character))
                            {
                                using (Bitmap bitmap = new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb))
                                {
                                    using (Graphics graphics = Graphics.FromImage(bitmap))
                                    {
                                        using (Brush brush = isWiped ?
                                            new SolidBrush(format.AfterBorderColor) :
                                            new SolidBrush(format.BeforeBorderColor))
                                        {
                                            using (Region region = new Region(widenPath))
                                            {
                                                region.Exclude(forePath);
                                                graphics.FillRegion(brush, region);
                                                region.Dispose();
                                            }
                                            brush.Dispose();
                                        }
                                        graphics.Dispose();
                                    }
                                    format.Rasterize(bitmap, widenPath);
                                    using (Graphics graphics = Graphics.FromImage(bitmaps[LAYER]))
                                    {
                                        graphics.DrawImage(bitmap, new Point(0, 0));
                                        graphics.Dispose();
                                    }
                                    bitmap.Dispose();
                                }
                                widenPath.Dispose();
                            }
                            forePath.Dispose();
                        }
                    }

                    //fore layer
                    using (GraphicsPath forePath = format.GetForePath(character))
                    {
                        using (Bitmap bitmap = new Bitmap(resolution.Width, resolution.Height, PixelFormat.Format32bppArgb))
                        {
                            using (Graphics graphics = Graphics.FromImage(bitmap))
                            {
                                using (Brush brush = isWiped ?
                                    new SolidBrush(format.AfterForeColor) :
                                    new SolidBrush(format.BeforeForeColor))
                                {
                                    graphics.FillPath(brush, forePath);
                                    brush.Dispose();
                                }
                                graphics.Dispose();
                            }
                            if (format.BorderWidthX <= 0 && format.BorderWidthY <= 0)
                            {
                                format.Rasterize(bitmap, forePath);
                            }
                            using (Graphics graphics = Graphics.FromImage(bitmaps[LAYER]))
                            {
                                graphics.DrawImage(bitmap, new Point(0, 0));
                                graphics.Dispose();
                            }
                            bitmap.Dispose();
                        }
                        forePath.Dispose();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static AssStyle CreateStyle(
            FormatBase format,
            List<AssStyle> styles,
            bool isWiped)
        {
            try
            {
                AssStyle style = new AssStyle(
                    (styles.Count + 1).ToString(),
                    format.FontName,
                    format.FontSize,
                    isWiped ? format.AfterForeColor : format.BeforeForeColor,
                    Color.Black,
                    isWiped ? format.AfterBorderColor : format.BeforeBorderColor,
                    isWiped ? format.AfterShadowColor : format.BeforeShadowColor,
                    format.IsBold,
                    format.IsItalic,
                    format.HasUnderline,
                    format.HasStrikeout,
                    format.ScaleX,
                    format.ScaleY,
                    0,
                    format.RotateZ,
                    AssBorderStyleKind.Normal,
                    format.BorderWidthX,
                    format.ShadowDepthX,
                    ContentAlignment.MiddleCenter,
                    0,
                    0,
                    0,
                    format.GdiCharSet);

                return Util.CheckContainedStyle(styles, style);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public static string GetPositionTag(
            FormatBase format,
            KsgCharacter character)
        {
            try
            {
                Corners corners = format.GetCharacterCorners(character);
                PointF position = Util.GetIntersectionPoint(corners);
                StringBuilder builder = new StringBuilder();
                builder.Append(@"{\pos(").Append(position.X)
                    .Append(",").Append(position.Y).Append("}");
                return builder.ToString();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return string.Empty;
            }
        }

        public static void AddDialogues(
            List<AssEvent> events,
            PlayTime start,
            PlayTime end,
            AssStyle style,
            string text)
        {
            try
            {
                AssEvent dialogue = new AssEvent(
                    AssEventKind.Dialogue,
                    LAYER,
                    start,
                    end,
                    style.Name,
                    string.Empty,
                    0,
                    0,
                    0,
                    string.Empty,
                    text);
                events.Add(dialogue);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
