﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    [TypeConverter(typeof(KsgSubtitleElementConverter))]
    public class KsgWord : ISubtitleElement, IHasFormatDictionary, IHasWordArrangementDictionary
    {
        public KsgWord()
        {
            try
            {
                MainTextCharacters = new List<KsgCharacter>();
                RubyTextCharacters = new List<KsgCharacter>();
                BaseWordArrangementName = CatalogManager.WordArrangementCatalog[0].Name;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void Initialize(KsgLine parentLine)
        {
            try
            {
                _parentLine = parentLine;
                foreach (KsgCharacter character in MainTextCharacters)
                {
                    character.Initialize(this);
                }
                foreach (KsgCharacter character in RubyTextCharacters)
                {
                    character.Initialize(this);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        [Category("位置")]
        [DisplayName("基準点")]
        [TypeConverter(typeof(PointFConverter))]
        public PointF BasePoint { get; set; }
        [Category("位置")]
        [DisplayName("基準軸の角度")]
        public float BaseAngle { get; set; }

        private string _baseWordArrangementName;

        [Category("単語内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseWordArrangementName
        {
            get
            {
                try
                {
                    return _baseWordArrangementName;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    CatalogItem<IWordArrangement> item = CatalogManager.WordArrangementCatalog.Find((CatalogItem<IWordArrangement> listItem) =>
                    {
                        return listItem.Name == value;
                    });
                    if (item != null)
                    {
                        _baseWordArrangementName = item.Name;
                        WordArrangement = (IWordArrangement)item.Plugin.Clone();
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IWordArrangement BaseWordArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBaseWordArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("単語内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(WordArrangementTypeListConverter))]
        [XmlIgnore]
        public Type WordArrangementType
        {
            get
            {
                try
                {
                    return WordArrangement != null ? WordArrangement.GetType() : null;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IWordArrangement WordArrangement { get; set; }

        [Browsable(false)]
        public WordArrangementWrapper WordArrangementWrapper
        {
            get
            {
                try
                {
                    return new WordArrangementWrapper(BaseWordArrangementName, WordArrangement);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    WordArrangement = value.Plugin;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> WordArrangementDictionary { get; set; }
        [Browsable(false)]
        public List<KsgCharacter> MainTextCharacters { get; set; }
        [Browsable(false)]
        public List<KsgCharacter> RubyTextCharacters { get; set; }
        private KsgLine _parentLine;

        [Browsable(false)]
        [XmlIgnore]
        public KsgSubtitle ParentSubtitle
        {
            get
            {
                try
                {
                    return ParentLine.ParentSubtitle;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgPart ParentPart
        {
            get
            {
                try
                {
                    return ParentLine.ParentPart;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgPage ParentPage
        {
            get
            {
                try
                {
                    return ParentLine.ParentPage;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgLine ParentLine
        {
            get
            {
                try
                {
                    return _parentLine;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> Characters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    characters.AddRange(MainTextCharacters);
                    characters.AddRange(RubyTextCharacters);
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("書式")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseFormatName
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormatName(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseFormatName(Characters, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IFormat BaseFormat
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormat(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("書式")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(FormatTypeListConverter))]
        [XmlIgnore]
        public Type FormatType
        {
            get
            {
                try
                {
                    return Elements.GetFormatType(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> FormatDictionary { get; set; }

        [Browsable(false)]
        [XmlIgnore]
        public GraphicsPath Path
        {
            get
            {
                try
                {
                    GraphicsPath path = new GraphicsPath();
                    foreach(KsgCharacter character in Characters)
                    {
                        GraphicsPath charactersPath = character.Path;
                        path.AddPath(charactersPath, false);
                        charactersPath.Dispose();
                    }
                    return path;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        public void DrawForPreview(
            SortedDictionary<int, Bitmap> bitmaps,
            bool isWiped)
        {
            try
            {
                foreach (KsgCharacter character in Characters)
                {
                    character.DrawForPreview(bitmaps, isWiped);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void ExportToAss(
            List<Ass.AssStyle> styles,
            List<Ass.AssEvent> events)
        {
            try
            {
                foreach (KsgCharacter character in Characters)
                {
                    character.ExportToAss(styles, events);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public float MeasureWidth()
        {
            try
            {
                return WordArrangement.Measure(this);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return 0;
            }
        }

        public void WordArrange()
        {
            try
            {
                WordArrangement.Arrange(this, BasePoint, BaseAngle);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void WordArrange(PointF basePoint, float baseAngle)
        {
            try
            {
                WordArrangement.Arrange(this, basePoint, baseAngle);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetFormatProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgCharacter character in Characters)
                {
                    character.SetFormatProperty(propertyName, value, false);
                }
                RefreshFormatProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshFormatProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgCharacter character in Characters)
                    {
                        character.RefreshFormatProperty(true, false);
                    }
                }
                FormatDictionary = Elements.GetFormat(Characters);
                if (tooParent && ParentLine != null)
                {
                    ParentLine.RefreshFormatProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetWordArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(WordArrangementType);
                PropertyDescriptor descriptor = descriptors.Find(propertyName, true);
                if (descriptor != null && descriptor.Name == propertyName && !descriptor.IsReadOnly && value != null)
                {
                    descriptor.SetValue(WordArrangement, value);
                }
                RefreshWordArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshWordArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(WordArrangementType);
                Dictionary<string, object> dictionary = new Dictionary<string, object>();
                foreach (PropertyDescriptor descriptor in descriptors)
                {
                    dictionary.Add(descriptor.Name, descriptor.GetValue(WordArrangement));
                }
                WordArrangementDictionary = dictionary;
                if (tooParent && ParentLine != null)
                {
                    ParentLine.RefreshWordArrangementProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
