﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    [TypeConverter(typeof(KsgSubtitleElementConverter))]
    public class KsgLine : ISubtitleElement, IHasFormatDictionary, IHasWordArrangementDictionary, IHasLineArrangementDictionary
    {
        public KsgLine()
        {
            try
            {
                Words = new List<KsgWord>();
                BaseLineArrangementName = CatalogManager.LineArrangementCatalog[0].Name;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void Initialize(KsgPage parentPage)
        {
            try
            {
                _parentPage = parentPage;
                foreach (KsgWord word in Words)
                {
                    word.Initialize(this);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        [Category("位置")]
        [DisplayName("基準点")]
        [TypeConverter(typeof(PointFConverter))]
        public PointF BasePoint { get; set; }
        [Category("位置")]
        [DisplayName("基準軸の角度")]
        public float BaseAngle { get; set; }

        private string _baseLineArrangementName;

        [Category("行内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseLineArrangementName
        {
            get
            {
                try
                {
                    return _baseLineArrangementName;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    CatalogItem<ILineArrangement> item = CatalogManager.LineArrangementCatalog.Find((CatalogItem<ILineArrangement> listItem) =>
                    {
                        return listItem.Name == value;
                    });
                    if (item != null)
                    {
                        _baseLineArrangementName = item.Name;
                        LineArrangement = (ILineArrangement)item.Plugin.Clone();
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public ILineArrangement BaseLineArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBaseLineArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("行内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(WordArrangementTypeListConverter))]
        [XmlIgnore]
        public Type LineArrangementType
        {
            get
            {
                try
                {
                    return LineArrangement != null ? LineArrangement.GetType() : null;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public ILineArrangement LineArrangement { get; set; }

        [Browsable(false)]
        public LineArrangementWrapper LineArrangementWrapper
        {
            get
            {
                try
                {
                    return new LineArrangementWrapper(BaseLineArrangementName, LineArrangement);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    LineArrangement = value.Plugin;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> LineArrangementDictionary { get; set; }
        [Browsable(false)]
        public List<KsgWord> Words { get; set; }
        private KsgPage _parentPage;

        [Browsable(false)]
        [XmlIgnore]
        public KsgSubtitle ParentSubtitle
        {
            get
            {
                try
                {
                    return ParentPage.ParentSubtitle;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgPart ParentPart
        {
            get
            {
                try
                {
                    return ParentPage.ParentPart;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgPage ParentPage
        {
            get
            {
                try
                {
                    return _parentPage;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> MainTextCharacters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgWord word in Words)
                    {
                        characters.AddRange(word.MainTextCharacters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> RubyTextCharacters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgWord word in Words)
                    {
                        characters.AddRange(word.RubyTextCharacters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public List<KsgCharacter> Characters
        {
            get
            {
                try
                {
                    List<KsgCharacter> characters = new List<KsgCharacter>();
                    foreach (KsgWord word in Words)
                    {
                        characters.AddRange(word.Characters);
                    }
                    return characters;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("単語内要素の配置")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseWordArrangementName
        {
            get
            {
                try
                {
                    return Elements.GetBaseWordArrangementName(Words);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseWordArrangementName(Words, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IWordArrangement BaseWordArrangement
        {
            get
            {
                try
                {
                    return Elements.GetBaseWordArrangement(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("単語内要素の配置")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(WordArrangementTypeListConverter))]
        [XmlIgnore]
        public Type WordArrangementType
        {
            get
            {
                try
                {
                    return Elements.GetWordArrangementType(Words);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> WordArrangementDictionary { get; set; }

        [Category("書式")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseFormatName
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormatName(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Elements.SetBaseFormatName(Characters, value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IFormat BaseFormat
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormat(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("書式")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(FormatTypeListConverter))]
        [XmlIgnore]
        public Type FormatType
        {
            get
            {
                try
                {
                    return Elements.GetFormatType(Characters);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> FormatDictionary { get; set; }

        [Browsable(false)]
        [XmlIgnore]
        public GraphicsPath Path
        {
            get
            {
                try
                {
                    GraphicsPath path = new GraphicsPath();
                    foreach (KsgCharacter character in Characters)
                    {
                        GraphicsPath charactersPath = character.Path;
                        path.AddPath(charactersPath, false);
                        charactersPath.Dispose();
                    }
                    return path;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        public void DrawForPreview(
            SortedDictionary<int, Bitmap> bitmaps,
            bool isWiped)
        {
            try
            {
                foreach (KsgCharacter character in Characters)
                {
                    character.DrawForPreview(bitmaps, isWiped);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void ExportToAss(
            List<Ass.AssStyle> styles,
            List<Ass.AssEvent> events)
        {
            try
            {
                foreach (KsgCharacter character in Characters)
                {
                    character.ExportToAss(styles, events);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public float MeasureLineWidth()
        {
            try
            {
                return LineArrangement.Measure(this);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return 0;
            }
        }

        public void LineArrange()
        {
            try
            {
                LineArrangement.Arrange(this, BasePoint, BaseAngle);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void LineArrange(PointF basePoint, float baseAngle)
        {
            try
            {
                LineArrangement.Arrange(this, basePoint, baseAngle);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void WordArrange()
        {
            try
            {
                foreach (KsgWord word in Words)
                {
                    word.WordArrange();
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetFormatProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgWord word in Words)
                {
                    word.SetFormatProperty(propertyName, value, false);
                }
                RefreshFormatProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshFormatProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgWord word in Words)
                    {
                        word.RefreshFormatProperty(true, false);
                    }
                }
                FormatDictionary = Elements.GetFormat(Characters);
                if (tooParent && ParentPage != null)
                {
                    ParentPage.RefreshFormatProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetWordArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                foreach (KsgWord word in Words)
                {
                    word.SetWordArrangementProperty(propertyName, value, false);
                }
                RefreshWordArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshWordArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                if (tooChildren)
                {
                    foreach (KsgWord word in Words)
                    {
                        word.RefreshWordArrangementProperty(true, false);
                    }
                }
                WordArrangementDictionary = Elements.GetWordArrangement(Words);
                if (tooParent && ParentPage != null)
                {
                    ParentPage.RefreshWordArrangementProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetLineArrangementProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(LineArrangementType);
                PropertyDescriptor descriptor = descriptors.Find(propertyName, true);
                if (descriptor != null && descriptor.Name == propertyName && !descriptor.IsReadOnly && value != null)
                {
                    descriptor.SetValue(LineArrangement, value);
                }
                RefreshLineArrangementProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshLineArrangementProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(LineArrangementType);
                Dictionary<string, object> dictionary = new Dictionary<string, object>();
                foreach (PropertyDescriptor descriptor in descriptors)
                {
                    dictionary.Add(descriptor.Name, descriptor.GetValue(LineArrangement));
                }
                LineArrangementDictionary = dictionary;
                if (tooParent && ParentPage != null)
                {
                    ParentPage.RefreshLineArrangementProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
