﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Text;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    public class CatalogItemWrapper<T> : IXmlSerializable where T : IPlugin
    {
        public CatalogItemWrapper() { }

        public CatalogItemWrapper(string name, T plugin)
        {
            try
            {
                Name = name;
                Plugin = plugin;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        [XmlIgnore]
        public string Name { get; set; }
        [XmlIgnore]
        public T Plugin { get; set; }

        public XmlSchema GetSchema()
        {
            try
            {
                return null;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public void ReadXml(XmlReader reader)
        {
            try
            {
                reader.Read();
                Name = reader.ReadElementString("Name");
                Type type;
                {
                    Func<AssemblyName, Assembly> assemblyResolver = (AssemblyName assemblyName) =>
                    {
                        string directoryPath =
                            Path.GetDirectoryName(
                            Assembly.GetExecutingAssembly().GetModules()[0].FullyQualifiedName)
                            + "\\Plugins";
                        if (!Directory.Exists(directoryPath))
                        {
                            throw new DirectoryNotFoundException();
                        }
                        string[] dlls = Directory.GetFiles(directoryPath, assemblyName.Name + ".dll", SearchOption.AllDirectories);
                        if (dlls.Length <= 0)
                        {
                            throw new DllNotFoundException();
                        }
                        return Assembly.LoadFrom(dlls[0]);
                    };
                    Func<Assembly, string, bool, Type> typeResolver = (Assembly assembly, string typeName, bool ignoreCase) =>
                    {
                        return assembly.GetType(typeName, true, ignoreCase);
                    };
                    type = Type.GetType(reader.ReadElementString("TypeName"), assemblyResolver, typeResolver, true);
                }
                {
                    XmlSerializer serializer = new XmlSerializer(type);
                    Plugin = (T)serializer.Deserialize(reader);
                }
                reader.Read();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void WriteXml(XmlWriter writer)
        {
            try
            {
                writer.WriteElementString("Name", Name);
                Type type = Plugin.GetType();
                writer.WriteElementString("TypeName", type.AssemblyQualifiedName);
                {
                    XmlSerializer serializer = new XmlSerializer(type);
                    serializer.Serialize(writer, Plugin);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
