﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga.StandardPlugins.PageArrangement
{
    [Serializable]
    [DisplayName("行頭余白均等増加")]
    [Description("各行の行頭余白が下段になるほど大きくなるように配置します。")]
    public class HeadMarginEvenlyIncreaseArrangement : IPageArrangement
    {
        public HeadMarginEvenlyIncreaseArrangement() { }

        public HeadMarginEvenlyIncreaseArrangement(
            bool isTopBase,
            bool isRightHead,
            PointF basePoint,
            float validWidth,
            float baseAngle,
            bool autoCentering,
            float overlapOfLines,
            float distanceToNextLine)
        {
            try
            {
                IsTopBase = isTopBase;
                IsRightHead = isRightHead;
                BasePoint = basePoint;
                ValidWidth = validWidth;
                BaseAngle = baseAngle;
                AutoCentering = autoCentering;
                OverlapOfLines = overlapOfLines;
                DistanceToNextLine = distanceToNextLine;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static CatalogItem<IPageArrangement> DefaultHeadMarginEvenlyIncreaseArrangement =
            new CatalogItem<IPageArrangement>(
                "行頭余白均等増加 デフォルト",
                new HeadMarginEvenlyIncreaseArrangement(
                    false,
                    false,
                    new PointF(90, 435),
                    684,
                    0,
                    true,
                    48,
                    90));

        [Category("ページ内要素の配置")]
        [DisplayName("上揃え")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool IsTopBase { get; set; }
        [Category("ページ内要素の配置")]
        [DisplayName("右揃え")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool IsRightHead { get; set; }
        [Category("ページ内要素の配置")]
        [DisplayName("基準点")]
        [TypeConverter(typeof(PointFConverter))]
        public virtual PointF BasePoint { get; set; }
        [Category("ページ内要素の配置")]
        [DisplayName("基準軸の有効長")]
        public virtual float ValidWidth { get; set; }
        [Category("ページ内要素の配置")]
        [DisplayName("基準軸の角度")]
        public virtual float BaseAngle { get; set; }
        [Category("ページ内要素の配置")]
        [DisplayName("自動センタリング")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool AutoCentering { get; set; }
        [Category("ページ内要素の配置")]
        [DisplayName("センタリング時の行の重なり")]
        public virtual float OverlapOfLines { get; set; }
        [Category("ページ内要素の配置")]
        [DisplayName("次の行までの距離")]
        public virtual float DistanceToNextLine { get; set; }

        public void Arrange(KsgPage page)
        {
            try
            {
                float sin, cos;
                sin = (float)Math.Sin(BaseAngle * Math.PI / 180);
                cos = (float)Math.Cos(BaseAngle * Math.PI / 180);
                List<float> widths = new List<float>();
                foreach (KsgLine line in page.Lines)
                {
                    float width = line.MeasureLineWidth();
                    widths.Add(width);
                }
                if (widths.Count == 1)
                {
                    PointF point = new PointF(
                        BasePoint.X + ((ValidWidth - widths[0]) / 2) * cos,
                        BasePoint.Y - ((ValidWidth - widths[0]) / 2) * sin);
                    if (IsTopBase)
                    {
                        point.X += DistanceToNextLine * sin;
                        point.Y += DistanceToNextLine * cos;
                    }
                    page.Lines[0].LineArrange(point, BaseAngle);
                }
                else if (widths.Count > 1)
                {
                    float increment = 0;
                    for (int i = widths.Count - 1; i >= 0; i++)
                    {
                        if (i <= 0)
                        {
                            increment = 0;
                            break;
                        }
                        increment = (ValidWidth - widths[i]) / i;
                        float maxWidth = 0;
                        for (int j = 0; j < widths.Count; j++)
                        {
                            maxWidth = Math.Max(maxWidth, widths[j] + increment * j);
                        }
                        if (maxWidth <= ValidWidth)
                        {
                            break;
                        }
                    }
                    float margin = 0;
                    if (AutoCentering)
                    {
                        float maxReminder = -OverlapOfLines;
                        for (int i = 0; i < widths.Count - 1; i++)
                        {
                            maxReminder = Math.Max(maxReminder, (widths[i] + increment * i) - (increment * (i + 1)));
                        }
                        if (maxReminder > -OverlapOfLines)
                        {
                            margin = (maxReminder + OverlapOfLines) / widths.Count;
                        }
                    }
                    PointF point = new PointF(BasePoint.X + (margin * cos), BasePoint.Y + (margin * sin));
                    if (IsTopBase)
                    {
                        point.X += DistanceToNextLine * sin;
                        point.Y += DistanceToNextLine * cos;
                    }
                    else
                    {
                        point.X -= DistanceToNextLine * (widths.Count - 1) * sin;
                        point.Y -= DistanceToNextLine * (widths.Count - 1) * cos;
                    }
                    for (int i = 0; i < widths.Count; i++)
                    {
                        page.Lines[i].LineArrange(point, BaseAngle);
                        point.X += increment * cos + DistanceToNextLine * sin;
                        point.Y += increment * sin + DistanceToNextLine * cos;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public object Clone()
        {
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(HeadMarginEvenlyIncreaseArrangement));
                StringWriter writer = new StringWriter();
                serializer.Serialize(writer, this);
                StringReader reader = new StringReader(writer.ToString());
                object obj = serializer.Deserialize(reader);
                reader.Close();
                writer.Close();
                return obj;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }
    }
}
