﻿using System;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Reflection;
using System.Text;
using System.Xml.Serialization;

namespace Kasuga.StandardPlugins.Formats
{
    public abstract class FormatForVSF239 : FormatBase
    {
        [Category("書式")]
        [DisplayName("フォント")]
        [TypeConverter(typeof(FontConverter))]
        [XmlIgnore]
        public override Font Font
        {
            get
            {
                try
                {
                    return base.Font;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    base.Font = value;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Category("書式")]
        [DisplayName("X方向の縮尺")]
        public override float ScaleX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縮尺")]
        public override float ScaleY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の縁取りの太さ")]
        public override float BorderWidthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縁取りの太さ")]
        public override float BorderWidthY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の影の深さ")]
        public override float ShadowDepthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の影の深さ")]
        public override float ShadowDepthY { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点を指定する")]
        public override bool UseOrigination { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点")]
        [TypeConverter(typeof(PointFConverter))]
        public override PointF Origination { get; set; }
        [Category("書式")]
        [DisplayName("X方向の回転角度")]
        public override float RotateX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の回転角度")]
        public override float RotateY { get; set; }
        [Category("書式")]
        [DisplayName("Z方向の回転角度")]
        public override float RotateZ { get; set; }
        [Category("書式")]
        [DisplayName("X方向の剪断変形")]
        public virtual float ShearX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の剪断変形")]
        public virtual float ShearY { get; set; }
        [Category("書式")]
        [DisplayName("縁取りのぼかしの強さ")]
        public virtual float Blur { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の内側の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color AfterForeColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の内側の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color BeforeForeColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の縁取りの色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color AfterBorderColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の縁取りの色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color BeforeBorderColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の影の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color AfterShadowColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の影の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color BeforeShadowColor { get; set; }

        public override PointF Transform(PointF point, RectangleF originalRectangle)
        {
            try
            {
                float x, y, z;
                x = point.X;
                y = point.Y;
                z = 0;
                //\fsc and \fa
                {
                    float _x, _y;
                    _x = x - originalRectangle.X;
                    _y = y - originalRectangle.Y;
                    x = ScaleX * (_x + ShearX * _y) + originalRectangle.X;
                    y = ScaleY * (_y + ShearY * _x) + originalRectangle.Y;
                }
                //\fr
                {
                    PointF origination = UseOrigination ? Origination : originalRectangle.Location;
                    x -= origination.X;
                    y -= origination.Y;

                    float sinX, cosX, sinY, cosY, sinZ, cosZ;
                    sinX = (float)Math.Sin((Math.PI / 180.0) * RotateX);
                    cosX = (float)Math.Cos((Math.PI / 180.0) * RotateX);
                    sinY = (float)Math.Sin((Math.PI / 180.0) * RotateY);
                    cosY = (float)Math.Cos((Math.PI / 180.0) * RotateY);
                    sinZ = (float)Math.Sin((Math.PI / 180.0) * RotateZ);
                    cosZ = (float)Math.Cos((Math.PI / 180.0) * RotateZ);
                    //\frz
                    {
                        float _x = x, _y = y;
                        x = _x * cosZ + _y * sinZ;
                        y = -(_x * sinZ - _y * cosZ);
                    }
                    //\frx
                    {
                        float _y = y, _z = z;
                        y = _y * cosX + _z * sinX;
                        z = _z * sinX - _y * cosX;
                    }
                    //\fry
                    {
                        float _x = x, _z = z;
                        x = _x * cosY + _z * sinY;
                        z = _x * sinY - _z * cosY;
                    }
                    z = Math.Max(z, -19000);
                    x = (x * 20000) / (z + 20000);
                    y = (y * 20000) / (z + 20000);

                    x += origination.X;
                    y += origination.Y;
                }
                return new PointF(x, y);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return PointF.Empty;
            }
        }

        public override void Rasterize(Bitmap bitmap, GraphicsPath widenPath)
        {
            try
            {
                if (widenPath == null)
                {
                    return;
                }

                //\blur
                if (Blur > 0)
                {
                    using (Graphics graphics = Graphics.FromImage(bitmap))
                    {
                        Rectangle replaceRectangle = new Rectangle(0, 0, bitmap.Width, bitmap.Height);
                        {
                            RectangleF clipRectangle = widenPath.GetBounds();
                            clipRectangle.Inflate(Blur * 3, Blur * 3);
                            clipRectangle.Intersect(replaceRectangle);
                            int left, top, right, bottom;
                            left = (int)Math.Floor(clipRectangle.Left);
                            top = (int)Math.Floor(clipRectangle.Top);
                            right = (int)Math.Ceiling(clipRectangle.Right);
                            bottom = (int)Math.Ceiling(clipRectangle.Bottom);
                            replaceRectangle = new Rectangle(left, top, right - left, bottom - top);
                        }
                        using (Bitmap tmpBitmap = new Bitmap(replaceRectangle.Height, replaceRectangle.Width, PixelFormat.Format32bppArgb))
                        {
                            using (Pen pen = new Pen(Color.Black, Blur * 2))
                            {
                                for (int y = replaceRectangle.Top; y < replaceRectangle.Bottom; y++)
                                {
                                    for (int x = replaceRectangle.Left; x < replaceRectangle.Right; x++)
                                    {
                                        Color color = GaussianBlurPointHorizontal(bitmap, x, y, Blur);
                                        tmpBitmap.SetPixel(y - replaceRectangle.Y, x - replaceRectangle.X, color);
                                    }
                                }
                                for (int x = 0; x < replaceRectangle.Height; x++)
                                {
                                    for (int y = 0; y < replaceRectangle.Width; y++)
                                    {
                                        Color color = GaussianBlurPointHorizontal(tmpBitmap, x, y, Blur);
                                        bitmap.SetPixel(y + replaceRectangle.X, x + replaceRectangle.Y, color);
                                    }
                                }
                                pen.Dispose();
                            }
                            tmpBitmap.Dispose();
                        }
                        graphics.Dispose();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private Color GaussianBlurPointHorizontal(Bitmap bitmap, int x, int y, float strongth)
        {
            try
            {
                int range = (int)(strongth * 3);
                double sum = 0, a = 0, r = 0, g = 0, b = 0;
                for (int i = x - range; i <= x + range; i++)
                {
                    if (i < 0 || i >= bitmap.Width)
                    {
                        continue;
                    }
                    double gauss =
                        Math.Exp(-Math.Pow(Math.Abs(i - x), 2) / (2 * Math.Pow(strongth, 2)));
                    Color color = bitmap.GetPixel(i, y);
                    a += color.A * gauss;
                    r += color.R * gauss;
                    g += color.G * gauss;
                    b += color.B * gauss;
                    sum += gauss;
                }
                Func<double, byte> adjustByte = (double value) =>
                {
                    return value < byte.MinValue ? byte.MinValue : value > byte.MaxValue ? byte.MaxValue : (byte)value;
                };
                return Color.FromArgb(
                    adjustByte(a / sum),
                    adjustByte(r / sum),
                    adjustByte(g / sum),
                    adjustByte(b / sum));
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return Color.Transparent;
            }
        }

        public string GetExtendTags()
        {
            try
            {
                StringBuilder builder = new StringBuilder();
                if (BorderWidthY != BorderWidthX)
                {
                    builder.Append(@"{\ybord").Append(BorderWidthY).Append("}");
                }
                if (ShadowDepthY != ShadowDepthX)
                {
                    builder.Append(@"{\yshad").Append(ShadowDepthY).Append("}");
                }
                if (RotateX % 360 != 0)
                {
                    builder.Append(@"{\frx").Append(RotateX).Append("}");
                }
                if (RotateY % 360 != 0)
                {
                    builder.Append(@"{\fry").Append(RotateY).Append("}");
                }
                if (UseOrigination)
                {
                    builder.Append(@"{\org(").Append(Origination.X)
                        .Append(",").Append(Origination.Y).Append("}");
                }
                if (ShearX != 1)
                {
                    builder.Append(@"{fax").Append(ShearX).Append("}");
                }
                if (ShearY != 1)
                {
                    builder.Append(@"{fay").Append(ShearY).Append("}");
                }
                if (Blur > 0)
                {
                    builder.Append(@"{blur").Append(Blur).Append("}");
                }
                return builder.ToString();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return string.Empty;
            }
        }
    }
}
