﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Design;
using System.Drawing.Drawing2D;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga
{
    [Serializable]
    [TypeConverter(typeof(KsgSubtitleElementConverter))]
    public class KsgCharacter : ISubtitleElement, IHasFormatDictionary
    {
        public KsgCharacter()
        {
            try
            {
                Char = char.MinValue;
                ViewStart = PlayTime.Zero;
                ViewEnd = PlayTime.Zero;
                SingTimes = new List<PositionAndPlayTime>();
                SingStart = PlayTime.Zero;
                SingEnd = PlayTime.Zero;
                Position = new PointF(0, 0);
                BaseFormatName = CatalogManager.FormatCatalog[0].Name;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void Initialize(KsgWord parentWord)
        {
            try
            {
                _parentWord = parentWord;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        [Browsable(false)]
        public char Char { get; set; }

        [Browsable(false)]
        [XmlIgnore]
        public bool IsRubyCharacter
        {
            get
            {
                try
                {
                    if (_parentWord.MainTextCharacters.Contains(this))
                    {
                        return false;
                    }
                    else if (_parentWord.RubyTextCharacters.Contains(this))
                    {
                        return true;
                    }
                    else
                    {
                        throw new NotImplementedException();
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return false;
                }
            }
        }

        [Category("時間")]
        [DisplayName("表示開始時間")]
        public PlayTime ViewStart { get; set; }
        [Category("時間")]
        [DisplayName("表示終了時間")]
        public PlayTime ViewEnd { get; set; }
        [Category("時間")]
        [DisplayName("ワイプ時間")]
        public List<PositionAndPlayTime> SingTimes { get; set; }
        [Browsable(false)]
        public PlayTime SingStart { get; set; }
        [Browsable(false)]
        public PlayTime SingEnd { get; set; }
        [Category("位置")]
        [DisplayName("表示位置")]
        [TypeConverter(typeof(PointFConverter))]
        public PointF Position { get; set; }

        [Category("位置")]
        [DisplayName("四隅の座標")]
        [TypeConverter(typeof(ExpandableObjectConverter))]
        [XmlIgnore]
        public Corners Corners
        {
            get
            {
                try
                {
                    return Format.GetCharacterCorners(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        private string _baseFormatName;

        [Category("書式")]
        [DisplayName("ベース")]
        [XmlIgnore]
        public string BaseFormatName
        {
            get
            {
                try
                {
                    return _baseFormatName;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    CatalogItem<IFormat> item = CatalogManager.FormatCatalog.Find((CatalogItem<IFormat> listItem) =>
                    {
                        return listItem.Name == value;
                    });
                    if (item != null)
                    {
                        _baseFormatName = item.Name;
                        Format = (IFormat)item.Plugin.Clone();
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IFormat BaseFormat
        {
            get
            {
                try
                {
                    return Elements.GetBaseFormat(this);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Category("書式")]
        [DisplayName("タイプ")]
        [TypeConverter(typeof(FormatTypeListConverter))]
        [XmlIgnore]
        public Type FormatType
        {
            get
            {
                try
                {
                    return Format != null ? Format.GetType() : null;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public IFormat Format { get; set; }

        [Browsable(false)]
        public FormatWrapper FormatWrapper
        {
            get
            {
                try
                {
                    return new FormatWrapper(BaseFormatName, Format);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    Format = value.Plugin;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [XmlIgnore]
        public Dictionary<string, object> FormatDictionary { get; set; }
        private KsgWord _parentWord;

        [Browsable(false)]
        [XmlIgnore]
        public KsgSubtitle ParentSubtitle
        {
            get
            {
                try
                {
                    return ParentWord.ParentSubtitle;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgPart ParentPart
        {
            get
            {
                try
                {
                    return ParentWord.ParentPart;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgPage ParentPage
        {
            get
            {
                try
                {
                    return ParentWord.ParentPage;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgLine ParentLine
        {
            get
            {
                try
                {
                    return ParentWord.ParentLine;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgWord ParentWord
        {
            get
            {
                try
                {
                    return _parentWord;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public GraphicsPath Path
        {
            get
            {
                try
                {
                    return Corners.Path;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public KsgCharacter NextCharacter
        {
            get
            {
                try
                {
                    KsgLine line = ParentLine;
                    List<KsgCharacter> characters;
                    if (line.MainTextCharacters.Contains(this))
                    {
                        characters = line.MainTextCharacters;
                    }
                    else if (line.RubyTextCharacters.Contains(this))
                    {
                        characters = line.RubyTextCharacters;
                    }
                    else
                    {
                        return null;
                    }

                    int nextIndex = characters.IndexOf(this) + 1;
                    if (nextIndex < characters.Count)
                    {
                        return characters[nextIndex];
                    }
                    else
                    {
                        return null;
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        public void DrawForPreview(
            SortedDictionary<int, Bitmap> bitmaps,
            bool isWiped)
        {
            try
            {
                Format.DrawForPreview(bitmaps, this, isWiped);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void ExportToAss(
            List<Ass.AssStyle> styles,
            List<Ass.AssEvent> events)
        {
            try
            {
                if (Format.GetType().IsSubclassOf(typeof(IFormatForAss)))
                {
                    ((IFormatForAss)Format).ExportToAss(styles, events, this);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void SetFormatProperty(string propertyName, object value, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(FormatType);
                PropertyDescriptor descriptor = descriptors.Find(propertyName, true);
                if (descriptor != null && descriptor.Name == propertyName && !descriptor.IsReadOnly && value != null)
                {
                    descriptor.SetValue(Format, value);
                }
                RefreshFormatProperty(false, tooParent);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void RefreshFormatProperty(bool tooChildren, bool tooParent)
        {
            try
            {
                PropertyDescriptorCollection descriptors = TypeDescriptor.GetProperties(FormatType);
                Dictionary<string, object> dictionary = new Dictionary<string, object>();
                foreach (PropertyDescriptor descriptor in descriptors)
                {
                    dictionary.Add(descriptor.Name, descriptor.GetValue(Format));
                }
                FormatDictionary = dictionary;
                if (tooParent && ParentWord != null)
                {
                    ParentWord.RefreshFormatProperty(false, true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
