#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui Core.
#
# Copyright (C) 2009-2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#

""" 
<comment-ja>
仮想マシン(VM)の状態取得、起動、停止を行う
</comment-ja>
<comment-en>
Get stats of each VM, and start/stop VM.
</comment-en>

@file:   virt.py

@author: Taizo ITO <taizo@karesansui-project.info>

@copyright:    

"""

import sys
#sys.path.insert(1, '/opt/hde/lib/python')
import string
import os, os.path
import time
import tempfile
import re
import libvirt
import libvirtmod
import logging

# define
from libvirt import VIR_DOMAIN_NOSTATE,VIR_DOMAIN_RUNNING,\
     VIR_DOMAIN_BLOCKED,VIR_DOMAIN_PAUSED,VIR_DOMAIN_SHUTDOWN,\
     VIR_DOMAIN_SHUTOFF,VIR_DOMAIN_CRASHED,\
     VIR_STORAGE_POOL_DELETE_NORMAL,\
     VIR_STORAGE_POOL_DELETE_ZEROED, \
     VIR_STORAGE_VOL_DELETE_NORMAL, \
     VIR_STORAGE_VOL_DELETE_ZEROED

import karesansui

from karesansui.lib.const import VIRT_LIBVIRT_DATA_DIR, VIRT_DOMAINS_DIR, \
     VIRT_XML_CONFIG_DIR, VIRT_NETWORK_CONFIG_DIR, \
     VIRT_XENDOMAINS_AUTO_DIR, VIRT_AUTOSTART_CONFIG_DIR, \
     KARESANSUI_GROUP, VNC_PORT_MIN_NUMBER, PORT_MAX_NUMBER, \
     DEFAULT_KEYMAP, VIRT_STORAGE_CONFIG_DIR, \
     DEFAULT_KVM_DISK_FORMAT, DEFAULT_XEN_DISK_FORMAT, \
     DISK_USES

from karesansui.lib.const import XEN_VIRT_CONFIG_DIR, \
                                 XEN_VIRTUAL_DISK_PREFIX, \
                                 XEN_VIRT_URI_RW, XEN_VIRT_URI_RO, \
                                 XEN_KARESANSUI_TMP_DIR, \
                                 XEN_KEYMAP_DIR

from karesansui.lib.const import KVM_VIRT_CONFIG_DIR, \
                                 KVM_VIRTUAL_DISK_PREFIX, \
                                 KVM_VIRT_URI_RW, KVM_VIRT_URI_RO, \
                                 KVM_KARESANSUI_TMP_DIR, \
                                 KVM_KEYMAP_DIR

from karesansui.lib.virt.config import ConfigParam, \
     XMLConfigGenerator, sync_config_generator

from karesansui.lib.virt.config_network import NetworkConfigParam
from karesansui.lib.virt.config_network import NetworkXMLConfigGenerator

from karesansui.lib.virt.config_storage import StorageVolumeConfigParam, \
     StorageVolumeXMLConfigGenerator, StoragePoolConfigParam, StoragePoolXMLConfigGenerator

from karesansui.lib.utils import get_xml_parse        as XMLParse
from karesansui.lib.utils import get_xml_xpath        as XMLXpath
from karesansui.lib.utils import get_nums_xml_xpath   as XMLXpathNum
from karesansui.lib.utils import uniq_sort            as UniqSort
from karesansui.lib.utils import generate_mac_address as GenMAC
from karesansui.lib.utils import execute_command      as ExecCmd
from karesansui.lib.utils import string_from_uuid     as StrFromUUID
from karesansui.lib.utils import generate_uuid        as GenUUID
from karesansui.lib.utils import next_number          as NextNumber
from karesansui.lib.utils import create_disk_img      as MakeDiskImage
from karesansui.lib.utils import copy_file            as CopyFile
from karesansui.lib.net.http import wget              as DownloadFile
from karesansui.lib.utils import is_uuid, get_ifconfig_info, r_chgrp, r_chmod, \
  getfilesize_str, get_filesize_MB, get_disk_img_info, available_virt_uris, \
  is_iso9660_filesystem_format, is_windows_bootable_iso, is_darwin_bootable_iso, \
  file_contents_replace

from karesansui.lib.utils import get_inspect_stack

from karesansui.lib.file.configfile import ConfigFile


os.environ['LIBVIRT_XM_CONFIG_DIR'] = XEN_VIRT_CONFIG_DIR

class KaresansuiVirtException(karesansui.KaresansuiLibException):
    pass

class KaresansuiVirtConnection:

    def __init__(self,uri=None,readonly=True):
        self.__prep()
        self.logger.debug(get_inspect_stack())
        try:
            self.open(uri,readonly)
        except:
            raise KaresansuiVirtException(_("Cannot open '%s'") % uri)

        self.__prep2()

    def __prep(self):
        if not os.path.exists(VIRT_DOMAINS_DIR):
          os.makedirs(VIRT_DOMAINS_DIR)
        if not os.path.exists(VIRT_XML_CONFIG_DIR):
          os.makedirs(VIRT_XML_CONFIG_DIR)
        self.logger = logging.getLogger('karesansui.virt')
        if os.getuid() == 0:
            r_chgrp(VIRT_LIBVIRT_DATA_DIR,KARESANSUI_GROUP)
            r_chmod(VIRT_DOMAINS_DIR,"o-rwx")

    def __prep2(self):
        if not os.path.exists(self.config_dir):
          os.makedirs(self.config_dir)
        self.logger = logging.getLogger('karesansui.virt')
        if os.getuid() == 0:
            r_chgrp(self.config_dir,KARESANSUI_GROUP)

    def open(self, uri,readonly=True):
        if uri == None:
            uris = available_virt_uris()
            try:
                uris["KVM"]
                uri = uris["KVM"]
            except:
                try:
                    uris["XEN"]
                    uri = uris["XEN"]
                except:
                    raise 'error: you must specify connect uri'

        if uri.lower()[0:3] == "xen":
            self.disk_bus    = "xen"
            self.hypervisor  = "XEN"
            self.disk_prefix = XEN_VIRTUAL_DISK_PREFIX
            self.config_dir  = XEN_VIRT_CONFIG_DIR

            if not os.access("/proc/xen", os.R_OK):
                raise 'System is not running a Xen kernel'

        if uri.lower()[0:4] == "qemu":
            self.disk_bus    = "ide"
            self.hypervisor  = "KVM"
            self.disk_prefix = KVM_VIRTUAL_DISK_PREFIX
            self.config_dir  = KVM_VIRT_CONFIG_DIR

            if False == True:
                raise 'System is not running a kvm module'

        if uri != None:
            self.uri = uri

        self.logger.debug('uid=%d' % os.getuid())
        self.logger.debug('gid=%d' % os.getgid())
        
        try:
            """
            if readonly == True:
                self.logger.info('libvirt.openReadOnly - %s' % self.uri)
                self._conn = libvirt.openReadOnly(self.uri)
            else:
                self.logger.info('libvirt.open - %s' % self.uri)
                self._conn = libvirt.open(self.uri)
            """
            self.logger.debug('libvirt.open - %s' % self.uri)
            self._conn = libvirt.open(self.uri)
        except:
            self.logger.error('failed to libvirt open - %s' % self.uri)

        self.logger.debug('succeed to libvirt open - %s' % self.uri)
        self.logger.debug('getType - %s' % self._conn.getType())

        self.guest = KaresansuiVirtGuest(self)
        self.network = KaresansuiVirtNetwork(self)
        self.storage_volume = KaresansuiVirtStorageVolume(self)
        self.storage_pool = KaresansuiVirtStoragePool(self)
        return self._conn

    def close(self, conn=None):
        self.logger.debug(get_inspect_stack())
        if conn == None:
            try:
                conn = self._conn
            except NameError:
                pass
        if conn != None:
            conn.__del__()
            self.logger.debug('succeed to libvirt close - %s' % self.uri)

    def get_virt_type(self):
        return self._conn.getType()

    def get_capabilities(self):
        retval = False
        try:
            xml = self._conn.getCapabilities()
            document = XMLParse(xml)
            host_cpu_arch       = XMLXpath(document,
                               '/capabilities/host/cpu/arch/text()')
            host_cpu_vmx        = XMLXpath(document,
                                '/capabilities/host/cpu/features/vmx')
            host_migration_live = XMLXpath(document,
                                '/capabilities/host/migration_features/live')
            host_migration_uri_transport = XMLXpath(document,
                                '/capabilities/host/migration_features/uri_transports/uri_transport/text()')

            guests = []
            guest_num = XMLXpathNum(document,'/capabilities/guest')
            for n in range(1, guest_num + 1):
                os_type       = XMLXpath(document,
                            '/capabilities/guest[%i]/os_type/text()' % n)
                arch_name     = XMLXpath(document,
                            '/capabilities/guest[%i]/arch/@name' % n)
                arch_wordsize = XMLXpath(document,
                            '/capabilities/guest[%i]/arch/wordsize/text()' % n)
                arch_emulator = XMLXpath(document,
                            '/capabilities/guest[%i]/arch/emulator/text()' % n)
                arch_machine_num   = XMLXpathNum(document,
                            '/capabilities/guest[%i]/arch/machine' % n)
                arch_machines = []
                for m in range(1, arch_machine_num + 1):
                    machine   = XMLXpath(document,
                     '/capabilities/guest[%i]/arch/machine[%i]/text()' % (n,m,))
                    arch_machines.append(machine)
                arch_domain_type = XMLXpath(document,
                            '/capabilities/guest[%i]/arch/domain/@type' % n)

                arch_info = { "name"    :arch_name,
                              "wordsize":arch_wordsize,
                              "emulator":arch_emulator,
                              "machine" :arch_machines,
                              "domain"  :
                                 {"type" :arch_domain_type,
                                 },
                            }
                info = { "os_type":os_type,
                         "arch"   :arch_info,
                       }
                guests.append(info)

            retval = {
               "host":
                  {"cpu":
                     {"arch"          :host_cpu_arch,
                      "vmx"           :host_cpu_vmx,
                     },
                   "migration_features":
                     {"live"          :host_migration_live,
                      "uri_transport" :host_migration_uri_transport,
                     },
                  },
               "guest":guests,
             }

        except:
            pass

        return retval

    def get_version(self):
        hvType = self.get_virt_type()
        ret = libvirtmod.virGetVersion(hvType)
        libVersion = ret[0]
        apiVersion = ret[1]

        libVersion_major = libVersion / 1000000
        libVersion %= 1000000
        libVersion_minor = libVersion / 1000
        libVersion_rel = libVersion % 1000
        #print "Using library: libvir %d.%d.%d" %(libVersion_major, libVersion_minor, libVersion_rel)

        apiVersion_major = apiVersion / 1000000
        apiVersion %= 1000000
        apiVersion_minor = apiVersion / 1000
        apiVersion_rel = apiVersion % 1000
        #print "Using API: %s %d.%d.%d" %(hvType, apiVersion_major, apiVersion_minor, apiVersion_rel)

        return { "libVersion"  : "%d.%d.%d" %(libVersion_major, libVersion_minor, libVersion_rel),
                 "apiVersion"  : "%s %d.%d.%d" %(hvType, apiVersion_major, apiVersion_minor, apiVersion_rel)
               }

    def get_nodeinfo(self):
        info = dict()
        data = self._conn.getInfo()
        info = {
            "model"        : data[0],
            "memory"       : data[1],
            "cpus"         : data[2],
            "mhz"          : data[3],
            "nodes"        : data[4],
            "sockets"      : data[5],
            "cores"        : data[6],
            "threads"      : data[7]
        }
        return info

    def get_mem_info(self):
        """<comment-ja>
        メモリの情報を取得する。
         - guest_alloc_mem: ゲストOSに割り当てているメモリサイズ,
         - host_max_mem: ホストOSのメモリサイズ,
         - host_free_mem: ホストOSの未割り当てメモリサイズ
         - 単位はMB
        @rtype: dict
        </comment-ja>
        <comment-en>
        </comment-en>
        """
        active_guests = self.list_active_guest()
        inactive_guests = self.list_inactive_guest()
        info = self.get_nodeinfo()
        host_max_mem = info['memory']

        guest_alloc_mem = 0
        
        for domname in active_guests + inactive_guests:
            if not domname == "Domain-0":
                virt = self.search_kvg_guests(domname)[0]
                info = virt.get_info()
                guest_alloc_mem += int(info["maxMem"])
                
        guest_alloc_mem /= 1000  # a unit 'MB'

        host_free_mem = host_max_mem - guest_alloc_mem
        if host_free_mem < 0: host_free_mem = 0

        info = {
            'guest_alloc_mem' : guest_alloc_mem,
            'host_max_mem' : host_max_mem,
            'host_free_mem' : host_free_mem,
        }
        return info

    def is_max_vcpus(self, type=None):
        """<comment-ja>
        ゲストに割り当て可能な仮想CPU数の最大値を取得できるか。

        @param type: ハイパーバイザー
        @return: the maximum number of virtual CPUs supported for a
          guest VM of a specific type.
        @rtype: bool
        </comment-ja>
        <comment-en>
        Get the maximum number of vcpu supported for guest.

        @param type: type of hypervisor
        @return: the maximum number of vcpus
        @rtype: bool
        </comment-en>
        """
        if type is None:
            type = self._conn.getType()
        try:
            max = self._conn.getMaxVcpus(type.lower())
            return True
        except libvirt.libvirtError:
            return False

    def get_max_vcpus(self, type=None):
        """<comment-ja>
        ゲストに割り当て可能な仮想CPU数の最大値を取得する

        @param type: ハイパーバイザー
        @return: the maximum number of virtual CPUs supported for a
          guest VM of a specific type.
        @rtype: integer
        </comment-ja>
        <comment-en>
        Get the maximum number of vcpu supported for guest.

        @param type: type of hypervisor
        @return: the maximum number of vcpus
        @rtype: integer
        </comment-en>
        """
        if type is None:
            type = self._conn.getType()
        try:
            max = self._conn.getMaxVcpus(type.lower())
        except libvirt.libvirtError:
            max = 32
        return max

    def get_physical_cpus(self):
        """<comment-ja>
        物理CPU数を取得する

        @return: 物理CPU数
        @rtype: integer
        </comment-ja>
        <comment-en>
        Get the number of phisical CPUs.

        @return: the number of physical CPUs
        @rtype: integer
        </comment-en>
        """
        info = self.get_nodeinfo()
        return info['nodes'] * info['sockets'] * info['cores'] * info['threads']

    """
    Domain-U
    """
    def set_domain_name(self,name=None):
        self.guest.set_domain_name(name)
    def get_domain_name(self):
        return self.guest.get_domain_name()

    def uuid_to_domname(self, uuid):
        try:
            #guest = self._conn.lookupByUUIDString(uuid)
            #return guest.name()
            for guests in self.search_guests():
                if uuid == guests.UUIDString():
                    return guests.name()
        except:
            return ''

    def domname_to_uuid(self, domname):
        try:
            return self.search_guests(domname)[0].UUIDString()
        except:
            return ''


    def list_inactive_guest(self):
        return self._conn.listDefinedDomains()

    def list_active_guest(self):
        names = []
        for id in self._conn.listDomainsID():
            dom = self._conn.lookupByID(id);
            names.append(dom.name())
        return names

    def search_guests(self, name=None):
        guests = []

        if is_uuid(name):
            name = self.uuid_to_domname(name)

        ids = self._conn.listDomainsID()
        for id in ids:
            guests.append(self._conn.lookupByID(id))
        names = self.list_inactive_guest()
        for _name in names:
            guests.append(self._conn.lookupByName(_name))

        if name == None:
            return guests

        for guest in guests:
            if guest.name() == name:
                return [guest]

        #return []
        raise KaresansuiVirtException("guest %s not found" % name)

    def search_kvg_guests(self, name=None):
        """<comment-ja>
        指定されたゲストOSオブジェクトをKaresansuiVirtGuestオブジェクトのlistにして返却する。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """

        if is_uuid(name):
            name = self.uuid_to_domname(name)

        guests = []
        for guest in self.search_guests(name):
            guests.append(
                KaresansuiVirtGuest(conn=self, name=guest.name()))

        return guests

    def list_used_vnc_port(self):
        ports = []
        for guest in self.search_guests(None):
            document = XMLParse(guest.XMLDesc(1))
            vnc_port = XMLXpath(document, '/domain/devices/graphics/@port')
            if vnc_port and int(vnc_port) > 0:
                ports.append(int(vnc_port))

        return UniqSort(ports)

    def list_used_mac_addr(self):
        addrs = []
        for guest in self.search_guests(None):
            document = XMLParse(guest.XMLDesc(1))
            if_num = XMLXpathNum(document,'/domain/devices/interface')
            for n in range(1, if_num + 1):
                mac_addr = XMLXpath(document,'/domain/devices/interface[%i]/mac/@address' % n)
                addrs.append(mac_addr.lower())
        return addrs

    def set_interface_format(self, format=None):

        if format is None:
            format = "b:xenbr0"

        self.interface_format = []
        for _format in format.split(','):
            (type, name) = _format.split(':')
            if type[0] == 'n':
                try:
                    netinfo = self.search_kvn_networks(name)[0].get_info()
                    self.interface_format.append( {"type": "bridge", "name":netinfo['bridge']['name']} )
                except:
                    raise
            else:
                self.interface_format.append( {"type": "bridge", "name":name} )

    def make_domain_dir(self, dir, name):
        # domain dir
        domain_dir = "%s/%s" % (dir, name,)
        domain_images_dir   = "%s/images"   % (domain_dir,)
        domain_boot_dir     = "%s/boot"     % (domain_dir,)
        domain_disk_dir     = "%s/disk"     % (domain_dir,)
        domain_snapshot_dir = "%s/snapshot" % (domain_dir,)
        
        if not os.path.exists(domain_dir):
            os.makedirs(domain_dir)
        if not os.path.exists(domain_images_dir):
            os.makedirs(domain_images_dir)
        if not os.path.exists(domain_boot_dir):
            os.makedirs(domain_boot_dir)
        if not os.path.exists(domain_disk_dir):
            os.makedirs(domain_disk_dir)
        if not os.path.exists(domain_snapshot_dir):
            os.makedirs(domain_snapshot_dir)
        if os.getuid() == 0:
            r_chgrp(domain_dir,KARESANSUI_GROUP)
            r_chmod(domain_dir,"o-rwx")

        return domain_dir

    def create_guest(self, name=None, type="xen", ram=256, disk=None, disksize=1024*16, 
                     mac=None, uuid=None, kernel=None, initrd=None, iso=None, vnc=None,
                     vcpus=None, extra=None, keymap=DEFAULT_KEYMAP, sparse=True,
                     bus=None, disk_format=None,
                     storage_pool=None, storage_volume=None):

        param = ConfigParam(name)

        # Disk
        if type == "kvm" and iso is not None:
            param.add_disk(iso, "hdc", "cdrom") # install iso image

        if bus is not None:
            self.disk_bus = bus

        # Pool
        pool_objs = self.search_kvn_storage_pools(storage_pool)
        if not pool_objs:
            raise KaresansuiVirtException(_("Storage pool could not be found. pool=%s") % \
                                              storage_pool)

        domains_dir = pool_objs[0].get_info()["target"]["path"]
        domain_dir = self.make_domain_dir(domains_dir, name)


        if pool_objs[0].get_info()["type"] == "iscsi":
            disk_type = "block"
            disk = self.get_storage_volume_path(storage_pool, storage_volume)
        else:
            disk_type = "file"
            #disk = "%s/images/%s.img" % (domain_dir, storage_volume)
            disk = "%s/images/%s.img" % (domain_dir, name)

        if disk is None:
            raise KaresansuiVirtException("%s pool=%s,volume=%s" % \
                                          (_("Storage path could not be retrieved."),
                                           storage_pool,
                                           storage_volume
                                           ))
        param.add_disk(disk, self.disk_prefix + "a", bus=self.disk_bus, disk_type=disk_type)

        if mac is None:
            mac = GenMAC()

        if uuid is None:
            uuid = StrFromUUID(GenUUID())

        if vcpus is None:
            vcpus = 1

        if vnc is None:
            used_ports = self.list_used_vnc_port()
            vnc = NextNumber(VNC_PORT_MIN_NUMBER,PORT_MAX_NUMBER,used_ports)

#        if os.path.exists(disk):
#            os.unlink(disk)

        param.set_domain_type(type)
        param.set_uuid(uuid)
        if type == "kvm":
            acpi_info_file = "/proc/acpi/info"
            if os.path.exists(acpi_info_file):
                param.set_features_acpi(True)
            if iso is not None:
                param.set_boot_dev("cdrom")
                if is_windows_bootable_iso(iso) is not False:
                    param.set_features_apic(True)
                elif is_darwin_bootable_iso(iso) is not False:
                    param.set_features_apic(True)
            else:
                param.set_kernel(kernel)
                param.set_initrd(initrd)
        else:
            param.set_kernel(kernel)
            param.set_initrd(initrd)

        param.set_max_vcpus(vcpus)
        param.set_memory(str(ram) + 'm')
        param.set_vnc_keymap(keymap)



        for _format in self.interface_format:
            if _format['name'][0:5] == 'xenbr':
                script = "vif-bridge"
            else:
                script = None

            if mac is None:
                mac = GenMAC()
                param.add_interface(mac,"bridge",_format['name'],script)
            else:
                param.add_interface(mac.lower(),"bridge",_format['name'],script)
                mac = None

        param.set_vnc_port(vnc)
        if extra != None:
            param.append_commandline(extra)
        param.set_behavior("on_shutoff","destroy")
        param.set_behavior("on_reboot","destroy")
        param.set_behavior("on_crash","destroy")

        r = re.compile(r"""(?:ftp|http)s?://""")

        if kernel is not None:
            (kfd, kfn) = tempfile.mkstemp(prefix="vmlinuz.", dir=domain_boot_dir)
            m = r.match(param.get_kernel())
            if m:
              os.close(kfd)
              DownloadFile(param.get_kernel(),kfn)
            else:
              kernel = open(param.get_kernel(),"r")
              os.write(kfd, kernel.read())
              os.close(kfd)
              kernel.close()
            param.set_kernel(kfn)

        if initrd is not None:
            (ifd, ifn) = tempfile.mkstemp(prefix="initrd.img.", dir=domain_boot_dir)
            m = r.match(param.get_initrd())
            if m:
              os.close(ifd)
              DownloadFile(param.get_initrd(),ifn)
            else:
              initrd = open(param.get_initrd(),"r")
              os.write(ifd, initrd.read())
              os.close(ifd)
              initrd.close()
            param.set_initrd(ifn)

        sync_config_generator(param)

        if self._conn is None:
            self._conn = self.open(None)

        generator = XMLConfigGenerator()
        try:
            cfgxml = generator.generate(param)
        except:
            raise

        dom = self._conn.createLinux(cfgxml, 0)
        time.sleep(2)
        self._conn.defineXML(cfgxml)
        time.sleep(1)
        try:
            self._conn.lookupByID(dom.ID())
        except libvirt.libvirtError:
            raise "create_guest() error. name:%s" % (name)

        if initrd is not None:
            os.unlink(param.get_initrd())
        if kernel is not None:
            os.unlink(param.get_kernel())
        param.set_kernel(None)
        param.set_initrd(None)
        param.cmdline = []
        if type == "xen":
            param.set_bootloader("/usr/bin/pygrub")
        elif type == "kvm":
            param.set_boot_dev("hd")
        if type == "kvm" and iso is not None:
            param.delete_disk("hdc")
        param.set_behavior("on_reboot","restart")
        param.set_behavior("on_crash","restart")

        sync_config_generator(param)

        config = "%s/%s.xml" %(VIRT_XML_CONFIG_DIR,name,)
        if os.path.exists(config):
            f = open(config, "r")
            cfgxml= f.read()
            f.close()
            self._conn.defineXML(cfgxml)

    def start_guest(self,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)

        name = self.guest.get_domain_name()
        config = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR,name,)
        if os.path.exists(config):
            f = open(config, "r")
            cfgxml= f.read()
            f.close()
            self._conn.defineXML(cfgxml)

        self.guest.create()

    def shutdown_guest(self,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        self.guest.shutdown()

    def reboot_guest(self,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        self.guest.reboot()

    def destroy_guest(self,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        self.guest.destroy()

    def delete_guest(self,name=None):
        self.destroy_guest(name)

        name = self.guest.get_domain_name()
        domain_dir = "%s/%s" % (VIRT_DOMAINS_DIR,name,)

        if os.path.exists(domain_dir):
            #os.removedirs(domain_dir)
            import shutil
            shutil.rmtree(domain_dir)

        try:
            self.guest.undefine()
        except:
            pass

    def suspend_guest(self,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        self.guest.suspend()

    def resume_guest(self,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        self.guest.resume()

    def take_snapshot(self, name, title=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        self.guest.snapshot(title)

    def apply_snapshot(self,title=None,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)

        name = self.guest.get_domain_name()
        domain_snapshot_dir = "%s/%s/snapshot" % (VIRT_DOMAINS_DIR,name,)
        filename = domain_snapshot_dir + "/" + title
        if os.path.exists(filename):
            try:
                self.guest.restore(filename)
            except:
                return False
        else:
            return False

        return True

    def list_snapshot(self,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)

        name = self.guest.get_domain_name()
        domain_snapshot_dir = "%s/%s/snapshot" % (VIRT_DOMAINS_DIR,name,)
        if os.path.exists(domain_snapshot_dir):
            return os.listdir(domain_snapshot_dir)
        else:
            return None

    def autostart_guest(self,flag=None,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        guests = self.search_guests(self.guest.get_domain_name())
        if len(guests):
            return self.guest.autostart(flag)
        else:
            return False

    def replicate_guest(self, name, source_name, mac=None, uuid=None, vnc=None):

        param = ConfigParam(name)

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, source_name)
        if not os.path.exists(xml_file):
            dom = self._conn.lookupByName(source_name)
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        source_disk = "%s/%s/images/%s.img" % (VIRT_DOMAINS_DIR,source_name,source_name,)
        disk        = "%s/%s/images/%s.img" % (VIRT_DOMAINS_DIR,name,name,)

        src_interfaces = param.interfaces
        param.interfaces = []
        for ifs in src_interfaces:
            script = ifs['script']
            if mac is None:
                mac = GenMAC()
                param.add_interface(mac,"bridge",ifs['bridge'],script)
            else:
                param.add_interface(mac.lower(),"bridge",ifs['bridge'],script)
                mac = None

        if uuid is None:
            uuid = StrFromUUID(GenUUID())

        if vnc is None:
            used_ports = self.list_used_vnc_port()
            vnc = NextNumber(VNC_PORT_MIN_NUMBER,PORT_MAX_NUMBER,used_ports)

        param.disks = []
        param.set_uuid(uuid)
        param.set_vnc_port(vnc)
        param.add_disk(disk,self.disk_prefix + "a")

        try:
            if not os.path.exists(disk):
                CopyFile(source_disk,disk)

            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)
        except:
            raise

        sync_config_generator(param, name)

    def export_guest(self, name, directory=None, title="", extra=None, progresscb=None):

        inactive_pools = self.list_inactive_storage_pool()
        active_pools   = self.list_active_storage_pool()

        pool_name = None
        for _pool in inactive_pools + active_pools:
            path = self.search_kvn_storage_pools(_pool)[0].get_info()["target"]["path"]
            if directory == path:
                pool_name = _pool

        if not os.path.exists(directory):
            raise KaresansuiVirtException(_("Directory '%s' not found.") % directory)

        if not (name is None):
            self.guest.set_domain_name(name)

        xml_file    = "%s/%s.xml"        % (VIRT_XML_CONFIG_DIR, name)
        config_file = "%s/%s"            % (self.config_dir, name)
        domain_dir  = "%s/%s"            % (VIRT_DOMAINS_DIR,name,)
        image_file  = "%s/images/%s.img" % (domain_dir,name)

        uuid = StrFromUUID(GenUUID())
        export_dir        = "%s/%s" % (directory,uuid)
        export_domain_dir = "%s/%s" % (export_dir,name)
        if os.path.exists(export_domain_dir):
            raise KaresansuiVirtException(_("Directory '%s' found.") % export_domain_dir)

        try:
            # copy domain image data
            if progresscb is not None:
                from karesansui.lib.utils import copy_file_cb
                import glob

                src_files = []
                dst_files = []
                for _sub in glob.glob("%s/*" % domain_dir):
                    if os.path.isdir(_sub):
                        dst_dir = "%s/%s" % (export_domain_dir,os.path.basename(_sub),)
                        for _sub2 in glob.glob("%s/*" % _sub):
                            if os.path.isfile(_sub2):
                                src_file = _sub2
                                dst_file = "%s/%s" % (dst_dir,os.path.basename(_sub2),)
                                src_files.append(src_file)
                                dst_files.append(dst_file)
                copy_file_cb(src_files,dst_files,progresscb,each=False)
            else:
                CopyFile(domain_dir,export_domain_dir)

            # copy domain configuration
            export_xml_file    = "%s/%s.xml"  % (export_dir,name,)
            CopyFile(xml_file,   export_xml_file)
            export_config_file = "%s/%s.conf" % (export_dir,name,)
            CopyFile(config_file,export_config_file)

            # symlink to recognize as libvirt pool
            export_image_file = "%s/images/%s.img" % (export_domain_dir,name)
            link_file = "%s/%s-%s.img" % (directory,uuid,name,)
            if os.path.exists(link_file) is False:
                os.symlink(export_image_file,link_file)

            # title to info
            from StringIO import StringIO
            from xml.dom.ext import PrettyPrint
            from xml.dom.DOMImplementation import implementation

            info_file = "%s/info.xml" % (export_dir,)
            doc = implementation.createDocument(None,None,None)

            export_elem = doc.createElement("export")
            export_elem.setAttribute("id", uuid)

            domain_elem = doc.createElement("domain")
            txt_n = doc.createTextNode(name)
            domain_elem.appendChild(txt_n)
            export_elem.appendChild(domain_elem)

            title_elem = doc.createElement("title")
            txt_n = doc.createTextNode(title)
            title_elem.appendChild(txt_n)
            export_elem.appendChild(title_elem)

            created_elem = doc.createElement("created")
            txt_n = doc.createTextNode(str(int(time.time())))
            created_elem.appendChild(txt_n)
            export_elem.appendChild(created_elem)

            # extra start
            extra_elem = doc.createElement("extra")

            for _attr in ["parent_id","notebook_id","created_user_id","modified_user_id","uniq_key","name","attribute","hypervisor","hostname","icon","tags_str","notebook.title","notebook.value"]:
                try:
                    _elem = doc.createElement(_attr)
                    exec("if extra.%s is not None: txt_n = doc.createTextNode(str(extra.%s)); _elem.appendChild(txt_n)" % (_attr,_attr,))
                    #_elem.appendChild(txt_n)
                    extra_elem.appendChild(_elem)
                except:
                    print _attr
                    pass

            export_elem.appendChild(extra_elem)
            # extra end

            doc.appendChild(export_elem)

            out = StringIO()
            PrettyPrint(doc, out)
            ConfigFile(info_file).write(out.getvalue())

            if os.getuid() == 0:
                r_chgrp(export_dir,KARESANSUI_GROUP)

            # TODO: refresh volume-list
            if pool_name is not None:
                try:
                    self.search_storage_pools(pool_name)[0].refresh(True)
                except:
                    pass

        except:
            raise

    def import_guest(self, directory=None, uuid=None, progresscb=None):

        if not os.path.exists(directory):
            raise KaresansuiVirtException(_("Directory '%s' not found.") % directory)

        export_dir = directory

        # read info.xml
        info_file = "%s/info.xml" % (export_dir,)
        try:
            document = XMLParse("".join(ConfigFile(info_file).read()))
            id       = XMLXpath(document, '/export/@id')
            name     = XMLXpath(document, '/export/domain/text()')
            created  = XMLXpath(document, '/export/created/text()')
            title    = XMLXpath(document, '/export/title/text()')
        except:
            raise KaresansuiVirtException(_("'%s' not found or invalid format.") % info_file)

        xml_file    = "%s/%s.xml"        % (VIRT_XML_CONFIG_DIR, name)
        config_file = "%s/%s"            % (self.config_dir, name)
        domain_dir  = "%s/%s"            % (VIRT_DOMAINS_DIR,name,)
        image_file  = "%s/images/%s.img" % (domain_dir,name)

        export_domain_dir = "%s/%s" % (export_dir,name,)

        if not os.path.exists(export_domain_dir):
            raise KaresansuiVirtException(_("Directory '%s' not found.") % export_domain_dir)

        if os.path.exists(domain_dir):
            raise KaresansuiVirtException(_("guest '%s' already exists.") % name)

        try:
            # copy domain image data
            if progresscb is not None:
                from karesansui.lib.utils import copy_file_cb
                import glob

                src_files = []
                dst_files = []
                for _sub in glob.glob("%s/*" % export_domain_dir):
                    if os.path.isdir(_sub):
                        dst_dir = "%s/%s" % (domain_dir,os.path.basename(_sub),)
                        for _sub2 in glob.glob("%s/*" % _sub):
                            if os.path.isfile(_sub2):
                                src_file = _sub2
                                dst_file = "%s/%s" % (dst_dir,os.path.basename(_sub2),)
                                src_files.append(src_file)
                                dst_files.append(dst_file)
                copy_file_cb(src_files,dst_files,progresscb,each=False)
            else:
                CopyFile(export_domain_dir,domain_dir)
                pass

            # copy domain configuration
            export_xml_file    = "%s/%s.xml"  % (export_dir,name,)
            export_config_file = "%s/%s.conf" % (export_dir,name,)
            if uuid is None:
                CopyFile(export_xml_file    ,xml_file)
                CopyFile(export_config_file ,config_file)
            else:
                old_pattern = "<uuid>.{36}</uuid>"
                new_string  = "<uuid>%s</uuid>" % uuid
                file_contents_replace(export_xml_file,xml_file,old_pattern,new_string)
                old_pattern = "^uuid = "
                new_string  = "uuid='%s'" % uuid
                file_contents_replace(export_config_file ,config_file,old_pattern,new_string)

            if os.path.exists(xml_file):
                self._conn.defineXML("".join(ConfigFile(xml_file).read()))

            if os.getuid() == 0:
                r_chgrp(domain_dir,KARESANSUI_GROUP)
                r_chmod(domain_dir,"o-rwx")
                r_chgrp(xml_file,    KARESANSUI_GROUP)
                r_chgrp(config_file, KARESANSUI_GROUP)

        except:
            raise


    """
    Network
    """
    def list_inactive_network(self):
        return self._conn.listDefinedNetworks()

    def list_active_network(self):
        names = []
        for name in self._conn.listNetworks():
            names.append(name)
        return names

    def search_networks(self, name=None):
        networks = []

        names = self._conn.listNetworks()
        for __name in names:
            networks.append(self._conn.networkLookupByName(__name))
        names = self.list_inactive_network()
        for __name in names:
            networks.append(self._conn.networkLookupByName(__name))

        if name == None:
            return networks

        regex_regex = re.compile(r"""^regex:(?P<regex>.*)""")
        m = regex_regex.match(name)

        n_networks = []
        for network in networks:
            network_name = network.name()
            if m == None:
                if network_name == name:
                    return [network]
            else:
                regex = m.group('regex')
                query_regex = re.compile(r""+regex+"")
                n = query_regex.search(network_name)
                if n != None:
                    n_networks.append(network)
        if len(n_networks):
            return n_networks

        #return []
        raise KaresansuiVirtException("network %s not found" % name)

    def search_kvn_networks(self, name=None):
        """<comment-ja>
        指定された仮想ネットワークオブジェクトをKaresansuiVirtNetworkオブジェクトのlistにして返却する。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """

        if is_uuid(name):
            name = self.uuid_to_domname(name)

        networks = []
        for network in self.search_networks(name):
            networks.append(
                KaresansuiVirtNetwork(conn=self, name=network.name()))

        return networks

    def create_network(self, name, cidr, dhcp_start=None, dhcp_end=None, forward=None, bridge=None):
        param = NetworkConfigParam(name)
        param.set_default_networks(cidr,dhcp_start,dhcp_end)
        param.set_ipaddr_and_netmask(cidr)
        if forward:
            if 'dev' in forward.keys():
                param.set_forward_dev(forward['dev'])
            if 'mode' in forward.keys():
                param.set_forward_mode(forward['mode'])
        if bridge:
            param.set_bridge(bridge)
        uuid = StrFromUUID(GenUUID())
        param.set_uuid(uuid)

        generator = NetworkXMLConfigGenerator()
        try:
            cfgxml = generator.generate(param)
        except:
            raise
        generator.writecfg(cfgxml)

        ret = libvirtmod.virNetworkCreateXML(self._conn._o,cfgxml)
        time.sleep(2)
        self._conn.networkDefineXML(cfgxml)

        return ret

    def update_network(self, name, cidr=None, dhcp_start=None, dhcp_end=None, forward=None, bridge=None):
        # パラメータをリセットする場合と引数が無い場合の区別 => リセットの場合は、空文字列を渡す。

        if not ( cidr or
                 dhcp_start or
                 dhcp_end or
                 forward or
                 bridge ):
            # Not changed, do nothing
            # 更新成功時と同じ返り値(0)を返す
            return 0

        try:
            param  = self.search_kvn_networks(name)[0].get_network_config_param()
        except:
            raise KaresansuiVirtException("Can't get parameters of network '%s'." % name)

        if cidr:
            param.set_ipaddr_and_netmask(cidr)
        if dhcp_start:
            param.set_dhcp_start(dhcp_start)
        if dhcp_end:
            param.set_dhcp_end(dhcp_end)
        if forward:
            if 'dev' in forward.keys():
                if forward['dev'] == '':
                    param.set_forward_dev(None)
                else:
                    param.set_forward_dev(forward['dev'])
            if 'mode' in forward.keys():
                if forward['mode'] == '':
                    param.set_forward_mode(None)
                else:
                    param.set_forward_mode(forward['mode'])
        if bridge:
            param.set_bridge(bridge)

        generator = NetworkXMLConfigGenerator()
        try:
            cfgxml = generator.generate(param)
        except:
            raise

        self.stop_network(name)

        generator.writecfg(cfgxml)

        ret = libvirtmod.virNetworkCreateXML(self._conn._o,cfgxml)
        time.sleep(2)
        self._conn.networkDefineXML(cfgxml)

        return ret

    def start_network(self,name=None):
        if not (name is None):
            self.network.set_network_name(name)
        self.network.start()

    def stop_network(self,name=None):
        if not (name is None):
            self.network.set_network_name(name)
        self.network.stop()

    def delete_network(self,name=None):
        self.stop_network(name)
        if len(self.search_networks(name)) > 0:
            self.network.undefine()

        config = "%s/%s.xml" %(VIRT_NETWORK_CONFIG_DIR, self.network.get_network_name())
        if os.path.exists(config):
            os.unlink(config)

        config = "%s/autostart/%s.xml" %(VIRT_NETWORK_CONFIG_DIR, self.network.get_network_name())
        if os.path.exists(config):
            os.unlink(config)

    def autostart_network(self,flag=None,name=None):
        if not (name is None):
            self.guest.set_domain_name(name)
        networks = self.search_networks(self.network.get_network_name())
        if len(networks):
            return self.network.autostart(flag)
        else:
            return False


    """
    Storage Pool
    """
    def list_inactive_storage_pool(self):
        pools = self._conn.listDefinedStoragePools()
        ret = []
        for i in xrange(len(pools)):
            path = "%s/%s.xml" % (VIRT_STORAGE_CONFIG_DIR, pools[i])
            if os.path.isfile(path) is False:
                continue
            ret.append(pools[i])
        return ret

    def list_active_storage_pool(self):
        names = []
        for name in self._conn.listStoragePools():
            names.append(name)
        return names

    def search_storage_pools(self, name=None):
        pools = []

        names = self._conn.listStoragePools()
        for __name in names:
            pools.append(self._conn.storagePoolLookupByName(__name))

        names = self.list_inactive_storage_pool()
        for __name in names:
            pools.append(self._conn.storagePoolLookupByName(__name))

        if name == None:
            return pools

        regex_regex = re.compile(r"""^regex:(?P<regex>.*)""")
        m = regex_regex.match(name)

        n_pools = []
        for pool in pools:
            pool_name = pool.name()
            if m == None:
                if pool_name == name:
                    return [pool]
            else:
                regex = m.group('regex')
                query_regex = re.compile(r""+regex+"")
                n = query_regex.search(storage_name)
                if n != None:
                    n_pools.append(pool)
        if len(n_pools):
            return n_pools

        #return []
        raise KaresansuiVirtException("Storage pool %s not found" % name)

    def search_kvn_storage_pools(self, name=None):
        """<comment-ja>
        指定されたStorage Pool をKaresansuiVirtStoragePoolオブジェクトのlistにして返却する。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """

        if is_uuid(name):
            name = self.uuid_to_domname(name)

        pools = []
        for pool in self.search_storage_pools(name):
            pools.append(
                KaresansuiVirtStoragePool(conn=self, name=pool.name()))

        return pools

    def start_storage_pool(self, name):
        return self.storage_pool.create(name)
        
    def create_storage_pool(self, name, type,
                            target_path=None, mkdir_force=None,
                            allocation=0, available=0, capacity=0,
                            source_f_type=None, source_dev_path=None, source_a_name=None,
                            source_dir_path=None, source_h_name=None,
                            target_p_group=None, target_p_label=None,
                            target_p_mode=None, target_p_owner=None,
                            target_e_format=None, target_encryption_s_type=None,
                            #target_encryption_s_uuid=None,
                            ):

        param = StoragePoolConfigParam(name)
        
        uuid = StrFromUUID(GenUUID())
        param.set_uuid(uuid)

        if type is not None:
            param.set_pool_type(type)

        if target_path is not None:
            param.set_target_path(target_path)

        param.set_allocation(allocation)
        param.set_available(available)
        param.set_capacity(capacity)
        
        if source_f_type is not None:
            param.set_source_f_type(source_f_type)

        if source_dev_path is not None:
            param.set_source_dev_path(source_dev_path)

        if source_dir_path is not None:
            param.set_source_dir_path(source_dir_path)

        if source_f_type is not None:
            param.set_source_f_type(source_f_type)

        if source_h_name is not None:
            param.set_source_h_name(source_h_name)

        if target_e_format is not None and \
               target_encryption_s_type is not None:
            param.set_target_e_format(target_e_format)
            param.set_target_encryption_s_type(target_encryption_s_type)
            target_encryption_s_uuid = StrFromUUID(GenUUID())
            param.set_target_encryption_s_uuid(target_encryption_s_uuid)

        if target_p_group is not None:
            param.set_target_permissions_group(target_p_group)

        if target_p_label is not None:
            param.set_target_permissions_label(target_p_label)

        if target_p_mode is not None:
            param.set_target_permissions_mode(target_p_mode)

        if target_p_owner is not None:
            param.set_target_permissions_owner(target_p_owner)

        generator = StoragePoolXMLConfigGenerator()
        try:
            cfgxml = generator.generate(param)
        except:
            raise
        generator.writecfg(cfgxml)

        if mkdir_force is True and os.path.isdir(target_path) is False:
            os.mkdir(target_path, 0770)

        if self.storage_pool.start(cfgxml, 0, name) == 0:
            return True
        else:
            return False

    def is_autostart_storage_pool(self, name=None):
        if not (name is None):
            self.storage_pool.set_storage_name(name)

        pools = self.search_storage_pools(self.storage_pool.get_storage_name())
        if len(pools):
            ret = self.storage_pool.autostart()
            if ret == 0:
                return False # OFF
            elif ret == 1:
                return True # ON
            else:
                return None # ERR
        else:
            return None # ERR
        
    def autostart_storage_pool(self, flag=None, name=None):
        if not (name is None):
            self.storage_pool.set_storage_name(name)

        pools = self.search_storage_pools(self.storage_pool.get_storage_name())
        if len(pools):
            if self.storage_pool.set_autostart(flag) == 0:
                return True
            else:
                return False
        else:
            return False

    def destroy_storage_pool(self,name=None):
        if name is not None:
            self.storage_pool.set_storage_name(name)
        if self.storage_pool.destroy() == 0:
            return True
        else:
            return False

    def delete_storage_pool(self,name=None, flags=False):
        if name is not None:
            self.storage_pool.set_storage_name(name)

        # autostart off
        self.autostart_storage_pool(False)

        path = "%s/%s.xml" % (VIRT_STORAGE_CONFIG_DIR, self.storage_pool.get_storage_name())
        if os.path.isfile(path) is True:
            os.unlink(path)

        mode = VIR_STORAGE_POOL_DELETE_NORMAL
        if flags is True:
            mode = VIR_STORAGE_POOL_DELETE_ZEROED
            
        if self.storage_pool.delete(mode) == 0:
            return True
        else:
            return False

    def is_storage_volume(self, path):
        """<comment-ja>
        指定したパスがストレージボリュームに含まれているか。
        </comment-ja>
        <comment-en>
        Storage volume that contains the specified path.
        </comment-en>
        """
        if os.path.isfile(path) is False:
            return False

        try:
            vir_storage_vol = self.storage_volume._conn.storageVolLookupByPath(path)
            return True
        except libvirt.libvirtError, e:
            # _("The specified path is not registered in the storage volume. '%s' (%s)")
            return False

    def get_storage_volume(self, pool_name, vol_name):
        try:
            pools = self.search_storage_pools(pool_name)
            if len(pools) <= 0:
                return None
            vol = pools[0].storageVolLookupByName(vol_name)
            return vol
        except libvirt.libvirtError, e:
            return None

    def get_storage_volume_path(self, pool_name, vol_name):
        try:
            vol = self.get_storage_volume(pool_name, vol_name)
            if vol is None:
                return None
            vol_path = vol.path()
            return vol_path
        except libvirt.libvirtError, e:
            return None

    def get_storage_pool_UUIDString2kvn_storage_pool(self, uuidstr):
        pool_obj = self._conn.storagePoolLookupByUUIDString(uuidstr)
        return self.search_kvn_storage_pools(pool_obj.name())
        
    def create_storage_volume(self, name, pool_name,
                              t_f_type,
                              use,
                              #t_path=None,
                              key=None, allocation=0, capacity=0, c_unit=None,
                              source=None,
                              t_p_owner=None,t_p_group=None,t_p_mode=None, t_p_label=None,
                              b_path=None,
                              b_format=None,
                              b_p_owner=None, b_p_group=None, b_p_mode=None, b_p_label=None):

        param = StorageVolumeConfigParam(name)

        uuid = StrFromUUID(GenUUID())
        param.set_uuid(uuid)
        param.set_storage_name(uuid)

        self.storage_pool.set_storage_name(pool_name)
        self.storage_volume.set_storage_volume_name(name)

        #if t_path is not None:
        #   param.set_target_path(t_path)

        param.set_target_f_type(t_f_type)

        if key is not None:
            set_key(key)

        param.set_allocation(allocation)
        param.set_capacity(capacity)

        if c_unit is not None and (capacity != 0 or allocation != 0):
            param.set_c_unit(c_unit)

        if source is not None:
            param.set_source(source)
                
        if t_p_owner is not None:
            param.set_target_permissions_owner(t_p_owner)

        if t_p_group is not None:
            param.set_target_permissions_group(t_p_group)

        if t_p_mode is not None:
            param.set_target_permissions_mode(t_p_mode)

        if t_p_label is not None:
            param.set_target_permissions_label(t_p_label)

        if b_path is not None:
            param.set_backingStore_path(b_path)
            if b_format is not None:
                param.set_backingStore_format(b_format)

            if b_p_owner is not None:
                param.set_backingStore_permissions_owner(b_p_owner)

            if b_p_group is not None:
                param.set_backingStore_permissions_group(b_p_group)

            if b_p_mode is not None:
                param.set_backingStore_permissions_mode(b_p_mode)

            if b_p_label is not None:
                param.set_backingStore_permissions_label(b_p_label)

        generator = StorageVolumeXMLConfigGenerator()
        try:
            cfgxml = generator.generate(param)
        except:
            raise
        
        generator.writecfg(cfgxml)
        
        vir_storage_vol = self.storage_pool.vol_createXML(cfgxml, 0)

        if not isinstance(vir_storage_vol, libvirt.virStorageVol):
            return False
        # storage dir
        pool_objs = self.search_kvn_storage_pools(pool_name)
        if not pool_objs:
            raise KaresansuiVirtException(_("Storage pool could not be found. pool=%s") % \
                                          pool_name)

        domains_dir = pool_objs[0].get_info()["target"]["path"]
        domain_dir = self.make_domain_dir(domains_dir, name)
        if use == DISK_USES["GUEST_OS"]:
            disk = "%s/images/%s.img" % (domain_dir, name)
            os.symlink("%s/%s" % (domains_dir, uuid), disk) # symlink
        else:
            disk = "%s/disk/%s.img" % (domain_dir, name)
            os.symlink("%s/%s" % (domains_dir, uuid), disk) # symlink

        return True

    def delete_storage_volume(self,pool_name, name, flags=False):
        self.storage_volume.set_storage_name(pool_name)
        self.storage_volume.set_storage_volume_name(name)

        mode = VIR_STORAGE_VOL_DELETE_NORMAL
        if flags is True:
            mode = VIR_STORAGE_VOL_DELETE_ZEROED

        if self.storage_volume.delete(mode) == 0:
            return True
        else:
            return False

class KaresansuiVirtGuest:

    def __init__(self, conn, name=None):
        self.connection = conn
        self._conn = self.connection._conn
        self.set_domain_name(name)

    def get_json(self):
        """<comment-ja>
        JSON形式でKaresansuiVirtGuest情報を返却する。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        
        ret = {}
        ret['name'] = self.get_domain_name()
        ret.update(self.get_info())
        return ret

    def set_domain_name(self,name=None):
        if is_uuid(name):
            self.domain = conn.uuid_to_domname(name)
        else:
            self.domain = name

    def get_domain_name(self):
        return self.domain

    def get_info(self):
        dom = self._conn.lookupByName(self.get_domain_name())
        data = dom.info()
        try:
            os_type = dom.OSType()
        except:
            os_type = None

        document = XMLParse(dom.XMLDesc(1))
        vm_type= XMLXpath(document,'/domain/@type')
        hypervisor = self._conn.getType()
        try:
          hvVersion = libvirtmod.virConnectGetVersion(self._conn._o)
          hvVersion_major = hvVersion / 1000000
          hvVersion %= 1000000
          hvVersion_minor = hvVersion / 1000
          hvVersion_rel = hvVersion % 1000
          hv_version = "%s %d.%d.%d" %(hypervisor, hvVersion_major, hvVersion_minor, hvVersion_rel)
        except:
          hv_version = None
        return {
                "state"     : data[0],
                "maxMem"    : data[1],
                "memory"    : data[2],
                "nrVirtCpu" : data[3],
                "cpuTime"   : data[4],
                "OSType"    : os_type,
                "VMType"    : vm_type.upper(),
                "hypervisor": hypervisor,
                "hv_version": hv_version,
        }

    def get_netinfo(self):
        info = {}
        dom = self._conn.lookupByName(self.get_domain_name())
        dom_id = dom.ID()
        if self.get_info()["VMType"] == "KVM":
            eth_info = get_ifconfig_info("regex:^eth")
            for dev,value in eth_info.iteritems():
                info[dev] = value
        else:
            vif_info = get_ifconfig_info("regex:^vif%d\.[0-9]" % dom_id)
            for dev,value in vif_info.iteritems():
                dev = dev.replace("vif%d." % (dom_id,), "eth")
                info[dev] = value
        return info

    def get_disk_info(self):
        infos = []
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if os.path.exists(xml_file):
            document = XMLParse(xml_file)
        else:
            document = XMLParse(dom.XMLDesc(1))

        disk_num = XMLXpathNum(document,'/domain/devices/disk')
        for n in range(1, disk_num + 1):
            driver = {}
            source = {}
            target = {}
            type   = XMLXpath(document,'/domain/devices/disk[%i]/@type' % n)
            device = XMLXpath(document,'/domain/devices/disk[%i]/@device' % n)
            driver['name'] = XMLXpath(document,'/domain/devices/disk[%i]/driver/@name' % n)
            driver['type'] = XMLXpath(document,'/domain/devices/disk[%i]/driver/@type' % n)
            if type == 'block':
                source['dev'] = XMLXpath(document,'/domain/devices/disk[%i]/source/@dev' % n)
                source_path = source['dev']
            else:
                source['file'] = XMLXpath(document,'/domain/devices/disk[%i]/source/@file' % n)
                source_path = source['file']
            target['dev'] = XMLXpath(document,'/domain/devices/disk[%i]/target/@dev' % n)
            target['bus'] = XMLXpath(document,'/domain/devices/disk[%i]/target/@bus' % n)
            
            if os.path.exists(source_path):
                img_info = get_disk_img_info(source_path)
                try:
                    img_size = img_info['virtual_size']
                except:
                    img_size = img_info['real_size']
                source['size'] = get_filesize_MB(img_size)
            else:
                source['size'] = 0
            info = {
                   "type":type,
                   "device":device,
                   "driver":driver,
                   "source":source,
                   "target":target,
                   }
            infos.append(info)
        return infos

    def get_vcpus_info(self):
        dom = self._conn.lookupByName(self.get_domain_name())

        document = XMLParse(dom.XMLDesc(1))
        try:
            max_vcpus = int(XMLXpath(document,'/domain/vcpu/text()'))
        except:
            max_vcpus = None
        try:
            if self.status() != VIR_DOMAIN_SHUTOFF:
                vcpus = self.get_info()['nrVirtCpu']
            else:
                vcpus = None
        except:
            vcpus = None
        try:
            xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
            document = XMLParse(xml_file)
            bootup_vcpus = int(XMLXpath(document,'/domain/vcpu/text()'))
        except:
            bootup_vcpus = None

        return {
                "max_vcpus"    :max_vcpus,
                "vcpus"        :vcpus,
                "bootup_vcpus" :bootup_vcpus,
               }

    def get_interface_info(self):
        infos = []
        dom = self._conn.lookupByName(self.get_domain_name())

        document = XMLParse(dom.XMLDesc(1))
        disk_num = XMLXpathNum(document,'/domain/devices/interface')
        for n in range(1, disk_num + 1):
            mac = {}
            source = {}
            script = {}
            target = {}
            type   = XMLXpath(document,'/domain/devices/interface[%i]/@type' % n)
            mac['address'] = XMLXpath(document,'/domain/devices/interface[%i]/mac/@address' % n)
            source['bridge'] = XMLXpath(document,'/domain/devices/interface[%i]/source/@bridge' % n)
            source['network'] = XMLXpath(document,'/domain/devices/interface[%i]/source/@network' % n)
            source['address'] = XMLXpath(document,'/domain/devices/interface[%i]/source/@address' % n)
            source['port'] = XMLXpath(document,'/domain/devices/interface[%i]/source/@port' % n)
            script['path'] = XMLXpath(document,'/domain/devices/interface[%i]/script/@path' % n)
            target['dev'] = XMLXpath(document,'/domain/devices/interface[%i]/target/@dev' % n)
            info = {
                   "type":type,
                   "mac":mac,
                   "source":source,
                   "script":script,
                   "target":target,
                   }
            infos.append(info)
        return infos

    def get_graphics_info(self):

        """ current info """
        dom = self._conn.lookupByName(self.get_domain_name())
        document = XMLParse(dom.XMLDesc(1))
        type = XMLXpath(document,'/domain/devices/graphics/@type')
        port = XMLXpath(document,'/domain/devices/graphics/@port')
        autoport = XMLXpath(document,'/domain/devices/graphics/@autoport')
        listen = XMLXpath(document,'/domain/devices/graphics/@listen')
        keymap = XMLXpath(document,'/domain/devices/graphics/@keymap')
        current_info = {
                       "type"    :type,
                       "port"    :port,
                       "autoport":autoport,
                       "listen"  :listen,
                       "keymap"  :keymap,
                       }

        """ current setting """
        param = ConfigParam(self.get_domain_name())
        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            dom = self._conn.lookupByName(self.get_domain_name())
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)
        port     = param.get_vnc_port()
        autoport = param.get_vnc_autoport()
        listen   = param.get_vnc_listen()
        keymap   = param.get_vnc_keymap()
        passwd   = param.get_vnc_passwd()
        current_setting = {
                       "port"    :port,
                       "autoport":autoport,
                       "listen"  :listen,
                       "keymap"  :keymap,
                       "passwd"  :passwd,
                       }

        return {"info":current_info,"setting":current_setting}

    def create(self):
        if self.is_creatable() is True:
            time.sleep(1)
            dom = self._conn.lookupByName(self.get_domain_name())
            dom.create()
            for x in range(0,5):
                time.sleep(1)
                if self.status() != VIR_DOMAIN_SHUTOFF:
                    break

    def shutdown(self):
        if self.is_shutdownable() is True:
            time.sleep(1)
            dom = self._conn.lookupByName(self.get_domain_name())
            dom.shutdown()
            for x in range(0,120):
                time.sleep(1)
                if self.status() == VIR_DOMAIN_SHUTOFF:
                    break

    def reboot(self):
        if self.is_shutdownable() is True:
            time.sleep(1)
            dom = self._conn.lookupByName(self.get_domain_name())

            """
            dom.reboot(0)
            """
            dom.shutdown()
            for x in range(0,480):
                time.sleep(1)
                if self.status() == VIR_DOMAIN_SHUTOFF:
                    break
            dom.create()

            for x in range(0,30):
                time.sleep(1)
                if self.status() != VIR_DOMAIN_SHUTOFF:
                    break

    def destroy(self):
        if self.is_destroyable() is True:
            time.sleep(1)
            dom = self._conn.lookupByName(self.get_domain_name())
            dom.destroy()
            for x in range(0,120):
                time.sleep(1)
                if self.status() == VIR_DOMAIN_SHUTOFF:
                    break

    def suspend(self):
        if self.is_suspendable() is True:
            time.sleep(1)
            dom = self._conn.lookupByName(self.get_domain_name())
            dom.suspend()
            for x in range(0,5):
                time.sleep(1)
                if self.status() == VIR_DOMAIN_PAUSED:
                    break

    def resume(self):
        if self.is_resumable() is True:
            time.sleep(1)
            dom = self._conn.lookupByName(self.get_domain_name())
            dom.resume()
            for x in range(0,5):
                time.sleep(1)
                if self.status() != VIR_DOMAIN_PAUSED:
                    break

    def undefine(self):
        dom = self._conn.lookupByName(self.get_domain_name())
        dom.undefine()

    def status(self):
        return self.get_info()["state"]

    def save(self,file):
        dom = self._conn.lookupByName(self.get_domain_name())
        dom.save(file)
        for x in range(0,120):
            time.sleep(1)
            if self.status() == VIR_DOMAIN_SHUTOFF:
                break

    def restore(self,file):

        if not os.path.exists(file):
            raise KaresansuiVirtException("file %s not found" % file)

        if self.status() != VIR_DOMAIN_SHUTOFF:
            self.destroy()

        self._conn.restore(file)
        for x in range(0,60):
            time.sleep(1)
            if self.status() != VIR_DOMAIN_SHUTOFF:
                break

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            dom = self._conn.lookupByName(self.get_domain_name())
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        param.set_current_snapshot(file)

        xml_generator = XMLConfigGenerator()
        cfgxml = xml_generator.generate(param)
        self._conn.defineXML(cfgxml)

        sync_config_generator(param, self.get_domain_name())

    def snapshot(self,title=None):

        status = self.status()
        if status == VIR_DOMAIN_PAUSED:
            self.resume()

        if status == VIR_DOMAIN_SHUTOFF:
            raise KaresansuiVirtException("guest %s is stopped." % self.get_domain_name())

        name = self.get_domain_name()
        domain_snapshot_dir = "%s/%s/snapshot" % (VIRT_DOMAINS_DIR,name,)
        if not os.path.exists(domain_snapshot_dir):
            os.makedirs(domain_snapshot_dir)

        if title == None:
            title = "%f" % time.time()

        filename = domain_snapshot_dir + "/" + title
        self.save(filename)

        if status != VIR_DOMAIN_SHUTOFF:
            self.restore(filename)

        if status == VIR_DOMAIN_PAUSED:
            self.suspend()

    def autostart(self, flag=None):

        if self._conn.getType() == "Xen":
            autostart_file = "%s/%s" %(VIRT_XENDOMAINS_AUTO_DIR,self.get_domain_name())

            if flag == True:
                if not os.path.exists(autostart_file):
                    command_args = [
                        "/bin/ln", "-s",
                        "%s/%s" %(self.connection.config_dir,self.get_domain_name()),
                        "%s" % VIRT_XENDOMAINS_AUTO_DIR
                    ]
                    ret = ExecCmd(command_args)
            elif flag == False:
                if os.path.exists(autostart_file):
                    os.unlink(autostart_file)
            else:
                return os.path.lexists(autostart_file)

        if self._conn.getType() == "QEMU":
            autostart_file = "%s/%s.xml" %(VIRT_AUTOSTART_CONFIG_DIR,self.get_domain_name())
            dom = self._conn.lookupByName(self.get_domain_name())
            if flag == True:
                return dom.setAutostart(flag)
            elif flag == False:
                return dom.setAutostart(flag)
            else:
                return os.path.exists(autostart_file)

    def next_disk_target(self,bus=None):
        dom = self._conn.lookupByName(self.get_domain_name())
        serials = []

        document = XMLParse(dom.XMLDesc(1))

        disk_num = XMLXpathNum(document,'/domain/devices/disk')

        if bus is None:
            bus = self.connection.disk_bus

        if bus == "virtio":
            prefix_regex = "vd"
        elif bus == "scsi":
            prefix_regex = "sd"
        else:
            prefix_regex = "hd|xvd"

        prefix = None
        for n in range(1, disk_num + 1):
            target_dev = XMLXpath(document,'/domain/devices/disk[%i]/target/@dev' % n)
            p = re.compile(r"""^(?P<prefix>%s)(?P<serial>[a-z])$""" % prefix_regex)
            m = p.match(target_dev)
            if m is not None:
                prefix = m.group("prefix")
                serials.append(m.group("serial"))

        if prefix is None:
            prefix = prefix_regex

        for i,_x in enumerate('abcdefghijklmnopqrstuvwxyz'):
          if not _x in serials:
            next_serial = _x
            break

        return "%s%s" %(prefix, next_serial)

    def add_disk(self, path, target, size, is_sparse=True, bus=None):
        name = self.get_domain_name()

        domain_disk_dir = "%s/%s/disk" % (VIRT_DOMAINS_DIR,name,)
        if not os.path.exists(domain_disk_dir):
            os.makedirs(domain_disk_dir)

        if bus is None:
            bus = self.connection.disk_bus

        try:
            MakeDiskImage(path,int(size),"raw", is_sparse)
            return self.append_disk(path,target,bus)
        except:
            if os.path.exists(path) is True:
                os.remove(path)
            raise

    def append_disk(self,path,target,bus=None):

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        if bus is None:
            bus = self.connection.disk_bus

        param.add_disk(path,target,"disk",bus)

        try:
            from karesansui.lib.virt.config import XMLDiskConfigGenerator
            generator = XMLDiskConfigGenerator()
            generator.set_path(path)
            generator.set_target(target)
            cfg = generator.generate(None)

            # qemu: cannot attach device on inactive domain
            if dom._conn.getType() == "QEMU" and dom.isActive() == 0:
                True
            # qemu: disk bus 'ide' cannot be hotplugged.
            elif dom._conn.getType() == "QEMU" and bus is not None and bus == "ide":
                True
            else:
                dom.attachDevice(cfg)

            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)

        except:
            raise

        sync_config_generator(param, self.get_domain_name())

    def delete_disk(self,target):
        status = self.status()
        if status == VIR_DOMAIN_PAUSED:
            self.resume()
            time.sleep(2)
            #raise KaresansuiVirtException("Domain %s is suspended." % self.get_domain_name())

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        path = param.get_disk_path(target)

        # physical disk remove
        if path is not None and os.path.exists(path) is True:
            try:
                os.remove(path)
            except:
                self.logger.info("You do not have a disk file. - %s" % path)
                raise
        param.delete_disk(target)

        try:
            from karesansui.lib.virt.config import XMLDiskConfigGenerator
            generator = XMLDiskConfigGenerator()
            generator.set_target(target)
            generator.set_path(path)
            cfg = generator.generate(None)
            """
            try:
                dom.detachDevice(cfg)
            except:
                xml_generator = XMLConfigGenerator()
                cfgxml = xml_generator.generate(param)
                self._conn.defineXML(cfgxml)
            """
            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)

            if status == VIR_DOMAIN_PAUSED:
                self.suspend()
        except:
            raise

        sync_config_generator(param, self.get_domain_name())

    def append_interface(self,mac,bridge=None,network=None):

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        if network is not None:
            netinfo = self.connection.search_kvn_networks(network)[0].get_info()
            bridge = netinfo['bridge']['name']

        if bridge[0:5] == 'xenbr':
            script = "vif-bridge"
        else:
            script = None
        mac = mac.lower()
        param.add_interface(mac,"bridge",bridge,script)

        try:
            from karesansui.lib.virt.config import XMLInterfaceConfigGenerator
            generator = XMLInterfaceConfigGenerator()
            generator.set_mac(mac)
            generator.set_bridge(bridge)
            if script is not None:
                generator.set_script(script)
            cfg = generator.generate(None)

            # qemu: cannot attach device on inactive domain
            if dom._conn.getType() == "QEMU" and dom.isActive() == 0:
                True
            else:
                dom.attachDevice(cfg)

            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)

        except:
            raise

        sync_config_generator(param, self.get_domain_name())

    def delete_interface(self,mac,force=False):

        status = self.status()
        if status == VIR_DOMAIN_PAUSED:
            self.resume()
            time.sleep(2)
            #raise KaresansuiVirtException("Domain %s is suspended." % self.get_domain_name())

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)

        param.load_xml_config(xml_file)
        current_snapshot = param.get_current_snapshot()
        if force is True:
            param.load_xml_config(dom.XMLDesc(0))
            if current_snapshot is not None:
                param.set_current_snapshot(current_snapshot)

        bridge = None
        for arr in param.interfaces:
            if arr["mac"] == mac:
                bridge = arr['bridge']

        if bridge is None:
            document = XMLParse(dom.XMLDesc(0))
            if_num = XMLXpathNum(document,'/domain/devices/interface')
            for n in range(1, if_num + 1):
                mac_addr = XMLXpath(document,'/domain/devices/interface[%i]/mac/@address' % n)
                if mac_addr == mac:
                    bridge = XMLXpath(document,'/domain/devices/interface[%i]/source/@bridge' % n)


        mac = mac.lower()
        param.delete_interface(mac)

        try:
            from karesansui.lib.virt.config import XMLInterfaceConfigGenerator
            generator = XMLInterfaceConfigGenerator()
            generator.set_mac(mac)
            if bridge is not None:
                generator.set_bridge(bridge)
            cfg = generator.generate(None)

            if self._conn.getType() == "Xen":
                try:
                    dom.detachDevice(cfg)
                except:
                    pass

            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)

            if status == VIR_DOMAIN_PAUSED:
                self.suspend()
        except:
            raise

        sync_config_generator(param, self.get_domain_name())

    def modify_mac_address(self,old,new):

        status = self.status()
        if status == VIR_DOMAIN_PAUSED:
            self.resume()
            time.sleep(2)
            #raise KaresansuiVirtException("Domain %s is suspended." % self.get_domain_name())

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        new_interfaces = []

        old = old.lower()
        new = new.lower()
        for arr in param.interfaces:
            if arr["mac"] == old:
                bridge = arr['bridge']
                arr["mac"] = new
            new_interfaces.append(arr)
        param.interfaces = new_interfaces

        try:
            """
            try:
                self.delete_interface(old,True)
                self.append_interface(new,bridge)
            except:
                xml_generator = XMLConfigGenerator()
                cfgxml = xml_generator.generate(param)
                self._conn.defineXML(cfgxml)
            """
            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)

            if status == VIR_DOMAIN_PAUSED:
                self.suspend()
        except:
            raise

        sync_config_generator(param, self.get_domain_name())

    def set_memory(self,maxmem=None,memory=None):

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        if maxmem:
            param.set_max_memory(maxmem)
        if memory:
            param.set_memory(memory)

        try:
            dom.setMaxMemory(param.get_max_memory("k"))
            dom.setMemory(param.get_memory("k"))

            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)
        except:
            raise

        sync_config_generator(param, self.get_domain_name())

    def set_vcpus(self,max_vcpus=None,vcpus=None):

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        if max_vcpus is not None:
            param.set_max_vcpus(int(max_vcpus))

        if vcpus is not None:
            param.set_vcpus(int(vcpus))

        param.set_max_vcpus_limit(int(self.connection.get_max_vcpus()))
        param.set_vcpus_limit(int(self.get_vcpus_info()['max_vcpus']))

        try:
            # qemu: cannot change vcpu count of an active domain
            if dom._conn.getType() == "QEMU" and dom.isActive() == 1:
                True
            else:
                dom.setVcpus(param.get_vcpus())

            xml_generator = XMLConfigGenerator()
            cfgxml = xml_generator.generate(param)
            self._conn.defineXML(cfgxml)
        except:
            raise

        sync_config_generator(param, self.get_domain_name())

    def set_vnc(self,port=None,listen=None,passwd=None,keymap=None):

        from karesansui.lib.virt.config import ConfigParam
        param = ConfigParam(self.get_domain_name())
        dom = self._conn.lookupByName(self.get_domain_name())

        xml_file = "%s/%s.xml" % (VIRT_XML_CONFIG_DIR, self.get_domain_name())
        if not os.path.exists(xml_file):
            ConfigFile(xml_file).write(dom.XMLDesc(0))
            if os.getuid() == 0 and os.path.exists(xml_file):
                r_chgrp(xml_file,KARESANSUI_GROUP)
        param.load_xml_config(xml_file)

        if port is not None:
            param.set_vnc_port(port)

        if listen is not None:
            param.set_vnc_listen(listen)

        if passwd is not None:
            param.set_vnc_passwd(passwd)

        if keymap is not None:
            param.set_vnc_keymap(keymap)

        xml_generator = XMLConfigGenerator()
        cfgxml = xml_generator.generate(param)
        self._conn.defineXML(cfgxml)

        sync_config_generator(param, self.get_domain_name())

    def is_creatable(self):
        """<comment-ja>
        ゲストOS(ドメイン)を起動することができるか。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        if self.status() == VIR_DOMAIN_SHUTOFF:
            return True
        else:
            return False

    def is_shutdownable(self):
        """<comment-ja>
        ゲストOS(ドメイン)をシャットダウンすることができるか。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        
        status = self.status()
        if status == VIR_DOMAIN_RUNNING \
               or status == VIR_DOMAIN_BLOCKED \
               or status == VIR_DOMAIN_PAUSED:
            return True
        else:
            return False

    def is_destroyable(self):
        """<comment-ja>
        ゲストOS(ドメイン)を強制停止することができるか。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        status = self.status()
        if status == VIR_DOMAIN_RUNNING \
               or status == VIR_DOMAIN_BLOCKED \
               or status == VIR_DOMAIN_PAUSED:
            return True
        else:
            return False

    def is_suspendable(self):
        """<comment-ja>
        ゲストOS(ドメイン)の一時停止することができるか。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        status = self.status()
        if status == VIR_DOMAIN_NOSTATE \
               or status ==VIR_DOMAIN_RUNNING \
               or status == VIR_DOMAIN_BLOCKED:
            return True
        else:
            return False

    def is_resumable(self):
        """<comment-ja>
        ゲストOS(ドメイン)再開することができるか。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        if self.status() == VIR_DOMAIN_PAUSED:
            return True
        else:
            return False

    def is_active(self):
        """<comment-ja>
        ゲストOSの状態がactiveか。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        return (self.get_domain_name() in self.connection.list_active_guest())

    def is_inactive(self):
        """<comment-ja>
        ゲストOSの状態がinactiveか。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        return (self.get_domain_name() in self.connection.list_inactive_guest())

    def is_takable_snapshot(self):
        """<comment-ja>
        スナップショットを作成できる状態か。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        if self.status() == VIR_DOMAIN_SHUTOFF:
            return False
        else:
            return True

    
class KaresansuiVirtNetwork:

    def __init__(self, conn, name=None):
        self.connection = conn
        self._conn = self.connection._conn
        self.set_network_name(name)

    def set_network_name(self,name=None):
        self.network_name = name
    def get_network_name(self):
        return self.network_name

    def load(self):
        param = NetworkConfigParam(self.get_network_name())
        param.load_xml_config("%s/%s.xml" % (VIRT_NETWORK_CONFIG_DIR, self.get_network_name()))

    def start(self):
        net = self._conn.networkLookupByName(self.get_network_name())
        try:
            net.create()
        except libvirt.libvirtError, e:
            raise KaresansuiVirtException(_("Could not start network '%s' (%s)") % (self.network_name, e))

    def stop(self):
        net = self._conn.networkLookupByName(self.get_network_name())
        try:
            if net.isActive() != 0:
                net.destroy()
        except libvirt.libvirtError, e:
            raise KaresansuiVirtException(_("Could not stop network '%s' (%s)") % (self.network_name, e))

    def undefine(self):
        net = self._conn.networkLookupByName(self.get_network_name())
        net.undefine()

    def autostart(self, flag=None):
        net = self._conn.networkLookupByName(self.get_network_name())
        net.setAutostart(flag)

    def get_json(self):
        """<comment-ja>
        JSON形式でKaresansuiVirtNetwork情報を返却する。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        
        ret = {}
        ret['name'] = self.get_network_name()
        ret.update(self.get_info())
        return ret

    def get_info(self):
        try:
            net = self._conn.networkLookupByName(self.get_network_name())

            document = XMLParse(net.XMLDesc(0))
            autostart = net.autostart()
        except KaresansuiVirtException, e:
            return False

        name = XMLXpath(document,'/network/name/text()')
        uuid = XMLXpath(document,'/network/uuid/text()')
        bridge_name = XMLXpath(document,'/network/bridge/@name')
        bridge_stp = XMLXpath(document,'/network/bridge/@stp')
        bridge_forwardDelay = XMLXpath(document,'/network/bridge/@forwardDelay')
        bridge = {
                   "name": bridge_name,
                   "stp": bridge_stp,
                   "forwardDelay": bridge_forwardDelay,
                 }
        dhcp_range_start = XMLXpath(document,'/network/ip/dhcp/range/@start')
        dhcp_range_end = XMLXpath(document,'/network/ip/dhcp/range/@end')
        dhcp = {
                   "start": dhcp_range_start,
                   "end" : dhcp_range_end,
               }
        ip_address = XMLXpath(document,'/network/ip/@address')
        ip_netmask = XMLXpath(document,'/network/ip/@netmask')
        ip = {
                   "address": ip_address,
                   "netmask": ip_netmask,
                   "dhcp"   : dhcp,
             }
        forward_mode = XMLXpath(document,'/network/forward/@mode')
        forward_dev = XMLXpath(document,'/network/forward/@dev')
        forward = {
                   "mode": forward_mode,
                   "dev" : forward_dev,
             }

        is_active = self.is_active()

        return {
                "name"     : name,
                "uuid"     : uuid,
                "bridge"   : bridge,
                "dhcp"     : dhcp,
                "ip"       : ip,
                "forward"  : forward,
                "autostart": autostart,
                "is_active" : is_active,
        }

    def get_network_config_param(self):
        return NetworkConfigParam(self.get_info())

    def is_active(self):
        return (self.network_name in self._conn.listNetworks())

    def is_inactive(self):
        return (self.network_name in self._conn.listDefinedNetworks())

class KaresansuiVirtStorage:

    def __init__(self, conn, name=None):
        self.connection = conn
        self._conn = self.connection._conn
        self.set_storage_name(name)

    def set_storage_name(self,name=None):
        self.storage_name = name

    def get_storage_name(self):
        return self.storage_name

class KaresansuiVirtStoragePool(KaresansuiVirtStorage):

    def build(self):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        try:
            return pool.build(libvirt.VIR_STORAGE_POOL_BUILD_NEW)
        except Exception, e:
            raise KaresansuiVirtException("Could not build storage pool: %s"
                                            % str(e))

    def create(self, name=None, flags=0):
        if name is not None:
            self.set_storage_name(name)
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        try:
            ret = pool.create(flags)
            pool.refresh(0)
            return ret
        except Exception, e:
            raise KaresansuiVirtException("Could not create storage pool: %s"
                                            % str(e))

    def start(self, cfgxml, flags, name=None):
        if name:
            self.set_storage_name(name)
            
        # define
        try:
            ret = self._conn.storagePoolDefineXML(cfgxml, flags) # virStoragePoolDefineXML
            #ret = libvirtmod.virStoragePoolCreateXML(self._conn._o,cfgxml, 0)
        except libvirt.libvirtError, e:
            raise KaresansuiVirtException("Could not start pool '%s' (%s)" \
                                          % (self.get_storage_name, e))
        ret1 = self.build()
        ret2 = self.create()
        return ret2

    def load(self):
        param = StoragePoolConfigParam(self.get_storage_name())
        param.load_xml_config("%s/%s.xml" \
                              % (VIRT_STORAGE_CONFIG_DIR, self.get_storage_name()))


    def destroy(self):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        try:
            return pool.destroy()
        except Exception, e:
            raise KaresansuiVirtException("Could not destroy storage pool: %s" \
                                            % str(e))

    def delete(self, flags):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        try:
            return pool.delete(flags)
        except Exception, e:
            raise KaresansuiVirtException("Could not delete storage pool: %s" \
                                            % str(e))

    def autostart(self):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        return pool.autostart()

    def set_autostart(self, flag=None):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        return pool.setAutostart(flag)

    def get_json(self):
        """<comment-ja>
        JSON形式でKaresansuiVirtNetwork情報を返却する。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        ret = {}
        ret['name'] = self.get_storage_name()
        ret.update(self.get_info())
        return ret

    def get_info(self):
        try:
            pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        except Exception, e:
            raise KaresansuiVirtException("Could not get the storage pool: %s" \
                                            % str(e))

        param = StoragePoolConfigParam(self.get_storage_name())
        param.load_xml_config(pool.XMLDesc(0))
        return {"name" : param.get_storage_name(),
                "uuid" : param.get_uuid(),
                "type" : param.get_pool_type(),
                "allocation" : param.get_allocation(),
                "capacity" : param.get_capacity(),
                "available" : param.get_available(),
                "source" : {"dev_path" : param.get_source_dev_path(),
                            "dir_path" : param.get_source_dir_path(),
                            "h_name" : param.get_source_h_name(),
                            "f_type" : param.get_source_f_type(),
                            },
                "target" : {"path" : param.get_target_path(),
                            "p_owner" : param.get_target_permissions_owner(),
                            "p_group" : param.get_target_permissions_group(),
                            "p_mode" : param.get_target_permissions_mode(),
                            "p_label" : param.get_target_permissions_label(),
                            "e_format" : param.get_target_e_format(),
                            "e_s_type" : param.get_target_encryption_s_type(),
                            "e_s_uuid" : param.get_target_encryption_s_uuid(),
                            },
                "is_active" : self.is_active(),
            }

    def is_active(self):
        return (self.storage_name in self._conn.listStoragePools())

    def is_inactive(self):
        return (self.storage_name in self._conn.listdefinedStoragePools())

    def vol_createXML(self, xmldesc, flags):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        try:
            return pool.createXML(xmldesc, flags)
        except Exception, e:
            raise KaresansuiVirtException("Could not create storage volume: %s" % str(e))

    def vol_numOfVolumes(self):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        return pool.numOfVolumes()
        
    def vol_storageVolLookupByName(self, name):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        return pool.storageVolLookupByName(name)
        
    def vol_listVolumes(self):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        return pool.listVolumes()

    def search_kvn_storage_volumes(self, conn):
        vols_obj = []
        for vol in self.vol_listVolumes():
            vol_obj = KaresansuiVirtStorageVolume(conn)
            vol_obj.set_storage_name(self.get_storage_name())
            vol_obj.set_storage_volume_name(vol)
            
            vols_obj.append(vol_obj)
        return vols_obj

class KaresansuiVirtStorageVolume(KaresansuiVirtStorage):

    storage_volume_name = None

    def set_storage_volume_name(self,name=None):
        self.storage_volume_name = name

    def get_storage_volume_name(self):
        return self.storage_volume_name

    def load(self):
        param = StorageVolumeConfigParam(self.get_storage_volume_name())
        param.load_xml_config("%s/%s.xml" \
                              % (VIRT_STORAGE_CONFIG_DIR, self.get_storage_volume_name()))

    def delete(self, flags):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        try:
            vol = pool.storageVolLookupByName(self.get_storage_volume_name())
            return vol.delete(flags)
        except Exception, e:
            raise KaresansuiVirtException("Could not delete storage volume: %s"
                                            % str(e))

    def get_json(self):
        """<comment-ja>
        JSON形式でKaresansuiVirtNetwork情報を返却する。
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        pass
        #ret = {}
        #ret['name'] = self.get_network_name()
        #ret.update(self.get_info())
        #return ret

    def get_info(self):
        pool = self._conn.storagePoolLookupByName(self.get_storage_name())
        try:
            vol = pool.storageVolLookupByName(self.get_storage_volume_name())
        except Exception, e:
            raise KaresansuiVirtException("Could not get the storage volume: %s"
                                            % str(e))

        param = StorageVolumeConfigParam(self.get_storage_name())
        param.load_xml_config(vol.XMLDesc(0))
        return {"name" : param.get_storage_name(),
                "uuid" : param.get_uuid(),
                "key" : param.get_key(),
                "allocation" : param.get_allocation(),
                "capacity" : param.get_capacity(),
                "c_unit" : param.get_c_unit(),
                "source" : param.get_source(),
                "target" : {"path" : param.get_target_path(),
                            "f_type" : param.get_target_f_type(),
                            "p_owner" : param.get_target_permissions_owner(),
                            "p_group" : param.get_target_permissions_group(),
                            "p_mode" : param.get_target_permissions_mode(),
                            "p_label" : param.get_target_permissions_label(),
                            "b_path" : param.get_backingStore_path(),
                            "b_format" : param.get_backingStore_format(),
                            "b_p_owner" : param.get_backingStore_permissions_owner(),
                            "b_p_group" : param.get_backingStore_permissions_group(),
                            "b_p_mode" : param.get_backingStore_permissions_mode(),
                            "b_p_label" : param.get_backingStore_permissions_label(),
                            },
                }
