# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

import pwd

import web
from web.utils import Storage

import karesansui
from karesansui.lib.rest import Rest, auth
from karesansui.lib.virt.virt import KaresansuiVirtConnection
from karesansui.lib.const import VIRT_COMMAND_CREATE_STORAGE_VOLUME, \
     VIRT_COMMAND_DELETE_STORAGE_VOLUME, STORAGE_VOLUME_SIZE_MIN_LENGTH, \
     STORAGE_VOLUME_SIZE_MAX_LENGTH, STORAGE_VOLUME_FORMAT, STORAGE_VOLUME_PWD

from karesansui.db.access.machine import findbyhost1
from karesansui.db.access._2pysilhouette import save_job_collaboration
from karesansui.db.access.machine2jobgroup import new as m2j_new
from karesansui.db.model._2pysilhouette import JobGroup, Job

from pysilhouette.command import dict2command

from karesansui.lib.storage import get_storage_volume_max_size_by_unit
from karesansui.lib.utils import is_param
from karesansui.lib.checker import Checker, CHECK_EMPTY, CHECK_ONLYSPACE, CHECK_VALID, \
    CHECK_MIN, CHECK_MAX


# validate
def validates_volume(obj, now_pools):
    checker = Checker()
    check = True

    _ = obj._
    checker.errors = []

    if is_param(obj.input, 'volume_pool_name'):
        check = checker.check_string(_('Storagepool Name'),
                                     obj.input.volume_pool_name,
                                     CHECK_EMPTY | CHECK_ONLYSPACE,
                                     None,
                                     ) and check
        if not obj.input.volume_pool_name in now_pools:
            check = False
            checker.add_error(_('%s "%s" is not exist.') % (_('Storagepool Name'), obj.input.volume_pool_name))

    else:
        check = False
        checker.add_error(_('"%s" is required.') %_('Storagepool Name'))


    if is_param(obj.input, 'volume_name'):
        check = checker.check_string(_('Storagepool Volume Name'),
                                     obj.input.volume_name,
                                     CHECK_EMPTY | CHECK_ONLYSPACE,
                                     None,
                                     ) and check

        if is_param(obj.input, 'volume_pool_name'):
            try:
                kvc = KaresansuiVirtConnection()
                if not kvc.get_storage_volume(obj.input.volume_pool_name, obj.input.volume_name) is None:
                    check = False
                    checker.add_error(_('%s "%s" is already exist in %s.') % (_('Storagepool Volume Name'), obj.input.volume_name, obj.input.volume_pool_name))
            finally:
                kvc.close()
    else:
        check = False
        checker.add_error(_('"%s" is required.') %_('Storagepool Volume Name'))


    if is_param(obj.input, 'volume_format'):
        if not obj.input.volume_format in STORAGE_VOLUME_FORMAT.values():
            check = False
            checker.add_error(_('%s "%s" is unknown format.') % (_('Storagepool Volume Format'), obj.input.volume_format))
    else:
        check = False
        checker.add_error(_('"%s" is required.') % _('Storagepool Volume Format'))


    if is_param(obj.input, 'volume_capacity'):
        if is_param(obj.input, 'volume_pool_name') and is_param(obj.input, 'volume_unit'):
            storage_volume_capacity_size_max = get_storage_volume_max_size_by_unit(obj.input.volume_pool_name, obj.input.volume_unit)
        else:
            storage_volume_capacity_size_max = STORAGE_VOLUME_SIZE_MAX_LENGTH

        check = checker.check_number(_('Storagepool Volume Capacity'),
                                     obj.input.volume_capacity,
                                     CHECK_VALID | CHECK_MIN | CHECK_MAX,
                                     STORAGE_VOLUME_SIZE_MIN_LENGTH,
                                     storage_volume_capacity_size_max,
                                     ) and check
    else:
        check = False
        checker.add_error(_('"%s" is required.') % _('Storagepool Volume Capacity'))

    if is_param(obj.input, 'volume_allocation'):
        if is_param(obj.input, 'volume_pool_name') and is_param(obj.input, 'volume_unit'):
            storage_volume_allocation_size_max = get_storage_volume_max_size_by_unit(obj.input.volume_pool_name, obj.input.volume_unit)
        else:
            storage_volume_allocation_size_max = STORAGE_VOLUME_SIZE_MAX_LENGTH

        check = checker.check_number(_('Storagepool Volume Allocation'),
                                     obj.input.volume_allocation,
                                     CHECK_VALID | CHECK_MIN | CHECK_MAX,
                                     STORAGE_VOLUME_SIZE_MIN_LENGTH,
                                     storage_volume_allocation_size_max,
                                     ) and check
    else:
        check = False
        checker.add_error(_('"%s" is required.') % _('Storagepool Volume Allocation'))


    if is_param(obj.input, 'volume_unit'):
        check = checker.check_string(_('Storagepool Volume Unit'),
                                     obj.input.volume_unit,
                                     CHECK_EMPTY | CHECK_ONLYSPACE,
                                     None,
                                     ) and check
    else:
        check = False
        checker.add_error(_('"%s" is required.') % _('Storagepool Volume Unit'))


    obj.view.alert = checker.errors
    return check

# create job
def create_volume_job(obj, machine, volume_name, pool_name, format_,
                      capacity, allocation, unit,
                      options={}, rollback_options={}):
    cmdname = u"Create Storage Volume"
    cmd = VIRT_COMMAND_CREATE_STORAGE_VOLUME

    options['name'] = volume_name
    options['pool_name'] = pool_name
    options['format'] = format_
    options['capacity'] = capacity
    options['allocation'] = allocation
    options['unit'] = unit
    options['permission_owner'] = pwd.getpwnam(STORAGE_VOLUME_PWD["OWNER"])[2]
    options['permission_group'] = pwd.getpwnam(STORAGE_VOLUME_PWD["GROUP"])[2]
    options['permission_mode'] = STORAGE_VOLUME_PWD["MODE"]

    _cmd = dict2command(
        "%s/%s" % (karesansui.config['application.bin.dir'], cmd), options)

    rollback_options["name"] = volume_name
    rollback_options["pool_name"] = pool_name
    rollback_cmd = dict2command(
        "%s/%s" % (karesansui.config['application.bin.dir'], VIRT_COMMAND_DELETE_STORAGE_VOLUME),
        rollback_options)

    _jobgroup = JobGroup(cmdname, karesansui.sheconf['env.uniqkey'])
    _job = Job('%s command' % cmdname, 0, _cmd)
    _job.rollback_command = rollback_cmd
    _jobgroup.jobs.append(_job)

    _machine2jobgroup = m2j_new(machine=machine,
                                jobgroup_id=-1,
                                uniq_key=karesansui.sheconf['env.uniqkey'],
                                created_user=obj.me,
                                modified_user=obj.me,
                                )

    save_job_collaboration(obj.orm,
                           obj.pysilhouette.orm,
                           _machine2jobgroup,
                           _jobgroup,
                           )
    return True


class HostBy1StoragePoolBy1StorageVolume(Rest):

    @auth
    def _GET(self, *param, **params):
        host_id = self.chk_hostby1(param)
        if host_id is None: return web.notfound()

        self.view.host_id = host_id

        try:
            kvc = KaresansuiVirtConnection()
            inactive_pool = kvc.list_inactive_storage_pool()
            active_pool = kvc.list_active_storage_pool()
            pools = inactive_pool + active_pool
            pools.sort()

            self.view.pools = pools
            pools_info = []
            for pool in pools:
                pool_obj = kvc.search_kvn_storage_pools(pool)[0]
                pools_info.append(pool_obj.get_info())
                if pool_obj.is_active() is True:
                    vols_obj = pool_obj.search_kvn_storage_volumes(kvc)
                    vols_info = []
                    for vol_obj in vols_obj:
                        vols_info.append(vol_obj.get_info())
        finally:
            kvc.close()

        self.view.pools_info = pools_info

        if self.is_mode_input() is True:
            # .input
            self.view.pool_types = (STORAGE_POOL_TYPE["TYPE_DIR"],
                                    STORAGE_POOL_TYPE["TYPE_ISCSI"])

            self.view.volume_formats = (STORAGE_VOLUME_FORMAT["TYPE_QCOW2"],
                                        STORAGE_VOLUME_FORMAT["TYPE_RAW"],
                                        STORAGE_VOLUME_FORMAT["TYPE_QCOW"],
                                        STORAGE_VOLUME_FORMAT["TYPE_COW"],
                                        STORAGE_VOLUME_FORMAT["TYPE_VMDK"],
                                        )
        return True

    @auth
    def _POST(self, *param, **params):
        host_id = self.chk_hostby1(param)
        if host_id is None: return web.notfound()
        self.view.host_id = host_id

        try:
            kvc = KaresansuiVirtConnection()
            inactive_pool = kvc.list_inactive_storage_pool()
            active_pool = kvc.list_active_storage_pool()
            pools = inactive_pool + active_pool
            pools.sort()
            self.view.pools = pools
            pools_obj = kvc.get_storage_pool_UUIDString2kvn_storage_pool(param[1])
            if len(pools_obj) <= 0:
                return web.notfound()
        finally:
            kvc.close()

        model = findbyhost1(self.orm, host_id)

        if not validates_volume(self, pools):
            return web.badrequest(self.view.alert)
        extra_opts = {}
        if create_volume_job(self,
                             model,
                             self.input.volume_name,
                             self.input.volume_pool_name,
                             self.input.volume_format,
                             self.input.volume_capacity,
                             self.input.volume_allocation,
                             self.input.volume_unit,
                             extra_opts) is True:
            #:TODO
            return web.accepted()
        else:
            #:TODO
            return False

urls = (
    '/host/(\d+)/storagepool/([a-z0-9]{8}-[a-z0-9]{4}-[a-z0-9]{4}-[a-z0-9]{4}-[a-z0-9]{12})/storagevolume/?(\.part)$', HostBy1StoragePoolBy1StorageVolume,
    )
