#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui Core.
#
# Copyright (C) 2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#

import sqlalchemy
from sqlalchemy.orm import mapper, clear_mappers, relation, backref
import karesansui
import karesansui.db.model
import karesansui.db.model.watch
from karesansui.lib.const import DEFAULT_LANGS

def get_watchdetail_table(metadata, now):
    """<comment-ja>
    監視Action(Watchdetail)のテーブル定義を返却します。
    @param metadata: MetaData
    @type metadata: sqlalchemy.schema.MetaData
    @param now: now
    @type now: Datatime
    @return: sqlalchemy.schema.Table
    </comment-ja>
    <comment-en>
    TODO: English Comment
    </comment-en>
    """
    return sqlalchemy.Table('watchdetail', metadata,
                            sqlalchemy.Column('id', sqlalchemy.Integer,
                                              primary_key=True,
                                              autoincrement=True,
                                              ),
                            sqlalchemy.Column('created_user_id', sqlalchemy.Integer,
                                              sqlalchemy.ForeignKey('user.id'),
                                              ),
                            sqlalchemy.Column('modified_user_id', sqlalchemy.Integer,
                                              sqlalchemy.ForeignKey('user.id'),
                                              ),
                            sqlalchemy.Column('watch_id', sqlalchemy.Integer,
                                              sqlalchemy.ForeignKey('watch.id'),
                                              ),
                            sqlalchemy.Column('shell', sqlalchemy.Text,
                                              nullable=True,
                                              ),
                            sqlalchemy.Column('value_selector', sqlalchemy.Text,
                                              nullable=False,
                                              ),
                            sqlalchemy.Column('created', sqlalchemy.DateTime,
                                              default=now,
                                              ),
                            sqlalchemy.Column('modified', sqlalchemy.DateTime,
                                              default=now,
                                              onupdate=now,
                                              ),
                            )

class Watchdetail(karesansui.db.model.Model):
    """<comment-ja>
    Watchdetailテーブルモデルクラス
    </comment-ja>
    <comment-en>
    TODO: English Comment
    </comment-en>
    """
    def __init__(self, created_user, modified_user,
                 watch, value_selector,
                 shell=None):
        """<comment-ja>
        @param created_user: 作成者
        @type created_user: User
        @param modified_user: 最終更新者
        @type modified_user: User
        @param watch:  
        @type watch: 
        @param value_selector: 
        @type value_selector: 
        @param shell: 
        @type shell: 
        </comment-ja>
        <comment-en>
        TODO: English Comment
        </comment-en>
        """
        self.created_user = created_user
        self.modified_user = modified_user
        self.watch = watch
        self.value_selector = value_selector
        self.shell = shell

    def get_json(self, languages):
        import web
        ret = {}
        ret["id"] = self.id
        ret["created_user"] = self.created_user.get_json(languages)
        ret["modified_user"] = self.modified_user.get_json(languages)
        ret["watch_id"] = self.watch_id
        ret["watch"] = self.watch.get_json(languages)
        ret["value_selector"] = self.value_selector
        ret["shell"] = self.shell
        ret["created"] = self.created.strftime(
            DEFAULT_LANGS[languages]['DATE_FORMAT'][1])
        ret["created_user_id"] = self.created_user_id
        ret["modified"] = self.modified.strftime(
            DEFAULT_LANGS[languages]['DATE_FORMAT'][1])
        ret["modified_user_id"] = self.modified_user_id
        return ret

    def __repr__(self):
        return "Watchdetail<'%s'>" % (str(self.id))

def reload_mapper(metadata, now):
    """<comment-ja>
    Watchdetail(Model)のマッパーをリロードします。
    @param metadata: リロードしたいMetaData
    @type metadata: sqlalchemy.schema.MetaData
    @param now: now
    @type now: Datatime
    </comment-ja>
    <comment-en>
    TODO: English Comment
    </comment-en>
    """
    t_watchdetail = get_watchdetail_table(metadata, now)
    t_user = metadata.tables['user']
    t_watch = metadata.tables['watch']
    
    mapper(Watchdetail, t_watchdetail, properties={
        'watch' : relation(karesansui.db.model.watch.Watch,
                           primaryjoin=t_watchdetail.c.watch_id==t_watch.c.id),
        'created_user' : relation(karesansui.db.model.user.User,
                                  primaryjoin=t_watchdetail.c.created_user_id==t_user.c.id),
        'modified_user' : relation(karesansui.db.model.user.User,
                                   primaryjoin=t_watchdetail.c.modified_user_id==t_user.c.id),
        })

if __name__ == '__main__':
    import sqlalchemy.orm
    bind_name = 'sqlite:///:memory:'
    engine = sqlalchemy.create_engine(bind_name,
                                      encoding="utf-8",
                                      convert_unicode=True,
                                      #assert_unicode='warn', # DEBUG
                                      echo=True,
                                      echo_pool=False
                                      )
    
    metadata = sqlalchemy.MetaData(bind=engine)

    # relation
    karesansui.db.model.watch.reload_mapper(metadata)
    karesansui.db.model.user.reload_mapper(metadata)

    reload_mapper(metadata)
    import pdb; pdb.set_trace()
    metadata.drop_all()
    metadata.create_all()
    Session = sqlalchemy.orm.sessionmaker(bind=engine, autoflush=False)
    session = Session()
    print ""
    # INSERT
    # SELECT One
    # UPDATE
    # DELETE

