#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2009-2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

import os
import sys
import logging
import re
from optparse import OptionParser

from ksscommand import KssCommand, KssCommandException, KssCommandOptException

import __cmd__

try:
    import karesansui
    from karesansui import __version__
    from karesansui.lib.virt.virt import KaresansuiVirtConnection
    from karesansui.lib.utils import load_locale
    from karesansui.lib.const import DEFAULT_KEYMAP

except ImportError:
    print >>sys.stderr, "[Error] karesansui package was not found."
    sys.exit(1)

_ = load_locale()

usage = '%prog [options]'

def getopts():
    optp = OptionParser(usage=usage, version=__version__)
    # basic
    optp.add_option('-n', '--name', dest='name', help=_('Domain name'))
    optp.add_option('-t', '--type', dest='type', help=_('VM Type'), default="xen")
    optp.add_option('-m', '--mem-size', dest='mem_size', help=_('Memory size (MB)'), default=256)
    optp.add_option('-d', '--disk', dest='disk', help=_('Disk image file'), default=None)
    optp.add_option('-v', '--vnc-port', dest='vnc_port', help=_('VNC port number'), default=None)
    optp.add_option('-u', '--uuid', dest='uuid', help=_('UUID'), default=None)
    optp.add_option('-a', '--mac', dest='mac', help=_('MAC address'), default=None)
    optp.add_option('-c', '--vcpus', dest='vcpus', help=_('Number of virtual CPUs to allocate'), default=1)
    optp.add_option('-f', '--interface-format', dest='interface_format', help=_('Interface format'), default='b:xenbr0')
    optp.add_option('-b', '--keymap', dest='keymap', help=_('VNC Keyboard Map'), default=DEFAULT_KEYMAP)
    optp.add_option('-e', '--extra', dest='extra', help=_('Extra kernel options'), default=None)
    # Storage pool only
    optp.add_option('-V', '--storage-volume', dest='storage_volume', help=_('Storage pool name'), default=None)
    optp.add_option('-P', '--storage-pool', dest='storage_pool', help=_('Storage volume name'), default=None)
    # make disk only
    optp.add_option('-D', '--disk-format', dest='disk_format', help=_('Disk format'), default=None)
    optp.add_option('-s', '--disk-size', dest='disk_size', help=_('Disk size (MB)'), default=1024*8)
    # raw only
    optp.add_option('-p', '--sparse', dest='sparse', action="store_true", help=_('Sparse file'))
    # IDE or SCSI or Virtio
    optp.add_option('-B', '--bus', dest='bus', help=_('Device bus type'), default=None)
    # ISO only
    optp.add_option('-o', '--iso', dest='iso', help=_('ISO image'), default=None)
    # Individual designation
    optp.add_option('-k', '--kernel', dest='kernel', help=_('Kernel image'), default=None)
    optp.add_option('-i', '--initrd', dest='initrd', help=_('initrd image'), default=None)

    return optp.parse_args()

def chkopts(opts):
    if not opts.name:
        raise KssCommandOptException('ERROR: -n or --name option is required.')

    if opts.disk and os.path.isfile(opts.disk) is False:
        raise KssCommandOptException('ERROR: %s not found.' % opts.disk)


    if opts.iso is not None:
        if opts.kernel is not None or opts.initrd is not None:
            raise KssCommandOptException('ERROR: %s option cannot be specified with %s options.' % ('--iso', '--kernel and --initrd',))

        if os.path.isfile(opts.iso) is False:
            raise KssCommandOptException('ERROR: The specified ISO image path does not exist. - %s' % opts.iso)

        from karesansui.lib.utils import is_iso9660_filesystem_format
        if is_iso9660_filesystem_format(opts.iso) is False:
            raise KssCommandOptException('ERROR: The specified ISO image is not valid ISO 9660 CD-ROM filesystem data. - %s' % opts.iso)

    else:
        _r_get_net = re.compile("^(ftp|http)://")

        if _r_get_net.match(opts.kernel) is None and os.path.isfile(opts.kernel) is False:
            raise KssCommandOptException('ERROR: The specified kernel image path does not exist. - %s' % opts.kernel)

        if _r_get_net.match(opts.initrd) is None and os.path.isfile(opts.initrd) is False:
            raise KssCommandOptException('ERROR: The specified initrd image path does not exist. - %s' % opts.initrd)

class CreateGuest(KssCommand):

    def process(self):
        (opts, args) = getopts()
        chkopts(opts)
        self.up_progress(10)

        conn = KaresansuiVirtConnection(readonly=False)
        try:
            conn.set_interface_format(opts.interface_format)
    
            active_guests = conn.list_active_guest()
            inactive_guests = conn.list_inactive_guest()

            self.up_progress(10)

            if opts.name in active_guests or opts.name in inactive_guests:
                raise KssCommandException('guest already exists. - dom=%s' % (opts.name))
            else:
                try:
                    self.up_progress(10)
                    conn.create_guest(opts.name,
                                      opts.type.lower(),
                                      opts.mem_size,
                                      opts.disk,
                                      opts.disk_size,
                                      opts.mac,
                                      opts.uuid,
                                      opts.kernel,
                                      opts.initrd,
                                      opts.iso,
                                      opts.vnc_port,
                                      opts.vcpus,
                                      opts.extra,
                                      opts.keymap,
                                      opts.sparse,
                                      opts.bus,
                                      opts.disk_format,
                                      opts.storage_pool,
                                      opts.storage_volume,
                                      )

                    self.up_progress(40)
                except Exception, e:
                    self.logger.error('Failed to create guest. - dom=%s - detail %s' % (opts.name, str(e.args)))
                    print >>sys.stderr, _('Failed to create guest. - dom=%s - detail %s') % (opts.name, str(e.args))
                    #raise KssCommandException(
                    #    'Failed to create guest. - dom=%s' % (opts.name))
                    raise

                self.up_progress(10)
                active_guests = conn.list_active_guest()
                inactive_guests = conn.list_inactive_guest()
                if opts.name in active_guests or opts.name in inactive_guests:
                    self.logger.info('created guest. - dom=%s' % (opts.name))
                    print >>sys.stderr, 'created guest. - dom=%s' % opts.name

            return True
        finally:
            conn.close()

if __name__ == "__main__":
    target = CreateGuest()
    sys.exit(target.run())
