#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

import os
import sys
import logging
import fcntl
from optparse import OptionParser

from ksscommand import KssCommand, KssCommandException, KssCommandOptException
import __cmd__

try:
    import karesansui
    from karesansui import __version__
    from karesansui.lib.utils import load_locale, execute_command
    from karesansui.lib.parser.iscsid import iscsidParser
    from karesansui.lib.dict_op import DictOp
    from karesansui.lib.iscsi import iscsi_parse_node, iscsi_print_format_node
    from karesansui.lib.const import ISCSI_CONFIG_KEY_AUTH_METHOD, ISCSI_CONFIG_KEY_AUTH_USER, \
        ISCSI_CONFIG_KEY_AUTH_PASSWORD, ISCSI_CONFIG_KEY_SATRTUP, ISCSI_CONFIG_VALUE_AUTH_METHOD_CHAP, \
        ISCSI_CONFIG_VALUE_AUTH_METHOD_NONE, ISCSI_CONFIG_VALUE_SATRTUP_ON, ISCSI_CONFIG_VALUE_SATRTUP_OFF, \
        ISCSI_CMD, ISCSI_CMD_OPTION_MODE, ISCSI_CMD_OPTION_MODE_DISCOVERY, ISCSI_CMD_OPTION_TYPE, \
        ISCSI_CMD_OPTION_TYPE_SENDTARGETS, ISCSI_CMD_OPTION_PORTAL

except ImportError:
    print >>sys.stderr, "[Error] karesansui package was not found."
    sys.exit(1)

_ = load_locale()

usage = '%prog [options]'

def getopts():
    optp = OptionParser(usage=usage, version=__version__)
    optp.add_option('-t', '--target', dest='host', help=_('Target host name'), default=None)
    optp.add_option('-a', '--auth', dest='auth', help=_('Authentication type'), default=None)
    optp.add_option('-u', '--user', dest='user', help=_('Authentication user name'), default=None)
    optp.add_option('-p', '--password', dest='password', help=_('Authentication password'), default=None)
    optp.add_option('-w', '--password-file', dest='password_file', help=_('Authentication password file'), default=None)
    optp.add_option('-s', '--autostart', dest='autostart', action="store_true", help=_('Autostart'), default=False)
    return optp.parse_args()

def chkopts(opts):
    if not opts.host:
        raise KssCommandOptException('ERROR: %s option is required.' % '-t or --target')

    if opts.auth:
        if not opts.auth == ISCSI_CONFIG_VALUE_AUTH_METHOD_CHAP and not opts.auth == ISCSI_CONFIG_VALUE_AUTH_METHOD_NONE:
            raise KssCommandOptException('ERROR: %s option is require %s or %s.' % '-a', ISCSI_CONFIG_VALUE_AUTH_METHOD_CHAP, ISCSI_CONFIG_VALUE_AUTH_METHOD_NONE)
        if opts.auth == ISCSI_CONFIG_VALUE_AUTH_METHOD_CHAP:
            if opts.user is None:
                raise KssCommandOptException('ERROR: %s option is required.' % '-u or --user')
            if opts.password is None and opts.password_file is None:
                raise KssCommandOptException('ERROR: %s option is required.' % '-p or --password or -w or --password-file')
            if opts.password_file is not None and not os.path.exists(opts.password_file):
                raise KssCommandOptException('ERROR: %s is not found.' % opts.password_file)

class AddIscsi(KssCommand):

    def process(self):
        (opts, args) = getopts()
        chkopts(opts)
        self.up_progress(10)

        original_parser = iscsidParser()
        new_parser = iscsidParser()
        dop = DictOp()

        dop.addconf("original", original_parser.read_conf())
        dop.addconf("new", new_parser.read_conf())

        self.up_progress(10)

        dop.cdp_set("new", ISCSI_CONFIG_KEY_AUTH_METHOD, opts.auth)
        if opts.auth == ISCSI_CONFIG_VALUE_AUTH_METHOD_CHAP:
            password = ""
            if opts.password is not None:
                password = opts.password
            elif opts.password_file is not None and os.path.exists(opts.password_file):
                try:
                    fp = open(opts.password_file, "r")
                    try:
                        self.up_progress(10)
                        fcntl.lockf(fp.fileno(), fcntl.LOCK_SH)
                        try:
                            password = fp.readline().strip("\n")
                        finally:
                            fcntl.lockf(fp.fileno(), fcntl.LOCK_UN)

                        self.up_progress(10)
                    finally:
                        fp.close()

                except Exception, e:
                    self.logger.error('Failed to read file. - target host=%s password_file=%s' \
                                      % (opts.host,opts.password_file))
                    print >>sys.stderr, _('Failed to read file. - target host=%s password_file=%s') \
                          % (opts.host, opts.password_file)
                    raise

                os.remove(opts.password_file)
                self.up_progress(10)


            dop.cdp_set("new", ISCSI_CONFIG_KEY_AUTH_METHOD, opts.auth)
            dop.cdp_set("new", ISCSI_CONFIG_KEY_AUTH_USER, opts.user)
            dop.cdp_set("new", ISCSI_CONFIG_KEY_AUTH_PASSWORD, password)
        else:
            dop.comment("new", ISCSI_CONFIG_KEY_AUTH_USER)
            dop.comment("new", ISCSI_CONFIG_KEY_AUTH_PASSWORD)

        if opts.autostart:
            dop.cdp_set("new", ISCSI_CONFIG_KEY_SATRTUP, ISCSI_CONFIG_VALUE_SATRTUP_ON)
        else:
            dop.cdp_set("new", ISCSI_CONFIG_KEY_SATRTUP, ISCSI_CONFIG_VALUE_SATRTUP_OFF)

        self.up_progress(10)
        new_parser.write_conf(dop.getconf("new"))
        self.up_progress(10)

        discovery_command_args = (ISCSI_CMD,
                                  ISCSI_CMD_OPTION_MODE,
                                  ISCSI_CMD_OPTION_MODE_DISCOVERY,
                                  ISCSI_CMD_OPTION_TYPE,
                                  ISCSI_CMD_OPTION_TYPE_SENDTARGETS,
                                  ISCSI_CMD_OPTION_PORTAL,
                                  opts.host
                                  )

        (discovery_rc,discovery_res) = execute_command(discovery_command_args)
        self.up_progress(10)

        original_parser.write_conf(dop.getconf("original"))
        self.up_progress(20)

        if discovery_rc != 0:
            raise KssCommandException('Failed to add iSCSI. - host=%s message=%s' % (opts.host, discovery_res))

        if discovery_res == []:
            raise KssCommandException('Failed to add iSCSI. - host=%s message=No exist permit iSCSI disk for target.' % (opts.host))

        for node_line in discovery_res:
            if not node_line:
                continue

            node = iscsi_parse_node(node_line)

            self.logger.info("%s" % (iscsi_print_format_node(node)))
            print >>sys.stdout, _("%s") % (iscsi_print_format_node(node))

        return True

if __name__ == "__main__":
    target = AddIscsi()
    sys.exit(target.run())
