/*
	x11-event.c
*/
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "x11-core.h"

static int quit_flag = 0;
static void (*idle_func)(void *data) = NULL;
static void *idle_data = NULL;
static void (*keyboard_func)(JtkEvent *event, void *data) = NULL;
static void *keyboard_data = NULL;
static JwkWindow *focus_window = NULL;

static void _x11_event(XEvent *event)
{
	JwkWindow *jw;
	JwkEvent je;

	if(XFilterEvent(event, None))
		return;

	if((jw = x11_wlist_find_window(event->xany.window)) == NULL)
		return;
			
	j_zero(&je, sizeof(JwkEvent));
	je.any.window = jw;
	je.any.widget = jw->widget;

	switch(event->type){
	case KeyPress:
		{
			int len;
			char buf[256];
			KeySym sym;
			Status st;
			
			j_zero(buf, sizeof(buf));
			len = XmbLookupString(jw->xic, &event->xkey, buf, 256, &sym, &st);
			
			if((st == XLookupChars) ||
				(st == XLookupBoth)){
				if((focus_window != NULL) &&
					(focus_window->callback != NULL) &&
					(focus_window->event_mask & JWK_EVENT_STRING)){
					je.type = JWK_EVENT_STRING;
					je.any.window = focus_window;
					je.any.widget = focus_window->widget;
					je.string.px = event->xkey.x;
					je.string.py = event->xkey.y;
					je.string.screen_px = event->xkey.x_root;
					je.string.screen_py = event->xkey.y_root;
					je.string.keymask = event->xkey.state;
					je.string.str = buf;
					je.string.len = len;
					if(len == 1){
						if((buf[0] >= 0x20) && (buf[0] <= 0x7e))
							focus_window->callback(&je);
					}else{
						focus_window->callback(&je);
					}
				}
			}
			if((st == XLookupKeySym) ||
				(st == XLookupBoth)){
				if(keyboard_func != NULL){
					JtkEvent je2;
					je2.type = JTK_EVENT_KEYDOWN;
					je2.keydown.widget = jw->widget;
					je2.keydown.data = jw->widget->data;
					je2.keydown.px = event->xkey.x;
					je2.keydown.py = event->xkey.y;
					je2.keydown.screen_px = event->xkey.x_root;
					je2.keydown.screen_py = event->xkey.y_root;
					je2.keydown.keymask = event->xkey.state;
					je2.keydown.keysym = sym;
					keyboard_func(&je2, keyboard_data);
				}
				if((focus_window != NULL) &&
					(focus_window->callback != NULL) &&
					(focus_window->event_mask & JWK_EVENTMASK_KEYDOWN)){
					je.type = JWK_EVENT_KEYDOWN;
					je.any.window = focus_window;
					je.any.widget = focus_window->widget;
					je.keydown.px = event->xkey.x;
					je.keydown.py = event->xkey.y;
					je.keydown.screen_px = event->xkey.x_root;
					je.keydown.screen_py = event->xkey.y_root;
					je.keydown.keymask = event->xkey.state;
					je.keydown.keysym = sym;
					focus_window->callback(&je);
				}
			}
		}
		break;
	case KeyRelease:
		if(!(jw->event_mask & JWK_EVENTMASK_KEYUP))
			break;
		je.type = JWK_EVENT_KEYUP;
		je.keyup.px = event->xkey.x;
		je.keyup.py = event->xkey.y;
		je.keyup.screen_px = event->xkey.x_root;
		je.keyup.screen_py = event->xkey.y_root;
		je.keyup.keymask = event->xkey.state;
		je.keyup.keysym = XKeycodeToKeysym(display, event->xkey.keycode, 0);
		
		if(keyboard_func != NULL){
			JtkEvent je2;
			je2.type = JTK_EVENT_KEYUP;
			je2.keyup.widget = jw->widget;
			je2.keyup.data = jw->widget->data;
			je2.keyup.px = je.keyup.px;
			je2.keyup.py = je.keyup.py;
			je2.keyup.screen_px = je.keyup.screen_px;
			je2.keyup.screen_py = je.keyup.screen_py;
			je2.keyup.keymask = je.keyup.keymask;
			je2.keyup.keysym = je.keyup.keysym;
			keyboard_func(&je2, keyboard_data);
		}
		
		if(jw->callback != NULL)
			jw->callback(&je);
		break;
	case ButtonPress:
		if((jw->event_mask & JWK_EVENTMASK_BUTTONDOWN) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_BUTTONDOWN;
			je.buttondown.px = event->xbutton.x;
			je.buttondown.py = event->xbutton.y;
			je.buttondown.screen_px = event->xbutton.x_root;
			je.buttondown.screen_py = event->xbutton.y_root;
			je.buttondown.keymask = event->xbutton.state;
			je.buttondown.button = event->xbutton.button;
			jw->callback(&je);
		}
		break;
	case ButtonRelease:
		if((jw->event_mask & JWK_EVENTMASK_BUTTONUP) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_BUTTONUP;
			je.buttonup.px = event->xbutton.x;
			je.buttonup.py = event->xbutton.y;
			je.buttonup.screen_px = event->xbutton.x_root;
			je.buttonup.screen_py = event->xbutton.y_root;
			je.buttonup.keymask = event->xbutton.state;
			je.buttonup.button = event->xbutton.button;
			jw->callback(&je);
		}
		break;
	case MotionNotify:
		if((jw->event_mask & JWK_EVENTMASK_MOTION) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_MOTION;
			je.motion.px = event->xmotion.x;
			je.motion.py = event->xmotion.y;
			je.motion.screen_px = event->xmotion.x_root;
			je.motion.screen_py = event->xmotion.y_root;
			je.motion.keymask = event->xmotion.state;
			jw->callback(&je);
		}
		break;
	case EnterNotify:
		if((jw->event_mask & JWK_EVENTMASK_ENTER) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_ENTER;
			je.enter.px = event->xcrossing.x;
			je.enter.py = event->xcrossing.y;
			je.enter.screen_px = event->xcrossing.x_root;
			je.enter.screen_py = event->xcrossing.y_root;
			je.enter.keymask = event->xcrossing.state;
			jw->callback(&je);
		}
		break;
	case LeaveNotify:
		if((jw->event_mask & JWK_EVENTMASK_LEAVE) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_LEAVE;
			je.leave.px = event->xcrossing.x;
			je.leave.py = event->xcrossing.y;
			je.leave.screen_px = event->xcrossing.x_root;
			je.leave.screen_py = event->xcrossing.y_root;
			je.leave.keymask = event->xcrossing.state;
			jw->callback(&je);
		}
		break;
	case FocusIn:
		if((jw->event_mask & JWK_EVENTMASK_FOCUSIN) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_FOCUSIN;
			jw->callback(&je);
		}
		break;
	case FocusOut:
		if((jw->event_mask & JWK_EVENTMASK_FOCUSOUT) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_FOCUSOUT;
			jw->callback(&je);
		}
		break;
	case MapNotify:
		if((jw->event_mask & JWK_EVENTMASK_MAP) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_MAP;
			jw->callback(&je);
		}
		break;
	case UnmapNotify:
		if((jw->event_mask & JWK_EVENTMASK_UNMAP) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_UNMAP;
			jw->callback(&je);
		}
		break;
	case ConfigureNotify:
		if((jw->event_mask & JWK_EVENTMASK_MOVE) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_MOVE;
			je.move.px = event->xconfigure.x;
			je.move.py = event->xconfigure.y;
			jw->callback(&je);
		}
		if((jw->event_mask & JWK_EVENTMASK_RESIZE) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_RESIZE;
			je.resize.width = event->xconfigure.width;
			je.resize.height = event->xconfigure.height;
			jw->callback(&je);
		}
		break;
	case Expose:
		if((jw->event_mask & JWK_EVENTMASK_EXPOSE) &&
			(jw->callback != NULL)){
			je.type = JWK_EVENT_EXPOSE;
			je.expose.x = event->xexpose.x;
			je.expose.y = event->xexpose.y;
			je.expose.width = event->xexpose.width;
			je.expose.height = event->xexpose.height;
			je.expose.count = event->xexpose.count;
			jw->callback(&je);
		}
		break;
	case ClientMessage:
		if(event->xclient.message_type == WM_PROTOCOLS){
			if(event->xclient.data.l[0] == WM_DELETE_WINDOW){
				je.type = JWK_EVENT_CLOSE;
				if(jw->callback != NULL)
					jw->callback(&je);
			}
		}
		break;
	default:
		break;
	}
}

/* Main event loop */
void jtkMain()
{
	XEvent event;

	quit_flag = 0;
	XFlush(display);
	while(!quit_flag){
		if(idle_func != NULL){
			if(XPending(display) > 0){
				XNextEvent(display, &event);
				_x11_event(&event);
			}else{
				idle_func(idle_data);
			}
		}else{
			XNextEvent(display, &event);
			_x11_event(&event);
		}
	}
}

void jtkMainQuit()
{
	quit_flag = 1;
}

void jtkSetIdleFunc(void (*idle)(void *data), void *data)
{
	idle_func = idle;
	idle_data = data;
}

void jtkSetKeyboardFunc(void (*callback)(JtkEvent *event, void *data), void *data)
{
	keyboard_func = callback;
	keyboard_data = data;
}

void jtkSleep(unsigned int ms)
{
	usleep(ms * 1000);
}

void jwkSetFocus(JwkWindow *jw)
{
	focus_window = jw;
}

JwkWindow* jwkGetFocus()
{
	return focus_window;
}
