#include "pthread-thread.h"

struct JtkThread
{
	void (*func)(void *data);
	void *data;
	pthread_t thread_id;
	
	JtkThread *next;
};

struct JtkSemaphore
{
	sem_t sem_id;
};

struct JtkMutex
{
	pthread_mutex_t mutex_id;
};

static JtkMutex *thread_list_mutex;
static JtkThread thread_list;
static JtkThread *thread_list_head;
static JtkThread *thread_list_foot;

static void thread_list_cleanup(void *arg)
{
	jtkUnlockMutex(thread_list_mutex);
}

void thread_list_init()
{
	j_zero(&thread_list, sizeof(JtkThread));
	thread_list_head = &thread_list;
	thread_list_foot = &thread_list;
	
	thread_list_mutex = jtkCreateMutex();
}

void thread_list_final()
{
	JtkThread *t1 = thread_list_head->next;
	JtkThread *t2;
	void *data;
	
	pthread_cleanup_push(thread_list_cleanup, NULL);
	jtkLockMutex(thread_list_mutex);
	while(t1 != NULL){
		t2 = t1;
		t1 = t1->next;
		pthread_cancel(t2->thread_id);
		pthread_join(t2->thread_id, &data);
		j_free(t2);
	}
	thread_list_head->next = NULL;
	thread_list_foot = &thread_list;
	jtkUnlockMutex(thread_list_mutex);
	pthread_cleanup_pop(0);
	
	jtkDestroyMutex(thread_list_mutex);
}

void thread_list_add(JtkThread *thread)
{
	pthread_cleanup_push(thread_list_cleanup, NULL);
	jtkLockMutex(thread_list_mutex);
	thread_list_foot->next = thread;
	thread_list_foot = thread;
	jtkUnlockMutex(thread_list_mutex);
	pthread_cleanup_pop(0);
}

void thread_list_remove(JtkThread *thread)
{
	JtkThread *t1 = thread_list_head->next;
	JtkThread *t2 = thread_list_head;
	
	pthread_cleanup_push(thread_list_cleanup, NULL);
	jtkLockMutex(thread_list_mutex);
	while(t1 != NULL){
		if(t1 == thread){
			t2->next = t1->next;
		}else{
			t2 = t1;
		}
		t1 = t1->next;
	}
	jtkUnlockMutex(thread_list_mutex);
	pthread_cleanup_pop(0);
}

void* thread_func(void *data)
{
	JtkThread *thread = data;
	if(thread->func != NULL)
		thread->func(thread->data);
	
	return NULL;
}

JtkThread* jtkCreateThread(void (*func)(void *data), void *data)
{
	JtkThread *thread;
	
	thread = j_malloc(sizeof(JtkThread));
	if(thread == NULL)
		return NULL;
	
	thread->func = func;
	thread->data = data;
	thread->next = NULL;
	
	if(pthread_create(&thread->thread_id, NULL, thread_func, thread) != 0){
		j_free(thread);
		return NULL;
	}
	
	thread_list_add(thread);
	
	return thread;
}

void jtkKillThread(JtkThread *thread)
{
	pthread_cancel(thread->thread_id);
	thread_list_remove(thread);
}

void jtkWaitThread(JtkThread *thread)
{
	void *data;
	pthread_join(thread->thread_id, &data);
	j_free(thread);
}

void jtkExitThread()
{
	pthread_exit(NULL);
}

JtkSemaphore* jtkCreateSemaphore(unsigned int value)
{
	JtkSemaphore *sem;
	
	sem = j_malloc(sizeof(JtkSemaphore));
	if(sem == NULL)
		return NULL;
	
	if(sem_init(&sem->sem_id, 0, value) != 0){
		j_free(sem);
		return NULL;
	}
	
	return sem;
}

JtkError jtkPostSemaphore(JtkSemaphore *sem)
{
	if(sem_post(&sem->sem_id) == 0)
		return JTK_NOERROR;
	
	return JTK_ERROR;
}

JtkError jtkWaitSemaphore(JtkSemaphore *sem)
{
	if(sem_wait(&sem->sem_id) == 0)
		return JTK_NOERROR;
	
	return JTK_ERROR;
}

JtkError jtkDestroySemaphore(JtkSemaphore *sem)
{
	if(sem_destroy(&sem->sem_id) != 0)
		return JTK_ERROR;
	
	j_free(sem);
	
	return JTK_NOERROR;
}

JtkMutex* jtkCreateMutex()
{
	JtkMutex *mutex;
	
	mutex = j_malloc(sizeof(JtkMutex));
	if(mutex == NULL)
		return NULL;
	
	if(pthread_mutex_init(&mutex->mutex_id, NULL) != 0){
		j_free(mutex);
		return NULL;
	}
	
	return mutex;
}

JtkError jtkLockMutex(JtkMutex *mutex)
{
	if(pthread_mutex_lock(&mutex->mutex_id) == 0)
		return JTK_NOERROR;
	
	return JTK_ERROR;
}

JtkError jtkUnlockMutex(JtkMutex *mutex)
{
	if(pthread_mutex_unlock(&mutex->mutex_id) == 0)
		return JTK_NOERROR;
	
	return JTK_ERROR;
}

JtkError jtkDestroyMutex(JtkMutex *mutex)
{
	if(pthread_mutex_destroy(&mutex->mutex_id) != 0)
		return JTK_ERROR;
	
	j_free(mutex);
	return JTK_NOERROR;
}
