/*
	linux-joystick.c
*/
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <linux/joystick.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "jtk.h"

struct JtkJoystick
{
	int id;
	int fd;
	char name[128];
	
	short axis_current[8];
	char button_current[32];
	short axis_buffer[8];
	char button_buffer[32];
	
	void (*callback)(JtkJoystickEvent *event);
	void *data;
	
	JtkThread *thread;
};

static void joystick_thread(void *arg)
{
	JtkJoystick *joystick;
	JtkJoystickEvent event;
	struct js_event je;
	
	joystick = arg;
	while(read(joystick->fd, &je, sizeof(struct js_event)) > 0){
		if(je.type & JS_EVENT_BUTTON){
			if(je.number <= 31){
				joystick->button_buffer[je.number] = je.value;
				if((joystick->callback != NULL) &&
					!(je.type & JS_EVENT_INIT)){
					event.type = JTK_JOYSTICK_EVENT_BUTTON;
					event.any.joystick = joystick;
					event.any.data = joystick->data;
					event.button.number = je.number;
					event.button.state = je.value;
					joystick->callback(&event);
				}
			}
		}else if(je.type & JS_EVENT_AXIS){
			if(je.number <= 7){
				joystick->axis_buffer[je.number] = je.value;
				if((joystick->callback != NULL) &&
					!(je.type & JS_EVENT_INIT)){
					event.type = JTK_JOYSTICK_EVENT_AXIS;
					event.any.joystick = joystick;
					event.any.data = joystick->data;
					event.axis.number = je.number;
					event.axis.value = je.value;
					joystick->callback(&event);
				}
			}
		}
	}
}

/* JTK Joystick API */
JtkJoystick* jtkOpenJoystick(int id)
{
	char device_file_name[256];
	JtkJoystick *joystick;
	
	if(id < 1)
		return NULL;
	
	joystick = malloc(sizeof(JtkJoystick));
	if(joystick == NULL)
		return NULL;
	memset(joystick, 0, sizeof(JtkJoystick));
	joystick->id = id;
	
	/* open joystick */
	sprintf(device_file_name, "/dev/js%d", id - 1);
	if((joystick->fd = open(device_file_name, O_RDONLY)) == -1){
		sprintf(device_file_name, "/dev/input/js%d", id - 1);
		if((joystick->fd = open(device_file_name, O_RDONLY)) == -1){
			free(joystick);
			return NULL;
		}
	}
	
	/* get joystick name */
	if(ioctl(joystick->fd, JSIOCGNAME(sizeof(joystick->name)), joystick->name) < 0)
		strncpy(joystick->name, "Unknown", sizeof(joystick->name));
	
	/* create thread */
	joystick->thread = jtkCreateThread(joystick_thread, joystick);
	if(joystick->thread == NULL){
		close(joystick->fd);
		free(joystick);
		return NULL;
	}
	
	return joystick;
}

void jtkCloseJoystick(JtkJoystick *joystick)
{
	jtkKillThread(joystick->thread);
	jtkWaitThread(joystick->thread);
	close(joystick->fd);
	free(joystick);
}

void jtkSetJoystickCallback(JtkJoystick *joystick, void *data,
								void (*callback)(JtkJoystickEvent *event))
{
	joystick->data = data;
	joystick->callback = callback;
}

void jtkUpdateJoystick(JtkJoystick *joystick)
{
	int i;
	
	for(i=0; i<8; i++)
		joystick->axis_current[i] = joystick->axis_buffer[i];
	
	for(i=0; i<32; i++)
		joystick->button_current[i] = joystick->button_current[i];
}

char* jtkGetJoystickName(JtkJoystick *joystick)
{
	return joystick->name;
}

int jtkGetJoystickAxisNum(JtkJoystick *joystick)
{
	char num_axis;
	
	ioctl(joystick->fd, JSIOCGAXES, &num_axis);
	return num_axis;
}

int jtkGetJoystickButtonNum(JtkJoystick *joystick)
{
	char num_button;
	
	ioctl(joystick->fd, JSIOCGBUTTONS, &num_button);
	return num_button;
}

short jtkGetJoystickAxis(JtkJoystick *joystick, int axis)
{
	if((axis >= 0) && (axis <= 7))
		return joystick->axis_current[axis];
	
	return 0;
}

int jtkGetJoystickButton(JtkJoystick *joystick, int button)
{
	if((button >= 0) && (button <= 31))
		return joystick->button_current[button];

	return 0;
}
