#include "jwk.h"

typedef struct TextEntry TextEntry;
struct TextEntry
{
	JwkWindow *entry;
	JtkGC *gc;
	JtkColor text_color;
	JtkColor cursor_color;
	int entry_pos;
	char *text;
};

static void entry_callback(JwkEvent *event)
{
	JtkWidget *widget = event->any.widget;
	TextEntry *entry = widget->widget;
	
	switch(event->type){
	case JWK_EVENT_KEYDOWN:
		{
			int key;
			JtkEvent je;
			
			je.type = JTK_EVENT_KEYDOWN;
			je.keydown.widget = widget;
			je.keydown.data = widget->data;
			je.keydown.px = event->keydown.px;
			je.keydown.py = event->keydown.py;
			je.keydown.screen_px = event->keydown.screen_px;
			je.keydown.screen_py = event->keydown.screen_py;
			je.keydown.keymask = event->keydown.keymask;
			je.keydown.keysym = event->keydown.keysym;
			if(widget->callback != NULL)
				widget->callback(&je);
			
			key = event->keydown.keysym; 
			if(key == JVK_BackSpace){
				entry->entry_pos--;
				if(entry->entry_pos < 0)
					entry->entry_pos = 0;
				entry->text[entry->entry_pos] = '\0';
			}
			jwkClearWindow(entry->entry, 0, 0, 0, 0, JTK_TRUE);
		}
		break;
	case JWK_EVENT_BUTTONDOWN:
		jtkSetFocus(widget);
		break;
	case JWK_EVENT_STRING:
		{
			int i;
			
			for(i = 0; i < event->string.len; i++){
				entry->text[entry->entry_pos] = event->string.str[i];
				entry->entry_pos++;
			}
			jwkClearWindow(entry->entry, 0, 0, 0, 0, JTK_TRUE);
		}
		break;
	case JWK_EVENT_EXPOSE:
		{
			JtkSize size1, size2;
			char *test_text = "abcあいう";
			
			jtkSetColor(entry->gc, entry->text_color);
			jtkDrawText(entry->gc, 5, 5, entry->text, j_strlen(entry->text));
			jtkSetColor(entry->gc, entry->cursor_color);
			jtkGetTextSize(entry->gc, entry->text, j_strlen(entry->text), &size1);
			jtkGetTextSize(entry->gc, test_text, j_strlen(test_text), &size2);
			jtkFillRect(entry->gc, 7 + size1.width, 5, 3, size2.height);
		}
		break;
	default:
		break;
	}
}

static void destroy_entry(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	jtkDestroyGC(entry->gc);
	jwkDestroyWindow(entry->entry);
	j_free(entry->text);
	j_free(entry);
}

static void map_entry(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	jwkMapWindow(entry->entry);
}

static void unmap_entry(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	jwkUnmapWindow(entry->entry);
}

static void clear_entry(JtkWidget *widget,
	int px, int py, int width, int height, JtkBool exposures)
{
	TextEntry *entry = widget->widget;
	
	jwkClearWindow(entry->entry,
		px, py, width, height, exposures);
}

static void raise_entry(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	jwkRaiseWindow(entry->entry);
}

static void lower_entry(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	jwkLowerWindow(entry->entry);
}

static void set_entry_pos(JtkWidget *widget, int px, int py)
{
	TextEntry *entry = widget->widget;
	
	jwkSetWindowPos(entry->entry, px, py);
}

static void set_entry_size(JtkWidget *widget, int width, int height)
{
	TextEntry *entry = widget->widget;
	
	jwkSetWindowSize(entry->entry, width, height);
}

static void get_entry_pos(JtkWidget *widget, JtkPoint *point)
{
	TextEntry *entry = widget->widget;
	
	jwkGetWindowPos(entry->entry, point);
}

static void get_entry_size(JtkWidget *widget, JtkSize *size)
{
	TextEntry *entry = widget->widget;
	
	jwkGetWindowSize(entry->entry, size);
}

static void set_entry_color(JtkWidget *widget, JtkColor color)
{
	TextEntry *entry = widget->widget;
	
	jwkSetWindowColor(entry->entry, color);
}

static void set_entry_border_size(JtkWidget *widget, int size)
{
	TextEntry *entry = widget->widget;
	
	jwkSetWindowBorderSize(entry->entry, size);
}

static void set_entry_border_color(JtkWidget *widget, JtkColor color)
{
	TextEntry *entry = widget->widget;
	
	jwkSetWindowBorderColor(entry->entry, color);
}

static void set_entry_text_color(JtkWidget *widget, JtkColor color)
{
	TextEntry *entry = widget->widget;
	
	entry->text_color = color;
}

static void set_entry_text(JtkWidget *widget, char *text)
{
	TextEntry *entry = widget->widget;
	
	j_zero(entry->text, 1024);
	j_strcpy(entry->text, text);
	entry->entry_pos = j_strlen(text);
	jwkClearWindow(entry->entry, 0, 0, 0, 0, JTK_TRUE);
}

static char* get_entry_text(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	return entry->text;
}

static void clear_entry_text(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	j_zero(entry->text, 1024);
	entry->entry_pos = 0;
	jwkClearWindow(entry->entry, 0, 0, 0, 0, JTK_TRUE);
}

static void set_entry_focus(JtkWidget *widget)
{
	TextEntry *entry = widget->widget;
	
	jwkSetFocus(entry->entry);
}

static JwkWidgetFunc entry_func =
{
	.destroy = destroy_entry,
	.map = map_entry,
	.unmap = unmap_entry,
	.clear = clear_entry,
	.raise = raise_entry,
	.lower = lower_entry,
	.set_pos = set_entry_pos,
	.set_size = set_entry_size,
	.get_pos = get_entry_pos,
	.get_size = get_entry_size,
	.set_color = set_entry_color,
	.set_border_size = set_entry_border_size,
	.set_border_color = set_entry_border_color,
	.set_text_color = set_entry_text_color,
	.set_text = set_entry_text,
	.get_text = get_entry_text,
	.clear_text = clear_entry_text,
	.set_focus = set_entry_focus,
};

JtkWidget* jtkCreateTextEntry(JtkWidget *parent)
{
	JtkWidget *widget;
	TextEntry *entry;
	
	if(parent == NULL)
		return NULL;
	
	widget = j_malloc(sizeof(JtkWidget));
	if(widget == NULL)
		return NULL;
	
	entry = j_malloc(sizeof(TextEntry));
	if(entry == NULL){
		j_free(widget);
		return NULL;
	}
	
	j_zero(widget, sizeof(JtkWidget));
	j_zero(entry, sizeof(TextEntry));
	
	entry->text = j_malloc(1024);
	if(entry->text == NULL){
		j_free(widget);
		j_free(entry);
		return NULL;
	}
	
	j_zero(entry->text, 1024);
	
	entry->entry = jwkCreateWindow(jwkGetContainer(parent));
	entry->gc = jwkCreateGC(entry->entry, JTK_GCMODE_SCREEN, JTK_GCTYPE_NORMAL);
	jwkSetWindowEventMask(entry->entry,
		JWK_EVENTMASK_KEYDOWN |
		JWK_EVENTMASK_BUTTONDOWN |
		JWK_EVENTMASK_EXPOSE |
		JWK_EVENTMASK_STRING);
	jwkSetWidget(entry->entry, entry_callback, widget);
	
	widget->widget = entry;
	widget->func = &entry_func;
	
	return widget;
}
