/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */
package jp.sourceforge.jpnvalidator.util.validator;

import jp.sourceforge.jpnvalidator.ValidatorResult;
import jp.sourceforge.jpnvalidator.constant.FixedValue;

/**
 * JapaneseStringValidateUtil<br>
 * <br>
 * Simple module that checks Japanese String value.<br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */
public class JapaneseStringValidateUtil extends AbstractStringValidateUtil
{

	private JapaneseStringValidateUtil()
	{
	}

	/*
	 * -------------------------------------------------------------------------
	 * JISKanjiLevel1(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check JIS Kanji Level 1(Japanese) <br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isZenkakuJisKanjiLevel1OnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isZenkakuJisKanjiLevel1Only(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isZenkakuJisKanjiLevel1Only(String str,
			ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** JIS Kanji Level 1(Japanese) specific ***<br>
	 * <br>
	 * 1st byte : 0x88, 2nd byte : 9F-FC<br>
	 * 1st byte : 0x89, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x90, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x91, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x92, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x93, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x94, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x95, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x96, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x97, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x98, 2nd byte : 40-72<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isZenkakuJisKanjiLevel1Only(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (_isDoubleByte(eachStrbytes))
		{
			int firstByte = _getHexDigit(eachStrbytes[0]);
			int secondByte = _getHexDigit(eachStrbytes[1]);
			if (!(firstByte == 0x88 && (secondByte >= 0x9F && secondByte <= 0xFC))
					&& !((firstByte >= 0x89 && firstByte <= 0x97) && ((secondByte >= 0x40 && secondByte <= 0x7E) || (secondByte >= 0x80 && secondByte <= 0xFC)))
					&& !(firstByte == 0x98 && (secondByte >= 0x40 && secondByte <= 0x72)))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/*
	 * -------------------------------------------------------------------------
	 * JISKanjiLevel2(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check JIS Kanji Level 2(Japanese)<br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isZenkakuJisKanjiLevel2OnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isZenkakuJisKanjiLevel2Only(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isZenkakuJisKanjiLevel2Only(String str,
			ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** JIS Kanji Level 2(Japanese) specific ***<br>
	 * <br>
	 * 1st byte : 0x98, 2nd byte : 9F-FC<br>
	 * 1st byte : 0x99, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x9A, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x9B, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x9C, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x9D, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x9E, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0x9F, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE0, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE1, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE2, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE3, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE4, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE5, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE6, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE7, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE8, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xE9, 2nd byte : 40-7E,80-FC<br>
	 * 1st byte : 0xEA, 2nd byte : 40-A4<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isZenkakuJisKanjiLevel2Only(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (_isDoubleByte(eachStrbytes))
		{
			int firstByte = _getHexDigit(eachStrbytes[0]);
			int secondByte = _getHexDigit(eachStrbytes[1]);
			if (!(firstByte == 0x98 && (secondByte >= 0x9F && secondByte <= 0xFC))
					&& !(((firstByte >= 0x99 && firstByte <= 0x9F) || (firstByte >= 0xE0 && firstByte <= 0xE9)) && ((secondByte >= 0x40 && secondByte <= 0x7E) || (secondByte >= 0x80 && secondByte <= 0xFC)))
					&& !(firstByte == 0xEA && (secondByte >= 0x40 && secondByte <= 0xA4)))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/*
	 * -------------------------------------------------------------------------
	 * DoubleByte Hiragana(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check DoubleByte Hiragana(Japanese) <br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isZenkakuHiraganaOnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isZenkakuHiraganaOnly(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isZenkakuHiraganaOnly(String str, ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** DoubleByte Hiragana(Japanese) specific ***<br>
	 * <br>
	 * 1st byte : 0x82, 2nd byte : 9F-F1<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isZenkakuHiraganaOnly(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (_isDoubleByte(eachStrbytes))
		{
			int firstByte = _getHexDigit(eachStrbytes[0]);
			int secondByte = _getHexDigit(eachStrbytes[1]);
			if (!(firstByte == 0x82 && (secondByte >= 0x9F && secondByte <= 0xF1)))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/*
	 * -------------------------------------------------------------------------
	 * DoubleByte Katakana(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check DoubleByte Katakana(Japanese) <br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isZenkakuKatakanaOnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isZenkakuKatakanaOnly(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isZenkakuKatakanaOnly(String str, ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** DoubleByte Katakana(Japanese) specific ***<br>
	 * <br>
	 * 1st byte : 0x83, 2nd byte : 40-7E<br>
	 * 1st byte : 0x83, 2nd byte : 80-96<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isZenkakuKatakanaOnly(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (_isDoubleByte(eachStrbytes))
		{
			int firstByte = _getHexDigit(eachStrbytes[0]);
			int secondByte = _getHexDigit(eachStrbytes[1]);
			if (!(firstByte == 0x83 && ((secondByte >= 0x40 && secondByte <= 0x7E) || (secondByte >= 0x80 && secondByte <= 0x96))))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/*
	 * -------------------------------------------------------------------------
	 * DoubleByte Alphabet(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check DoubleByte Alphabet(Japanese) <br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isZenkakuAlphabetOnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isZenkakuAlphabetOnly(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isZenkakuAlphabetOnly(String str, ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** DoubleByte Alphabet(Japanese) specific ***<br>
	 * <br>
	 * 1st byte : 0x82, 2nd byte : 60-79<br>
	 * 1st byte : 0x82, 2nd byte : 81-9A<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isZenkakuAlphabetOnly(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (_isDoubleByte(eachStrbytes))
		{
			int firstByte = _getHexDigit(eachStrbytes[0]);
			int secondByte = _getHexDigit(eachStrbytes[1]);
			if (!(firstByte == 0x82 && ((secondByte >= 0x60 && secondByte <= 0x79) || (secondByte >= 0x81 && secondByte <= 0x9A))))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/*
	 * -------------------------------------------------------------------------
	 * DoubleByte Number(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check DoubleByte Number(Japanese) <br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isZenkakuNumberOnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isZenkakuNumberOnly(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isZenkakuNumberOnly(String str, ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** DoubleByte Number(Japanese) specific ***<br>
	 * <br>
	 * 1st byte : 0x82, 2nd byte : 4F-58<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isZenkakuNumberOnly(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (_isDoubleByte(eachStrbytes))
		{
			int firstByte = _getHexDigit(eachStrbytes[0]);
			int secondByte = _getHexDigit(eachStrbytes[1]);
			if (!(firstByte == 0x82 && (secondByte >= 0x4F && secondByte <= 0x58)))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/*
	 * -------------------------------------------------------------------------
	 * DoubleByte Symbol(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check DoubleByte Symbol(Japanese) <br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isZenkakuSymbolOnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isZenkakuSymbolOnly(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isZenkakuSymbolOnly(String str, ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** DoubleByte Alphabet(Japanese) specific ***<br>
	 * <br>
	 * <br>
	 * 、 。 ， ． ・ ： ； ？ ！ ゛ ゜ ´ ｀ ¨ ＾ <br>
	 * <br>
	 * ￣ ＿ ヽ ヾ ゝ ゞ 〃 仝 々 〆 〇 ー ― ‐ ／ ￥<br>
	 * <br>
	 * ～ ∥ ｜ … ‥ ‘ ’ “ ” （ ） 〔 〕 ［ ］ ｛<br>
	 * <br>
	 * ｝ 〈 〉 《 》 「 」 『 』 【 】 ＋ － ± ×<br>
	 * <br>
	 * ÷ ＝ ≠ ＜ ＞ ≦ ≧ ∞ ∴ ♂ ♀ ° ′ ″ ℃ ￥<br>
	 * <br>
	 * ＄ ￠ ￡ ％ ＃ ＆ ＊ ＠ § ☆ ★ ○ ● ◎ ◇ ◆<br>
	 * □ ■ △ ▲ ▽ ▼ ※ 〒 → ← ↑ ↓ 〓<br>
	 * <br>
	 * ∈ ∋ ⊆ ⊇ ⊂ ⊃ ∪ ∩<br>
	 * <br>
	 * ∧ ∨ ￢ ⇒ ⇔ ∀ ∃<br>
	 * <br>
	 * ∠ ⊥ ⌒ ∂ ∇ ≡<br>
	 * <br>
	 * ≒ ≪ ≫ √ ∽ ∝ ∵ ∫ ∬ Å ‰ ♯ ♭ ♪ † ‡ ¶ ◯<br>
	 * Α Β Γ Δ Ε Ζ Η Θ Ι Κ Λ Μ Ν Ξ Ο Π Ρ Σ Τ Υ Φ Χ Ψ Ω α<br>
	 * β γ δ ε ζ η θ ι κ λ μ ν ξ ο π ρ σ τ υ φ χ ψ ω<br>
	 * А Б В Г Д Е Ё Ж З И Й К Л М Н О<br>
	 * П Р С Т У Ф Х Ц Ч Ш Щ Ъ Ы Ь Э Ю Я<br>
	 * а б в г д е ё ж з и й к л м н<br>
	 * о п р с т у ф х ц ч ш щ ъ ы ь э<br>
	 * ю я ─<br>
	 * │ ┌ ┐ ┘ └ ├ ┬ ┤ ┴ ┼ ━ ┃ ┏ ┓ ┛ ┗<br>
	 * <br>
	 * ┣ ┳ ┫ ┻ ╋ ┠ ┯ ┨ ┷ ┿ ┝ ┰ ┥ ┸ ╂<br>
	 * <br>
	 * ① ② ③ ④ ⑤ ⑥ ⑦ ⑧ ⑨ ⑩ ⑪ ⑫ ⑬ ⑭ ⑮ ⑯<br>
	 * <br>
	 * ⑰ ⑱ ⑲ ⑳ Ⅰ Ⅱ Ⅲ Ⅳ Ⅴ Ⅵ Ⅶ Ⅷ Ⅸ Ⅹ ・ ㍉<br>
	 * <br>
	 * ㌔ ㌢ ㍍ ㌘ ㌧ ㌃ ㌶ ㍑ ㍗ ㌍ ㌦ ㌣ ㌫ ㍊ ㌻ ㎜<br>
	 * <br>
	 * ㎝ ㎞ ㎎ ㎏ ㏄ ㎡ 〝 〟 № ㏍ ℡ ㊤ ㊥ ㊦ ㊧ ㊨ ㈱ ㈲ ㈹ ㍾ ㍽ ㍼<br>
	 * <br>
	 * ≒ ≡ ∫ ∮ ∑ √ ⊥ ∠ ∟ ⊿ ∵ ∩ ∪<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isZenkakuSymbolOnly(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (_isDoubleByte(eachStrbytes))
		{
			int firstByte = _getHexDigit(eachStrbytes[0]);
			int secondByte = _getHexDigit(eachStrbytes[1]);
			if (!(firstByte == 0x81 && ((secondByte >= 0x40 && secondByte <= 0x7E)
					|| (secondByte >= 0x80 && secondByte <= 0xAC)
					|| (secondByte >= 0xB8 && secondByte <= 0xBF)
					|| (secondByte >= 0xC8 && secondByte <= 0xCE)
					|| (secondByte >= 0xDA && secondByte <= 0xDF)
					|| (secondByte >= 0xE0 && secondByte <= 0xE8)
					|| (secondByte >= 0xF0 && secondByte <= 0xF7) || secondByte >= 0xFC))
					&& !(firstByte == 0x83 && ((secondByte >= 0x9F && secondByte <= 0xB6) || (secondByte >= 0xBF && secondByte <= 0xD6)))
					&& !(firstByte == 0x84 && ((secondByte >= 0x40 && secondByte <= 0x60)
							|| (secondByte >= 0x70 && secondByte <= 0x7E)
							|| (secondByte >= 0x80 && secondByte <= 0x91) || (secondByte >= 0x9F && secondByte <= 0xBE)))
					&& !(firstByte == 0x87 && ((secondByte >= 0x40 && secondByte <= 0x75)
							|| secondByte >= 0x7E || (secondByte >= 0x80 && secondByte <= 0x9C))))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/*
	 * -------------------------------------------------------------------------
	 * OneByte Katakana(Japanese)
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check OneByte Katakana(Japanese) <br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isHankakuKatakanaOnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isHankakuKatakanaOnly(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isHankakuKatakanaOnly(String str, ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** OneByte Katakana(Japanese) specific ***<br>
	 * <br>
	 * 0xA0-0xDF<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isHankakuKatakanaOnly(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (eachStrbytes.length == 1)
		{
			// one byte katakana or not
			boolean isOneByteKatakana = false;
			for (int j = 0; j < FixedValue.ONE_BYTE_KATAKANA_ALL.length(); j++)
			{
				String listEachChar = FixedValue.ONE_BYTE_KATAKANA_ALL
						.substring(j, j + 1);
				if (listEachChar.equals(each))
				{
					isOneByteKatakana = true;
					break;
				}
			}
			if (!isOneByteKatakana)
			{
				return false;
			}
			int eachDigit = _getHexDigit(eachStrbytes[0]);
			if (eachDigit != FixedValue.CARRIAGE_RETURN_BYTE
					&& eachDigit != FixedValue.LINE_FEED_BYTE
					&& !(eachDigit >= 0xA0 && eachDigit <= 0xDF))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

}