/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.jpnvalidator.util.validator;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

import jp.sourceforge.jpnvalidator.ValidatorResult;
import jp.sourceforge.jpnvalidator.constant.Encoding;
import jp.sourceforge.jpnvalidator.constant.FixedValue;
import jp.sourceforge.jpnvalidator.util.file.SimpleFileReader;

/**
 * BaseStringValidateUtil<br>
 * <br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

abstract class AbstractStringValidateUtil
{

	/**
	 * Simple File Reader
	 */
	protected static SimpleFileReader reader = new SimpleFileReader();

	/**
	 * Get byte array encoded by CP932
	 * 
	 * @param str
	 *            target String value
	 * @return encoded byte array
	 */
	protected static byte[] _getMS932ByteArray(String str)
	{
		byte[] retByteArr = null;
		try
		{
			retByteArr = str.getBytes(Encoding.MS932);
		} catch (UnsupportedEncodingException ignore)
		{
		}
		return retByteArr;
	}

	/**
	 * Convert byte to hex digit
	 * 
	 * @param singleByte
	 *            target byte
	 * @return hex digit
	 */
	protected static int _getHexDigit(byte singleByte)
	{
		if (Integer.toHexString(singleByte).length() <= 2)
		{
			return singleByte;
		} else
		{
			char hexDigit1[] = Integer.toHexString(singleByte).substring(6, 8)
					.toCharArray();
			int hexValue = (Character.digit(hexDigit1[0], 16)) * 16
					+ (Character.digit(hexDigit1[1], 16));
			return hexValue;
		}
	}

	/**
	 * Check arg byte array means double-byte char
	 * 
	 * @param eachStrbytes
	 *            target byte array
	 * @return check result
	 */
	protected static boolean _isDoubleByte(byte[] eachStrbytes)
	{
		if (eachStrbytes == null)
		{
			return false;
		}
		return (eachStrbytes.length > 1
				&& Integer.toHexString(eachStrbytes[0]).length() == 8 && (Integer
				.toHexString(eachStrbytes[1]).length() == 8 || Integer.toHexString(
				eachStrbytes[1]).length() == 2)) ? true : false;
	}

	/**
	 * Get private execute method<br>
	 * 
	 * @return executable Method object
	 */
	protected static Method _getExecuteMethod()
	{
		return _getExecuteMethod(FixedValue.EMPTY_STRING, FixedValue.EMPTY_STRING);
	}

	/**
	 * Invoke executable Method
	 * 
	 * @param method
	 *            executable Method object
	 * @param args
	 *            executable Method args
	 * @return executable Method return value
	 */
	protected static Object _invokeStaticMethod(Method method, Object... args)
	{
		Object retObj = null;
		try
		{
			retObj = method.invoke(null, args);
		} catch (IllegalArgumentException e)
		{
			// TODO
			e.printStackTrace();
		} catch (IllegalAccessException e)
		{
			// TODO
			e.printStackTrace();
		} catch (InvocationTargetException e)
		{
			// TODO
			e.printStackTrace();
		}
		return retObj;
	}

	/**
	 * Get private execute method<br>
	 * 
	 * @param prefix
	 *            method name prefix
	 * @return executable Method object
	 */
	protected static Method _getExecuteMethod(String prefix)
	{
		return _getExecuteMethod(prefix, FixedValue.EMPTY_STRING);
	}

	/**
	 * 
	 * Get private execute method<br>
	 * 
	 * @param prefix
	 *            method name prefix
	 * @param suffix
	 *            method name suffix
	 * @return executable Method object
	 */
	protected static Method _getExecuteMethod(String prefix, String suffix)
	{
		Method executeMethod = null;
		try
		{
			executeMethod = JapaneseStringValidateUtil.class.getDeclaredMethod(prefix
					+ new Throwable().getStackTrace()[2].getMethodName() + suffix,
					String.class);
		} catch (SecurityException e)
		{
			e.printStackTrace();
			throw new RuntimeException(e);
		} catch (NoSuchMethodException e)
		{
			e.printStackTrace();
			throw new RuntimeException(e);
		}
		return executeMethod;
	}

	/**
	 * Check all characters of arg String value<br>
	 * 
	 * @param str
	 *            check target String value
	 * @param executeMethod
	 *            executable Method object
	 * @return check result
	 */
	protected static boolean _checkAllCharsFast(String str, Method executeMethod)
	{
		if (str == null || str.length() == 0)
		{
			return false;
		}
		// check each character
		int strLen = str.length();
		for (int i = 0; i < strLen; i++)
		{
			String each = str.substring(i, i + 1);
			try
			{
				executeMethod.setAccessible(true);
				if (!(Boolean) executeMethod.invoke(null, each))
				{
					return false;
				}
			} catch (IllegalArgumentException e)
			{
				e.printStackTrace();
				break;
			} catch (IllegalAccessException e)
			{
				e.printStackTrace();
				break;
			} catch (InvocationTargetException e)
			{
				e.printStackTrace();
				break;
			}
		}
		return true;
	}

	/**
	 * Check all characters of arg String value<br>
	 * 
	 * @param str
	 *            check target String value
	 * @param executeMethod
	 *            executable Method object
	 * @return check result
	 */
	protected static ValidatorResult _checkAllChars(String str, Method executeMethod,
			ValidatorResult result)
	{
		if (str == null || str.length() == 0)
		{
			return null;
		}
		ValidatorResult valResult = (result == null) ? new ValidatorResult() : result;
		// check each character
		int strLen = str.length();
		for (int i = 0; i < strLen; i++)
		{
			String each = str.substring(i, i + 1);
			if (_isRegistered(valResult.getValidCharList(), each))
				continue;
			try
			{
				executeMethod.setAccessible(true);
				// check execute
				if (!(Boolean) executeMethod.invoke(null, each))
				{
					if (!_isRegistered(valResult.getInvalidCharList(), each))
						valResult.getInvalidCharList().add(each);
				} else
				{
					if (!_isRegistered(valResult.getValidCharList(), each))
					{
						valResult.getValidCharList().add(each);
					}
					valResult.getInvalidCharList().remove(each);
				}
			} catch (IllegalArgumentException e)
			{
				e.printStackTrace();
				break;
			} catch (IllegalAccessException e)
			{
				e.printStackTrace();
				break;
			} catch (InvocationTargetException e)
			{
				e.printStackTrace();
				break;
			}
		}
		if (valResult.getInvalidCharList().size() > 0)
		{
			valResult.setValue(false);
		} else
		{
			valResult.setValue(true);
		}
		return valResult;
	}

	protected static byte[] _readFile(File file) throws IOException
	{
		return reader.read2ByteArray(file);
	}

	protected static boolean _isRegistered(List<String> list, String str)
	{
		for (String each : list)
		{
			if (each.equals(str))
				return true;
		}
		return false;
	}

	/*
	 * -------------------------------------------------------------------------
	 * ASCII
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Check ASCII<br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isAsciiOnlyFast(String str)
	{
		return _checkAllCharsFast(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING));
	}

	public static ValidatorResult isAsciiOnly(String str)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING), null);
	}

	public static ValidatorResult isAsciiOnly(String str, ValidatorResult result)
	{
		return _checkAllChars(str, _getExecuteMethod(FixedValue.UNDER_SCORE_STRING),
				result);
	}

	/**
	 * *** ASCII specific ***<br>
	 * <br>
	 * 0x00～0x7f<br>
	 * 
	 * @param each
	 *            each String value
	 * @return check result
	 */
	@SuppressWarnings("unused")
	private static boolean _isAsciiOnly(String each)
	{
		byte[] eachStrbytes = _getMS932ByteArray(each);
		if (eachStrbytes != null && eachStrbytes.length == 1)
		{
			byte eachByte = eachStrbytes[0];
			// ASCII or out of CP932
			boolean isASCIIChar = false;
			for (int j = 0; j < FixedValue.ASCII_CODE_STRING_ALL.length(); j++)
			{
				String listEachChar = FixedValue.ASCII_CODE_STRING_ALL
						.substring(j, j + 1);
				if (listEachChar.equals(each))
				{
					isASCIIChar = true;
					break;
				}
			}
			if (!isASCIIChar)
			{
				return false;
			}
			if (!(eachByte >= 0x00 && eachByte <= 0x7f))
			{
				return false;
			}
		} else
		{
			return false;
		}
		return true;
	}

	/**
	 * Check Carriage Return<br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isCrOnlyFast(String str)
	{
		// TODO
		return false;
	}

	public static ValidatorResult isCrOnly(String str)
	{
		// TODO
		return null;
	}

	public static ValidatorResult isCrOnly(String str, ValidatorResult result)
	{
		// TODO
		return result;
	}

	/**
	 * Check Line Feed<br>
	 * <br>
	 * Returns true in the following cases<br>
	 * (1) arg String value is null or empty String value.<br>
	 * 
	 * @param str
	 *            target String value
	 * @return check result
	 */
	public static boolean isLfOnlyFast(String str)
	{
		// TODO
		return false;
	}

	public static ValidatorResult isLfOnly(String str)
	{
		// TODO
		return null;
	}

	public static ValidatorResult isLfOnly(String str, ValidatorResult result)
	{
		// TODO
		return result;
	}

	public static boolean isSpaceOnlyFast(String str)
	{
		return false;
	}

	public static ValidatorResult isSpaceOnly(String str)
	{
		return null;
	}

	public static ValidatorResult isSpaceOnly(String str, ValidatorResult result)
	{
		return result;
	}

	public static boolean isSpaceOrBreakOnlyFast(String str)
	{
		return false;
	}

	public static ValidatorResult isSpaceOrBreakOnly(String str)
	{
		return null;
	}

	public static ValidatorResult isSpaceOrBreakOnly(String str, ValidatorResult result)
	{
		return result;
	}

}