/* 
 * Copyright 2009 Kazuhiro Sera. 
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language 
 * governing permissions and limitations under the License. 
 */

package jp.sourceforge.jpnvalidator.util.file;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.jpnvalidator.constant.Encoding;

/**
 * SimpleFileReader<br>
 * <br>
 * 
 * @author Kazuhiro Sera
 * @version 1.0
 */

public class SimpleFileReader
{
	/*
	 * -------------------------------------------------------------------------
	 * Instance Field
	 * -------------------------------------------------------------------------
	 */

	/**
	 * encoding to read file content
	 */
	private String encodingToRead = Encoding.MS932;

	/**
	 * default array length
	 */
	private Integer defaultArrayLength = 128;

	/*
	 * -------------------------------------------------------------------------
	 * char array
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Read file content(convert to char array)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public char[] read2CharArray(String resourcePathOrFilePath) throws IOException
	{
		InputStream is = null;
		InputStreamReader isr = null;
		try
		{
			is = this.getClass().getClassLoader().getResourceAsStream(
					resourcePathOrFilePath);
			if (is != null)
			{
				// resource on class path
				isr = new InputStreamReader(is, this.encodingToRead);
			} else
			{
				// file system
				isr = new InputStreamReader(new FileInputStream(resourcePathOrFilePath),
						this.encodingToRead);
			}
			// substantially doubled if over default array length
			return _getFileContentInCharArray(isr, null, this.defaultArrayLength);
		} finally
		{
			FileResourceUtil.close(is);
			FileResourceUtil.close(isr);
		}
	}

	/**
	 * 
	 * Read file content(convert to char array)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public char[] read2CharArray(File file) throws IOException
	{
		InputStreamReader isr = null;
		try
		{
			// file system
			isr = new InputStreamReader(new FileInputStream(file), this.encodingToRead);
			// substantially doubled if over default array length
			return _getFileContentInCharArray(isr, null, this.defaultArrayLength);
		} finally
		{
			FileResourceUtil.close(isr);
		}
	}

	/*
	 * -------------------------------------------------------------------------
	 * byte array
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Read file content(convert to byte array)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public byte[] read2ByteArray(String resourcePathOrFilePath) throws IOException
	{
		InputStream is = null;
		try
		{
			is = this.getClass().getClassLoader().getResourceAsStream(
					resourcePathOrFilePath);
			if (is == null)
			{
				// file system
				is = new FileInputStream(resourcePathOrFilePath);
			}
			// substantially doubled if over default array length
			byte[] retArr = _getFileContentInByteArray(is, null, this.defaultArrayLength);
			// retArr = new String(retArr).getBytes(this.encodingToRead);
			return retArr;
		} finally
		{
			FileResourceUtil.close(is);
		}
	}

	/**
	 * Read file content(convert to byte array)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public byte[] read2ByteArray(File file) throws IOException
	{
		BufferedInputStream bis = null;
		byte[] retByteArr = null;
		try
		{
			bis = new BufferedInputStream(new FileInputStream(file));
			retByteArr = new byte[(int) file.length()];
			bis.read(retByteArr);
		} finally
		{
			FileResourceUtil.close(bis);
		}
		return retByteArr.toString().getBytes(this.encodingToRead);
	}

	/*
	 * -------------------------------------------------------------------------
	 * String List
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Read file content(convert to String List)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public List<String> read2LineList(String resourcePathOrFilePath) throws IOException
	{
		List<String> retList = new ArrayList<String>();
		InputStream is = null;
		InputStreamReader isr = null;
		BufferedReader br = null;
		try
		{
			is = this.getClass().getClassLoader().getResourceAsStream(
					resourcePathOrFilePath);
			if (is == null)
			{
				// file system
				is = new FileInputStream(resourcePathOrFilePath);
			}
			isr = new InputStreamReader(is, this.encodingToRead);
			br = new BufferedReader(isr);
			while (retList.add(br.readLine()))
			{
			}
		} finally
		{
			FileResourceUtil.close(is);
			FileResourceUtil.close(isr);
			FileResourceUtil.close(br);
		}
		return retList;
	}

	/**
	 * 
	 * Read file content(convert to String List)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public List<String> read2LineList(File file) throws IOException
	{
		List<String> retList = new ArrayList<String>();
		InputStream is = null;
		InputStreamReader isr = null;
		BufferedReader br = null;
		try
		{
			// file system
			is = new FileInputStream(file);
			isr = new InputStreamReader(is, this.encodingToRead);
			br = new BufferedReader(isr);
			while (retList.add(br.readLine()))
			{
			}
		} finally
		{
			FileResourceUtil.close(is);
			FileResourceUtil.close(isr);
			FileResourceUtil.close(br);
		}
		return retList;
	}

	/*
	 * -------------------------------------------------------------------------
	 * String array
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Read file content(convert to String array)
	 * 
	 * @param resourcePathOrFilePath
	 *            readable file resource of file path
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public String[] read2LineArray(String resourcePathOrFilePath) throws IOException
	{
		List<String> strList = this.read2LineList(resourcePathOrFilePath);
		String[] retArr = new String[strList.size()];
		int retArrLen = retArr.length;
		for (int i = 0; i < retArrLen; i++)
		{
			retArr[i] = strList.get(i);
		}
		return retArr;
	}

	/**
	 * 
	 * Read file content(convert to String array)
	 * 
	 * @param file
	 *            readable file object
	 * @return file content
	 * @throws IOException
	 *             file read error
	 */
	public String[] read2LineArray(File file) throws IOException
	{
		List<String> strList = this.read2LineList(file);
		String[] retArr = new String[strList.size()];
		int retArrLen = retArr.length;
		for (int i = 0; i < retArrLen; i++)
		{
			retArr[i] = strList.get(i);
		}
		return retArr;
	}

	/*
	 * -------------------------------------------------------------------------
	 * Private Implementation
	 * -------------------------------------------------------------------------
	 */

	/**
	 * Get byte array(file content) in necessary and sufficient array length
	 * 
	 * @param isr
	 *            InputStreamReader obj
	 * @param lastArr
	 *            arg char array from invoker(recursive execute)
	 * @return return char array
	 * @throws IOException
	 *             file read error
	 */
	private byte[] _getFileContentInByteArray(InputStream is, byte[] lastArr,
			int newArrLen) throws IOException
	{
		byte[] byteArr = new byte[newArrLen];
		is.read(byteArr);
		Integer resizeArrLen = null;
		int startIdx = 0;
		if (lastArr != null)
		{
			startIdx = lastArr.length;
		}
		for (int i = startIdx; i < newArrLen; i++)
		{
			if (byteArr[i] == 0)
			{
				resizeArrLen = i;
				break;
			}
		}
		if (resizeArrLen == null)
		{
			// recursive execute
			return _getFileContentInByteArray(is, byteArr, newArrLen * 2);
		} else
		{
			byte[] resized = new byte[resizeArrLen];
			for (int i = 0; i < resizeArrLen; i++)
			{
				resized[i] = byteArr[i];
			}
			return resized;
		}
	}

	/**
	 * Get char array(file content) in necessary and sufficient array length
	 * 
	 * @param isr
	 *            InputStreamReader obj
	 * @param lastArr
	 *            arg char array from invoker(recursive execute)
	 * @return return char array
	 * @throws IOException
	 *             file read error
	 */
	private char[] _getFileContentInCharArray(InputStreamReader isr, char[] lastArr,
			int arrLen) throws IOException
	{
		char[] charArr = new char[arrLen];
		isr.read(charArr);
		Integer resizeArrLen = null;
		int charArrLen = charArr.length;
		for (int i = 0; i < charArrLen; i++)
		{
			if (charArr[i] == 0)
			{
				resizeArrLen = i;
				break;
			}
		}
		if (resizeArrLen == null)
		{
			// recursive execute
			return _getFileContentInCharArray(isr, charArr, arrLen * 2);
		} else
		{
			char[] resized = new char[resizeArrLen];
			for (int i = 0; i < resizeArrLen; i++)
			{
				resized[i] = charArr[i];
			}
			return resized;
		}
	}

	/*
	 * -------------------------------------------------------------------------
	 * Getter/Setter
	 * -------------------------------------------------------------------------
	 */

	public String getEncodingToRead()
	{
		return encodingToRead;
	}

	public void setEncodingToRead(String encodingToRead)
	{
		this.encodingToRead = encodingToRead;
	}

	public Integer getDefaultArrayLength()
	{
		return defaultArrayLength;
	}

	public void setDefaultArrayLength(Integer defaultArrayLength)
	{
		this.defaultArrayLength = defaultArrayLength;
	}
}
