
# Examples (and some documentation) for the Words in the Library


```python
from notebook_preamble import J, V
```

# Stack Chatter
This is what I like to call the functions that just rearrange things on the stack.  (One thing I want to mention is that during a hypothetical compilation phase these "stack chatter" words effectively disappear, because we can map the logical stack locations to registers that remain static for the duration of the computation.  This remains to be done but it's "off the shelf" technology.)

### `clear`


```python
J('1 2 3 clear')
```

    


### `dup` `dupd`


```python
J('1 2 3 dup')
```

    1 2 3 3



```python
J('1 2 3 dupd')
```

    1 2 2 3


### `enstacken` `disenstacken` `stack` `unstack`
(I may have these paired up wrong.  I.e. `disenstacken` should be `unstack` and vice versa.)


```python
J('1 2 3 enstacken') # Replace the stack with a quote of itself.
```

    [3 2 1]



```python
J('4 5 6 [3 2 1] disenstacken')  # Unpack a list onto the stack.
```

    4 5 6 3 2 1



```python
J('1 2 3 stack')  # Get the stack on the stack.
```

    1 2 3 [3 2 1]



```python
J('1 2 3 [4 5 6] unstack')  # Replace the stack with the list on top.
                            # The items appear reversed but they are not,
                            # 4 is on the top of both the list and the stack.
```

    6 5 4


### `pop` `popd` `popop`


```python
J('1 2 3 pop')
```

    1 2



```python
J('1 2 3 popd')
```

    1 3



```python
J('1 2 3 popop')
```

    1


### `roll<` `rolldown` `roll>` `rollup`
The "down" and "up" refer to the movement of two of the top three items (displacing the third.)


```python
J('1 2 3 roll<')
```

    2 3 1



```python
J('1 2 3 roll>')
```

    3 1 2


### `swap`


```python
J('1 2 3 swap')
```

    1 3 2


### `tuck` `over`


```python
J('1 2 3 tuck')
```

    1 3 2 3



```python
J('1 2 3 over')
```

    1 2 3 2


### `unit` `quoted` `unquoted`


```python
J('1 2 3 unit')
```

    1 2 [3]



```python
J('1 2 3 quoted')
```

    1 [2] 3



```python
J('1 [2] 3 unquoted')
```

    1 2 3



```python
V('1 [dup] 3 unquoted')  # Unquoting evaluates.  Be aware.
```

                  . 1 [dup] 3 unquoted
                1 . [dup] 3 unquoted
          1 [dup] . 3 unquoted
        1 [dup] 3 . unquoted
        1 [dup] 3 . [i] dip
    1 [dup] 3 [i] . dip
          1 [dup] . i 3
                1 . dup 3
              1 1 . 3
            1 1 3 . 


# List words

### `concat` `swoncat` `shunt`


```python
J('[1 2 3] [4 5 6] concat')
```

    [1 2 3 4 5 6]



```python
J('[1 2 3] [4 5 6] swoncat')
```


    ---------------------------------------------------------------------------

    KeyError                                  Traceback (most recent call last)

    <ipython-input-22-15579491b615> in <module>()
    ----> 1 J('[1 2 3] [4 5 6] swoncat')
    

    /home/sforman/Desktop/ArchLayer/System/source/Thun/docs/notebook_preamble.py in J(text, stack, dictionary)
         30 
         31 def J(text, stack=S, dictionary=D):
    ---> 32     print stack_to_string(run(text, stack, dictionary)[0])
         33 
         34 


    /home/sforman/Desktop/ArchLayer/System/source/Thun/venv/local/lib/python2.7/site-packages/joy/joy.pyc in run(text, stack, dictionary, viewer)
         77 	'''
         78         expression = text_to_expression(text)
    ---> 79         return joy(stack, expression, dictionary, viewer)
         80 
         81 


    /home/sforman/Desktop/ArchLayer/System/source/Thun/venv/local/lib/python2.7/site-packages/joy/joy.pyc in joy(stack, expression, dictionary, viewer)
         56                 term, expression = expression
         57                 if isinstance(term, Symbol):
    ---> 58                         term = dictionary[term]
         59                         stack, expression, dictionary = term(stack, expression, dictionary)
         60                 else:


    KeyError: swoncat



```python
J('[1 2 3] [4 5 6] shunt')
```

### `cons` `swons` `uncons`


```python
J('1 [2 3] cons')
```


```python
J('[2 3] 1 swons')
```


```python
J('[1 2 3] uncons')
```

### `first` `second` `third` `rest`


```python
J('[1 2 3 4] first')
```


```python
J('[1 2 3 4] second')
```


```python
J('[1 2 3 4] third')
```


```python
J('[1 2 3 4] rest')
```

### `flatten`


```python
J('[[1] [2 [3] 4] [5 6]] flatten')
```

### `getitem` `at` `of` `drop` `take`

`at` and `getitem` are the same function. `of == swap at`


```python
J('[10 11 12 13 14] 2 getitem')
```


```python
J('[1 2 3 4] 0 at')
```


```python
J('2 [1 2 3 4] of')
```


```python
J('[1 2 3 4] 2 drop')
```


```python
J('[1 2 3 4] 2 take')  # reverses the order
```

`reverse` could be defines as `reverse == dup size take`

### `remove`


```python
J('[1 2 3 1 4] 1 remove')
```

### `reverse`


```python
J('[1 2 3 4] reverse')
```

### `size`


```python
J('[1 1 1 1] size')
```

### `swaack`
"Swap stack" swap the list on the top of the stack for the stack, and put the old stack on top of the new one.  Think of it as a context switch.  Niether of the lists/stacks change their order.


```python
J('1 2 3 [4 5 6] swaack')
```

### `choice` `select`


```python
J('23 9 1 choice')
```


```python
J('23 9 0 choice')
```


```python
J('[23 9 7] 1 select')  # select is basically getitem, should retire it?
```


```python
J('[23 9 7] 0 select')
```

### `zip`


```python
J('[1 2 3] [6 5 4] zip')
```


```python
J('[1 2 3] [6 5 4] zip [sum] map')
```

# Math words

### `+` `add`


```python
J('23 9 +')
```

### `-` `sub`


```python
J('23 9 -')
```

### `*` `mul`


```python
J('23 9 *')
```

### `/` `div` `floordiv` `truediv`


```python
J('23 9 /')
```


```python
J('23 -9 truediv')
```


```python
J('23 9 div')
```


```python
J('23 9 floordiv')
```


```python
J('23 -9 div')
```


```python
J('23 -9 floordiv')
```

### `%` `mod` `modulus` `rem` `remainder`


```python
J('23 9 %')
```

### `neg`


```python
J('23 neg -5 neg')
```

### pow


```python
J('2 10 pow')
```

### `sqr` `sqrt`


```python
J('23 sqr')
```


```python
J('23 sqrt')
```

### `++` `succ` `--` `pred`


```python
J('1 ++')
```


```python
J('1 --')
```

### `<<` `lshift` `>>` `rshift`


```python
J('8 1 <<')
```


```python
J('8 1 >>')
```

### `average`


```python
J('[1 2 3 5] average')
```

### `range` `range_to_zero` `down_to_zero`


```python
J('5 range')
```


```python
J('5 range_to_zero')
```


```python
J('5 down_to_zero')
```

### `product`


```python
J('[1 2 3 5] product')
```

### `sum`


```python
J('[1 2 3 5] sum')
```

### `min`


```python
J('[1 2 3 5] min')
```

### `gcd`


```python
J('45 30 gcd')
```

### `least_fraction`
If we represent fractions as a quoted pair of integers [q d] this word reduces them to their ... least common factors or whatever.


```python
J('[45 30] least_fraction')
```


```python
J('[23 12] least_fraction')
```

# Logic and Comparison

### `?` `truthy`
Get the Boolean value of the item on the top of the stack.


```python
J('23 truthy')
```


```python
J('[] truthy')  # Python semantics.
```


```python
J('0 truthy')
```

    ? == dup truthy


```python
V('23 ?')
```


```python
J('[] ?')
```


```python
J('0 ?')
```

### `&` `and` 


```python
J('23 9 &')
```

### `!=` `<>` `ne`


```python
J('23 9 !=')
```

The usual suspects:
- `<` `lt`
- `<=` `le`  
- `=` `eq`
- `>` `gt`
- `>=` `ge`
- `not`
- `or`

### `^` `xor`


```python
J('1 1 ^')
```


```python
J('1 0 ^')
```

# Miscellaneous

### `help`


```python
J('[help] help')
```

### `parse`


```python
J('[parse] help')
```


```python
J('1 "2 [3] dup" parse')
```

### `run`
Evaluate a quoted Joy sequence.


```python
J('[1 2 dup + +] run')
```

# Combinators

### `app1` `app2` `app3`


```python
J('[app1] help')
```


```python
J('10 4 [sqr *] app1')
```


```python
J('10 3 4 [sqr *] app2')
```


```python
J('[app2] help')
```


```python
J('10 2 3 4 [sqr *] app3')
```

### `anamorphism`
Given an initial value, a predicate function `[P]`, and a generator function `[G]`, the `anamorphism` combinator creates a sequence.

       n [P] [G] anamorphism
    ---------------------------
              [...]

Example, `range`:

    range == [0 <=] [1 - dup] anamorphism


```python
J('3 [0 <=] [1 - dup] anamorphism')
```

### `branch`


```python
J('3 4 1 [+] [*] branch')
```


```python
J('3 4 0 [+] [*] branch')
```

### `cleave`
    ... x [P] [Q] cleave

From the original Joy docs: "The cleave combinator expects two quotations, and below that an item `x`
It first executes `[P]`, with `x` on top, and saves the top result element.
Then it executes `[Q]`, again with `x`, and saves the top result.
Finally it restores the stack to what it was below `x` and pushes the two
results P(X) and Q(X)."

Note that `P` and `Q` can use items from the stack freely, since the stack (below `x`) is restored.  `cleave` is a kind of *parallel* primitive, and it would make sense to create a version that uses, e.g. Python threads or something, to actually run `P` and `Q` concurrently.  The current implementation of `cleave` is a definition in terms of `app2`:

    cleave == [i] app2 [popd] dip


```python
J('10 2 [+] [-] cleave')
```

### `dip` `dipd` `dipdd`


```python
J('1 2 3 4 5 [+] dip')
```


```python
J('1 2 3 4 5 [+] dipd')
```


```python
J('1 2 3 4 5 [+] dipdd')
```

### `dupdip`
Expects a quoted program `[Q]` on the stack and some item under it, `dup` the item and `dip` the quoted program under it.

    n [Q] dupdip == n Q n


```python
V('23 [++] dupdip *')  # N(N + 1)
```

### `genrec` `primrec`


```python
J('[genrec] help')
```


```python
J('3 [1 <=] [] [dup --] [i *] genrec')
```

### `i`


```python
V('1 2 3 [+ +] i')
```

### `ifte`
    [predicate] [then] [else] ifte


```python
J('1 2 [1] [+] [*] ifte')
```


```python
J('1 2 [0] [+] [*] ifte')
```

### `infra`


```python
V('1 2 3 [4 5 6] [* +] infra')
```

### `loop`


```python
J('[loop] help')
```


```python
V('3 dup [1 - dup] loop')
```

### `map` `pam`


```python
J('10 [1 2 3] [*] map')
```


```python
J('10 5 [[*][/][+][-]] pam')
```

### `nullary` `unary` `binary` `ternary`
Run a quoted program enforcing [arity](https://en.wikipedia.org/wiki/Arity).


```python
J('1 2 3 4 5 [+] nullary')
```


```python
J('1 2 3 4 5 [+] unary')
```


```python
J('1 2 3 4 5 [+] binary')  # + has arity 2 so this is technically pointless...
```


```python
J('1 2 3 4 5 [+] ternary')
```

### `step`


```python
J('[step] help')
```


```python
V('0 [1 2 3] [+] step')
```

### `times`


```python
V('3 2 1 2 [+] times')
```

### `b`


```python
J('[b] help')
```


```python
V('1 2 [3] [4] b')
```

### `while`
    [predicate] [body] while


```python
J('3 [0 >] [dup --] while')
```

### `x`


```python
J('[x] help')
```


```python
V('1 [2] [i 3] x')  # Kind of a pointless example.
```

# `void`
Implements [**Laws of Form** *arithmetic*](https://en.wikipedia.org/wiki/Laws_of_Form#The_primary_arithmetic_.28Chapter_4.29) over quote-only datastructures (that is, datastructures that consist soley of containers, without strings or numbers or anything else.)


```python
J('[] void')
```


```python
J('[[]] void')
```


```python
J('[[][[]]] void')
```


```python
J('[[[]][[][]]] void')
```
