
# Joypy

## Joy in Python

This implementation is meant as a tool for exploring the programming model and method of Joy.  Python seems like a great implementation language for Joy for several reasons.

We can lean on the Python immutable types for our basic semantics and types: ints, floats, strings, and tuples, which enforces functional purity.  We get garbage collection for free.  Compilation via Cython.  Glue language with loads of libraries.

### [Read-Eval-Print Loop (REPL)](https://en.wikipedia.org/wiki/Read%E2%80%93eval%E2%80%93print_loop)
The main way to interact with the Joy interpreter is through a simple [REPL](https://en.wikipedia.org/wiki/Read%E2%80%93eval%E2%80%93print_loop) that you start by running the package:

    $ python -m joy
    Joypy - Copyright © 2017 Simon Forman
    This program comes with ABSOLUTELY NO WARRANTY; for details type "warranty".
    This is free software, and you are welcome to redistribute it
    under certain conditions; type "sharing" for details.
    Type "words" to see a list of all words, and "[<name>] help" to print the
    docs for a word.


     <-top

    joy? _

The `<-top` marker points to the top of the (initially empty) stack.  You can enter Joy notation at the prompt and a [trace of evaluation](#The-TracePrinter.) will be printed followed by the stack and prompt again:

    joy? 23 sqr 18 +
           . 23 sqr 18 +
        23 . sqr 18 +
        23 . dup mul 18 +
     23 23 . mul 18 +
       529 . 18 +
    529 18 . +
       547 . 

    547 <-top

    joy? 


# Stacks (aka list, quote, sequence, etc.)

In Joy, in addition to the types Boolean, integer, float, and string, there is a single sequence type represented by enclosing a sequence of terms in brackets `[...]`.  This sequence type is used to represent both the stack and the expression.  It is a [cons list](https://en.wikipedia.org/wiki/Cons#Lists) made from Python tuples.


```python
import inspect
import joy.utils.stack


print inspect.getdoc(joy.utils.stack)
```

    When talking about Joy we use the terms "stack", "quote", "sequence",
    "list", and others to mean the same thing: a simple linear datatype that
    permits certain operations such as iterating and pushing and popping
    values from (at least) one end.
    
    There is no "Stack" Python class, instead we use the  `cons list`_, a 
    venerable two-tuple recursive sequence datastructure, where the
    empty tuple ``()`` is the empty stack and ``(head, rest)`` gives the recursive
    form of a stack with one or more items on it::
    
        stack := () | (item, stack)
    
    Putting some numbers onto a stack::
    
        ()
        (1, ())
        (2, (1, ()))
        (3, (2, (1, ())))
        ...
    
    Python has very nice "tuple packing and unpacking" in its syntax which
    means we can directly "unpack" the expected arguments to a Joy function.
    
    For example::
    
            def dup((head, tail)):
                    return head, (head, tail)
    
    We replace the argument "stack" by the expected structure of the stack,
    in this case "(head, tail)", and Python takes care of unpacking the
    incoming tuple and assigning values to the names.  (Note that Python
    syntax doesn't require parentheses around tuples used in expressions
    where they would be redundant.)
    
    Unfortunately, the Sphinx documentation generator, which is used to generate this
    web page, doesn't handle tuples in the function parameters.  And in Python 3, this
    syntax was removed entirely.  Instead you would have to write::
    
            def dup(stack):
                    head, tail = stack
                    return head, (head, tail)
    
    
    We have two very simple functions, one to build up a stack from a Python
    iterable and another to iterate through a stack and yield its items
    one-by-one in order.  There are also two functions to generate string representations
    of stacks.  They only differ in that one prints the terms in stack from left-to-right while the other prints from right-to-left.  In both functions *internal stacks* are
    printed left-to-right.  These functions are written to support :doc:`../pretty`.
    
    .. _cons list: https://en.wikipedia.org/wiki/Cons#Lists


### The utility functions maintain order.
The 0th item in the list will be on the top of the stack and *vise versa*.


```python
joy.utils.stack.list_to_stack([1, 2, 3])
```




    (1, (2, (3, ())))




```python
list(joy.utils.stack.iter_stack((1, (2, (3, ())))))
```




    [1, 2, 3]



This requires reversing the sequence (or iterating backwards) otherwise:


```python
stack = ()

for n in [1, 2, 3]:
    stack = n, stack

print stack
print list(joy.utils.stack.iter_stack(stack))
```

    (3, (2, (1, ())))
    [3, 2, 1]


### Purely Functional Datastructures.
Because Joy lists are made out of Python tuples they are immutable, so all Joy datastructures are *[purely functional](https://en.wikipedia.org/wiki/Purely_functional_data_structure)*.

# The `joy()` function.
## An Interpreter
The `joy()` function is extrememly simple.  It accepts a stack, an expression, and a dictionary, and it iterates through the expression putting values onto the stack and delegating execution to functions it looks up in the dictionary.

Each function is passed the stack, expression, and dictionary and returns them.  Whatever the function returns becomes the new stack, expression, and dictionary.  (The dictionary is passed to enable e.g. writing words that let you enter new words into the dictionary at runtime, which nothing does yet and may be a bad idea, and the `help` command.)


```python
import joy.joy

print inspect.getsource(joy.joy.joy)
```

    def joy(stack, expression, dictionary, viewer=None):
    	'''Evaluate a Joy expression on a stack.
    
    	This function iterates through a sequence of terms which are either
    	literals (strings, numbers, sequences of terms) or function symbols.
    	Literals are put onto the stack and functions are looked up in the
    	disctionary and executed.
    
    	The viewer is a function that is called with the stack and expression
    	on every iteration, its return value is ignored.
    
    	:param stack stack: The stack.
    	:param stack expression: The expression to evaluate.
    	:param dict dictionary: A ``dict`` mapping names to Joy functions.
    	:param function viewer: Optional viewer function.
    	:rtype: (stack, (), dictionary)
    
    	'''
    	while expression:
    
    		if viewer: viewer(stack, expression)
    
    		term, expression = expression
    		if isinstance(term, Symbol):
    			term = dictionary[term]
    			stack, expression, dictionary = term(stack, expression, dictionary)
    		else:
    			stack = term, stack
    
    	if viewer: viewer(stack, expression)
    	return stack, expression, dictionary
    


### View function
The `joy()` function accepts a "viewer" function which it calls on each iteration passing the current stack and expression just before evaluation.  This can be used for tracing, breakpoints, retrying after exceptions, or interrupting an evaluation and saving to disk or sending over the network to resume later.  The stack and expression together contain all the state of the computation at each step.

### The `TracePrinter`.

A `viewer` records each step of the evaluation of a Joy program.  The `TracePrinter` has a facility for printing out a trace of the evaluation, one line per step.  Each step is aligned to the current interpreter position, signified by a period separating the stack on the left from the pending expression ("continuation") on the right.

### [Continuation-Passing Style](https://en.wikipedia.org/wiki/Continuation-passing_style)
One day I thought, What happens if you rewrite Joy to use [CSP](https://en.wikipedia.org/wiki/Continuation-passing_style)?  I made all the functions accept and return the expression as well as the stack and found that all the combinators could be rewritten to work by modifying the expression rather than making recursive calls to the `joy()` function.

# Parser


```python
import joy.parser

print inspect.getdoc(joy.parser)
```

    This module exports a single function for converting text to a joy
    expression as well as a single Symbol class and a single Exception type.
    
    The Symbol string class is used by the interpreter to recognize literals
    by the fact that they are not Symbol objects.
    
    A crude grammar::
    
        joy = term*
        term = int | float | string | '[' joy ']' | symbol
    
    A Joy expression is a sequence of zero or more terms.  A term is a
    literal value (integer, float, string, or Joy expression) or a function
    symbol.  Function symbols are unquoted strings and cannot contain square
    brackets.   Terms must be separated by blanks, which can be omitted
    around square brackets.


The parser is extremely simple, the undocumented `re.Scanner` class does most of the tokenizing work and then you just build the tuple structure out of the tokens.  There's no Abstract Syntax Tree or anything like that.


```python
print inspect.getsource(joy.parser._parse)
```

    def _parse(tokens):
    	'''
    	Return a stack/list expression of the tokens.
    	'''
    	frame = []
    	stack = []
    	for tok in tokens:
    		if tok == '[':
    			stack.append(frame)
    			frame = []
    			stack[-1].append(frame)
    		elif tok == ']':
    			try:
    				frame = stack.pop()
    			except IndexError:
    				raise ParseError('Extra closing bracket.')
    			frame[-1] = list_to_stack(frame[-1])
    		else:
    			frame.append(tok)
    	if stack:
    		raise ParseError('Unclosed bracket.')
    	return list_to_stack(frame)
    


That's pretty much all there is to it.


```python
joy.parser.text_to_expression('1 2 3 4 5')  # A simple sequence.
```




    (1, (2, (3, (4, (5, ())))))




```python
joy.parser.text_to_expression('[1 2 3] 4 5')  # Three items, the first is a list with three items
```




    ((1, (2, (3, ()))), (4, (5, ())))




```python
joy.parser.text_to_expression('1 23 ["four" [-5.0] cons] 8888')  # A mixed bag. cons is
                                                                 # a Symbol, no lookup at
                                                                 # parse-time.  Haiku docs.
```




    (1, (23, (('four', ((-5.0, ()), (cons, ()))), (8888, ()))))




```python
joy.parser.text_to_expression('[][][][][]')  # Five empty lists.
```




    ((), ((), ((), ((), ((), ())))))




```python
joy.parser.text_to_expression('[[[[[]]]]]')  # Five nested lists.
```




    ((((((), ()), ()), ()), ()), ())



# Library
The Joy library of functions (aka commands, or "words" after Forth usage) encapsulates all the actual functionality (no pun intended) of the Joy system.  There are simple functions such as addition `add` (or `+`, the library module supports aliases), and combinators which provide control-flow and higher-order operations.


```python
import joy.library

print ' '.join(sorted(joy.library.initialize()))
```

    != % & * *fraction *fraction0 + ++ - -- / // /floor < << <= <> = > >= >> ? ^ _Tree_add_Ee _Tree_delete_R0 _Tree_delete_clear_stuff _Tree_get_E abs add anamorphism and app1 app2 app3 at average b binary bool branch ccons choice clear cleave cmp codireco concat cond cons dinfrirst dip dipd dipdd disenstacken divmod down_to_zero drop dup dupd dupdd dupdip dupdipd enstacken eq first first_two flatten floor floordiv fork fourth gcd ge genrec getitem gt help i id ifte ii infer infra inscribe le least_fraction loop lshift lt make_generator map max min mod modulus mul ne neg not nullary of or over pam parse pick pm pop popd popdd popop popopd popopdd pow pred primrec product quoted range range_to_zero rem remainder remove rest reverse roll< roll> rolldown rollup round rrest rshift run second select sharing shunt size sort sqr sqrt stack step step_zero stuncons stununcons sub succ sum swaack swap swons take ternary third times truediv truthy tuck unary uncons unique unit unquoted unstack unswons void warranty while words x xor zip •


Many of the functions are defined in Python, like `dip`:


```python
print inspect.getsource(joy.library.dip)
```

    @inscribe
    @combinator_effect(_COMB_NUMS(), a1, s1)
    @FunctionWrapper
    def dip(stack, expression, dictionary):
    	'''
    	The dip combinator expects a quoted program on the stack and below it
    	some item, it hoists the item into the expression and runs the program
    	on the rest of the stack.
    	::
    
    			 ... x [Q] dip
    		-------------------
    				 ... Q x
    
    	'''
    	(quote, (x, stack)) = stack
    	expression = (x, expression)
    	return stack, concat(quote, expression), dictionary
    


Some functions are defined in equations in terms of other functions.  When the interpreter executes a definition function that function just pushes its body expression onto the pending expression (the continuation) and returns control to the interpreter.


```python
print joy.library.definitions
```

    ii == [dip] dupdip i
    of == swap at
    product == 1 swap [*] step
    flatten == [] swap [concat] step
    quoted == [unit] dip
    unquoted == [i] dip
    enstacken == stack [clear] dip
    ? == dup truthy
    disenstacken == ? [uncons ?] loop pop
    dinfrirst == dip infra first
    nullary == [stack] dinfrirst
    unary == nullary popd
    binary == nullary [popop] dip
    ternary == unary [popop] dip
    pam == [i] map
    run == [] swap infra
    sqr == dup mul
    size == 0 swap [pop ++] step
    fork == [i] app2
    cleave == fork [popd] dip
    average == [sum 1.0 *] [size] cleave /
    gcd == 1 [tuck modulus dup 0 >] loop pop
    least_fraction == dup [gcd] infra [div] concat map
    *fraction == [uncons] dip uncons [swap] dip concat [*] infra [*] dip cons
    *fraction0 == concat [[swap] dip * [*] dip] infra
    down_to_zero == [0 >] [dup --] while
    range_to_zero == unit [down_to_zero] infra
    anamorphism == [pop []] swap [dip swons] genrec
    range == [0 <=] [1 - dup] anamorphism
    while == swap [nullary] cons dup dipd concat loop
    dupdipd == dup dipd
    primrec == [i] genrec
    step_zero == 0 roll> step
    codireco == cons dip rest cons
    make_generator == [codireco] ccons
    ifte == [nullary not] dipd branch
    


Currently, there's no function to add new definitions to the dictionary from "within" Joy code itself.  Adding new definitions remains a meta-interpreter action.  You have to do it yourself, in Python, and wash your hands afterward.

It would be simple enough to define one, but it would open the door to *name binding* and break the idea that all state is captured in the stack and expression.  There's an implicit *standard dictionary* that defines the actual semantics of the syntactic stack and expression datastructures (which only contain symbols, not the actual functions.  Pickle some and see for yourself.)

#### "There should be only one."

Which brings me to talking about one of my hopes and dreams for this notation:  "There should be only one."  What I mean is that there should be one universal standard dictionary of commands, and all bespoke work done in a UI for purposes takes place by direct interaction and macros.  There would be a *Grand Refactoring* biannually (two years, not six months, that's semi-annually) where any new definitions factored out of the usage and macros of the previous time, along with new algorithms and such, were entered into the dictionary and posted to e.g. IPFS.

Code should not burgeon wildly, as it does today.  The variety of code should map more-or-less to the well-factored variety of human computably-solvable problems.  There shouldn't be dozens of chat apps, JS frameworks, programming languages.  It's a waste of time, a [fractal "thundering herd" attack](https://en.wikipedia.org/wiki/Thundering_herd_problem) on human mentality.

#### Literary Code Library

If you read over the other notebooks you'll see that developing code in Joy is a lot like doing simple mathematics, and the descriptions of the code resemble math papers.  The code also works the first time, no bugs.  If you have any experience programming at all, you are probably skeptical, as I was, but it seems to work: deriving code mathematically seems to lead to fewer errors.

But my point now is that this great ratio of textual explanation to wind up with code that consists of a few equations and could fit on an index card is highly desirable.  Less code has fewer errors.  The structure of Joy engenders a kind of thinking that seems to be very effective for developing structured processes.

There seems to be an elegance and power to the notation.



```python
  
```
