﻿/* テスト */
/*
 * File: obj-util.c
 * Purpose: Object list maintenance and other object utilities
 *
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */
extern "C"
{
#include "angband.h"
#include "randname.h"
#include "tvalsval.h"
#include "effects.h" 
#include "game-cmd.h"
}

extern const _TCHAR *quark_str(quark_t q);
extern void msg_print(const _TCHAR *msg);
extern void prt(const _TCHAR *str, int row, int col);
extern bool (*item_tester_hook)(const object_type*);
extern size_t object_desc(_TCHAR *buf, size_t max, const object_type *o_ptr, bool prefix, int mode);
extern size_t strnfmt(_TCHAR *buf, size_t max, const _TCHAR *fmt, ...);

/*
 * Hold the titles of scrolls, 6 to 14 characters each.
 */
_TCHAR scroll_adj[MAX_TITLES][16];

static void flavor_assign_fixed(void)
{
	int i, j;

	for (i = 0; i < z_info->flavor_max; i++)
	{
		flavor_type *flavor_ptr = &flavor_info[i];

		/* Skip random flavors */
		if (flavor_ptr->sval == SV_UNKNOWN) continue;

		for (j = 0; j < z_info->k_max; j++)
		{
			/* Skip other objects */
			if ((k_info[j].tval == flavor_ptr->tval) &&
			    (k_info[j].sval == flavor_ptr->sval))
			{
				/* Store the flavor index */
				k_info[j].flavor = i;
			}
		}
	}
}

static void flavor_assign_random(byte tval)
{
	int i, j;
	int flavor_count = 0;
	int choice;

	/* Count the random flavors for the given tval */
	for (i = 0; i < z_info->flavor_max; i++)
	{
		if ((flavor_info[i].tval == tval) &&
		    (flavor_info[i].sval == SV_UNKNOWN))
		{
			flavor_count++;
		}
	}
	for (i = 0; i < z_info->k_max; i++)
	{
		/* Skip other object types */
		if (k_info[i].tval != tval) continue;

		/* Skip objects that already are flavored */
		if (k_info[i].flavor != 0) continue;

		/* HACK - Ordinary food is "boring" */
		if ((tval == TV_FOOD) && (k_info[i].sval < SV_FOOD_MIN_SHROOM))
			continue;

		if (!flavor_count) quit_fmt(L"Not enough flavors for tval %d.", tval);

		/* Select a flavor */
		choice = randint0(flavor_count);
	
		/* Find and store the flavor */
		for (j = 0; j < z_info->flavor_max; j++)
		{
			/* Skip other tvals */
			if (flavor_info[j].tval != tval) continue;

			/* Skip assigned svals */
			if (flavor_info[j].sval != SV_UNKNOWN) continue;

			if (choice == 0)
			{
				/* Store the flavor index */
				k_info[i].flavor = j;

				/* Mark the flavor as used */
				flavor_info[j].sval = k_info[i].sval;

				/* One less flavor to choose from */
				flavor_count--;

				break;
			}
			choice--;
		}
	}
}

/*
 * Prepare the "variable" part of the "k_info" array.
 *
 * The "color"/"metal"/"type" of an item is its "flavor".
 * For the most part, flavors are assigned randomly each game.
 *
 * Initialize descriptions for the "colored" objects, including:
 * Rings, Amulets, Staffs, Wands, Rods, Food, Potions, Scrolls.
 *
 * The first 4 entries for potions are fixed (Water, Apple Juice,
 * Slime Mold Juice, Unused Potion).
 *
 * Scroll titles are always between 6 and 14 letters long.  This is
 * ensured because every title is composed of whole words, where every
 * word is from 2 to 8 letters long, and that no scroll is finished
 * until it attempts to grow beyond 15 letters.  The first time this
 * can happen is when the current title has 6 letters and the new word
 * has 8 letters, which would result in a 6 letter scroll title.
 *
 * Hack -- make sure everything stays the same for each saved game
 * This is accomplished by the use of a saved "random seed", as in
 * "town_gen()".  Since no other functions are called while the special
 * seed is in effect, so this function is pretty "safe".
 */
void flavor_init(void)
{
	int i, j;

	/* Hack -- Use the "simple" RNG */
	Rand_quick = TRUE;

	/* Hack -- Induce consistant flavors */
	Rand_value = seed_flavor;

	flavor_assign_fixed();

	flavor_assign_random(TV_RING);
	flavor_assign_random(TV_AMULET);
	flavor_assign_random(TV_STAFF);
	flavor_assign_random(TV_WAND);
	flavor_assign_random(TV_ROD);
	flavor_assign_random(TV_FOOD);
	flavor_assign_random(TV_POTION);
	flavor_assign_random(TV_SCROLL);

	/* Scrolls (random titles, always white) */
	for (i = 0; i < MAX_TITLES; i++)
	{
		_TCHAR buf[24];
		_TCHAR *end = buf;
		int titlelen = 0;
		int wordlen;
		bool okay = TRUE;

		wordlen = randname_make(RANDNAME_SCROLL, 2, 8, end, 24);
		while (titlelen + wordlen < (int)(_countof(scroll_adj[0]) - 1))
		{
			end[wordlen] = ' ';
			titlelen += wordlen + 1;
			end += wordlen + 1;
			wordlen = randname_make(RANDNAME_SCROLL, 2, 8, end, 24 - titlelen);
		}
		buf[titlelen - 1] = 0;

		/* Check the scroll name hasn't already been generated */
		for (j = 0; j < i; j++)
		{
			if (streq(buf, scroll_adj[j]))
			{
				okay = FALSE;
				break;
			}
		}
		if (okay)
		{
			_tcscpy_s(scroll_adj[i], _countof(scroll_adj[0]), buf);
		}
		else
		{
			/* Have another go at making a name */
			i--;
		}
	}
	/* Hack -- Use the "complex" RNG */
	Rand_quick = FALSE;

	/* Analyze every object */
	for (i = 1; i < z_info->k_max; i++)
	{
		object_kind *k_ptr = &k_info[i];

		/* Skip "empty" objects */
		if (!k_ptr->name) 
			continue;

		/* No flavor yields aware */
		if (!k_ptr->flavor) k_ptr->aware = TRUE;
	}
}

#ifdef ALLOW_BORG_GRAPHICS
extern void init_translate_visuals(void);
#endif /* ALLOW_BORG_GRAPHICS */

/*
 * Reset the "visual" lists
 *
 * This involves resetting various things to their "default" state.
 *
 * If the "prefs" flag is TRUE, then we will also load the appropriate
 * "user pref file" based on the current setting of the "use_graphics"
 * flag.  This is useful for switching "graphics" on/off.
 *
 * The features, objects, and monsters, should all be encoded in the
 * relevant "font.pref" and/or "graf.prf" files.  XXX XXX XXX
 *
 * The "prefs" parameter is no longer meaningful.  XXX XXX XXX
 */
void reset_visuals(bool unused)
{
	int i;

	UNREFERENCED_PARAMETER(unused);

	/* Extract default attr/char code for features */
	for (i = 0; i < z_info->f_max; i++)
	{
		feature_type *f_ptr = &f_info[i];

		/* Assume we will use the underlying values */
		f_ptr->x_attr = f_ptr->d_attr;
		f_ptr->x_char = f_ptr->d_char;
	}
	/* Extract default attr/char code for objects */
	for (i = 0; i < z_info->k_max; i++)
	{
		object_kind *k_ptr = &k_info[i];

		/* Default attr/char */
		k_ptr->x_attr = k_ptr->d_attr;
		k_ptr->x_char = k_ptr->d_char;
	}
	/* Extract default attr/char code for monsters */
	for (i = 0; i < z_info->r_max; i++)
	{
		monster_race *r_ptr = &r_info[i];

		/* Default attr/char */
		r_ptr->x_attr = r_ptr->d_attr;
		r_ptr->x_char = r_ptr->d_char;
	}
	/* Extract default attr/char code for flavors */
	for (i = 0; i < z_info->flavor_max; i++)
	{
		flavor_type *flavor_ptr = &flavor_info[i];

		/* Default attr/char */
		flavor_ptr->x_attr = flavor_ptr->d_attr;
		flavor_ptr->x_char = flavor_ptr->d_char;
	}
	/* Extract attr/chars for inventory objects (by tval) */
	for (i = 0; i < (int)N_ELEMENTS(tval_to_attr); i++)
	{
		/* Default to white */
		tval_to_attr[i] = TERM_WHITE;
	}
	/* Graphic symbols */
	if (use_graphics)
	{
		/* Process "graf.prf" */
		process_pref_file(L"graf.prf");
	}
	else /* Normal symbols */
	{
		/* Process "font.prf" */
		process_pref_file(L"font.prf");
	}
#ifdef ALLOW_BORG_GRAPHICS
	/* Initialize the translation table for the borg */
	init_translate_visuals();
#endif /* ALLOW_BORG_GRAPHICS */
}

/*
 * Obtain the "flags" for an item
 */
void object_flags(const object_type *o_ptr, u32b flags[OBJ_FLAG_N])
{
	object_kind *k_ptr = &k_info[o_ptr->k_idx];

	flags[0] = k_ptr->flags[0];
	flags[1] = k_ptr->flags[1];
	flags[2] = (k_ptr->flags[2] & ~TR2_CURSE_MASK);

	if (o_ptr->name1)
	{
		artifact_type *a_ptr = &a_info[o_ptr->name1];

		flags[0] = a_ptr->flags[0];
		flags[1] = a_ptr->flags[1];
		flags[2] = (a_ptr->flags[2] & ~TR2_CURSE_MASK);
	}
	if (o_ptr->name2)
	{
		ego_item_type *e_ptr = &e_info[o_ptr->name2];

		flags[0] |= e_ptr->flags[0];
		flags[1] |= e_ptr->flags[1];
		flags[2] |= (e_ptr->flags[2] & ~TR2_CURSE_MASK);
	}
	flags[0] |= o_ptr->flags[0];
	flags[1] |= o_ptr->flags[1];
	flags[2] |= o_ptr->flags[2];
}

/*
 * Obtain the "flags" for an item which are known to the player
 */
void object_flags_known(const object_type *o_ptr, u32b flags[])
{
	memcpy(flags, o_ptr->known_flags, sizeof(o_ptr->known_flags)); // TODO Check this works
}

/*
 * Convert an inventory index into a one character label.
 *
 * Note that the label does NOT distinguish inven/equip.
 */
char index_to_label(int i)
{
	/* Indexes for "inven" are easy */
	if (i < INVEN_WIELD) return (I2A(i));

	/* Indexes for "equip" are offset */
	return (I2A(i - INVEN_WIELD));
}

/*
 * Convert a label into the index of an item in the "inven".
 *
 * Return "-1" if the label does not indicate a real item.
 */
s16b label_to_inven(int c) /* TODO Should this really be int? */
{
	int i;

	/* Convert */
	i = (islower(c) ? A2I(c) : -1);

	/* Verify the index */
	if ((i < 0) || (i > INVEN_PACK)) 
		return (-1);

	/* Empty slots can never be chosen */
	if (!inventory[i].k_idx) 
		return (-1);

	/* Return the index */
	return (i);
}

/*
 * Convert a label into the index of a item in the "equip".
 *
 * Return "-1" if the label does not indicate a real item.
 */
s16b label_to_equip(int c) /* TODO Should this really be int? */
{
	int i;

	/* Convert */
	i = (islower(c) ? A2I(c) : -1) + INVEN_WIELD;

	/* Verify the index */
	if ((i < INVEN_WIELD) || (i >= INVEN_TOTAL)) 
		return (-1);

	/* Empty slots can never be chosen */
	if (!inventory[i].k_idx) return (-1);

	/* Return the index */
	return (i);
}

/*
 * Hack -- determine if an item is "wearable" (or a missile)
 */
bool wearable_p(const object_type *o_ptr)
{
	/* Valid "tval" codes */
	switch (o_ptr->tval)
	{
		case TV_SHOT:
		case TV_ARROW:
		case TV_BOLT:
		case TV_BOW:
		case TV_DIGGING:
		case TV_HAFTED:
		case TV_POLEARM:
		case TV_SWORD:
		case TV_BOOTS:
		case TV_GLOVES:
		case TV_HELM:
		case TV_CROWN:
		case TV_SHIELD:
		case TV_CLOAK:
		case TV_SOFT_ARMOR:
		case TV_HARD_ARMOR:
		case TV_DRAG_ARMOR:
		case TV_LITE:
		case TV_AMULET:
		case TV_RING:
			return (TRUE);
	}
	/* Nope */
	return (FALSE);
}

/*
 * Determine which equipment slot (if any) an item likes
 */
s16b wield_slot(const object_type *o_ptr)
{
	/* Slot for equipment */
	switch (o_ptr->tval)
	{
		case TV_DIGGING:
		case TV_HAFTED:
		case TV_POLEARM:
		case TV_SWORD:
			return (INVEN_WIELD);

		case TV_BOW:
			return (INVEN_BOW);

		case TV_RING:
		{
			/* Use the right hand first */
			if (!inventory[INVEN_RIGHT].k_idx) return (INVEN_RIGHT);

			/* Use the left hand for swapping (by default) */
			return (INVEN_LEFT);
		}

		case TV_AMULET:
			return (INVEN_NECK);

		case TV_LITE:
			return (INVEN_LITE);

		case TV_DRAG_ARMOR:
		case TV_HARD_ARMOR:
		case TV_SOFT_ARMOR:
			return (INVEN_BODY);

		case TV_CLOAK:
			return (INVEN_OUTER);

		case TV_SHIELD:
			return (INVEN_ARM);

		case TV_CROWN:
		case TV_HELM:
			return (INVEN_HEAD);

		case TV_GLOVES:
			return (INVEN_HANDS);

		case TV_BOOTS:
			return (INVEN_FEET);
	}
	/* No slot available */
	return (-1);
}

/*
 * Return a string mentioning how a given item is carried
 */
const _TCHAR *mention_use(int slot)
{
	switch (slot)
	{
		case INVEN_WIELD:
		{
			if (adj_str_hold[p_ptr->state.stat_ind[A_STR]] < inventory[slot].weight / 10)
				return L"Just lifting";
			else
				return L"Wielding";
		}
		case INVEN_BOW:
		{
			if (adj_str_hold[p_ptr->state.stat_ind[A_STR]] < inventory[slot].weight / 10)
				return L"Just holding";
			else
				return L"Shooting";
		}
		case INVEN_LEFT:  
			return L"On left hand";
		case INVEN_RIGHT: 
			return L"On right hand";
		case INVEN_NECK:  
			return L"Around neck";
		case INVEN_LITE:  
			return L"Light source";
		case INVEN_BODY:  
			return L"On body";
		case INVEN_OUTER: 
			return L"About body";
		case INVEN_ARM:   
			return L"On arm";
		case INVEN_HEAD:  
			return L"On head";
		case INVEN_HANDS: 
			return L"On hands";
		case INVEN_FEET:  
			return L"On feet";
	}
	return L"In pack";
}

/*
 * Return a string describing how a given item is being worn.
 * Currently, only used for items in the equipment, not inventory.
 */
const _TCHAR *describe_use(int i)
{
	const _TCHAR *p;

	switch (i)
	{
		case INVEN_WIELD: p = L"attacking monsters with"; break;
		case INVEN_BOW:   p = L"shooting missiles with"; break;
		case INVEN_LEFT:  p = L"wearing on your left hand"; break;
		case INVEN_RIGHT: p = L"wearing on your right hand"; break;
		case INVEN_NECK:  p = L"wearing around your neck"; break;
		case INVEN_LITE:  p = L"using to light the way"; break;
		case INVEN_BODY:  p = L"wearing on your body"; break;
		case INVEN_OUTER: p = L"wearing on your back"; break;
		case INVEN_ARM:   p = L"wearing on your arm"; break;
		case INVEN_HEAD:  p = L"wearing on your head"; break;
		case INVEN_HANDS: p = L"wearing on your hands"; break;
		case INVEN_FEET:  p = L"wearing on your feet"; break;
		default:          p = L"carrying in your pack"; break;
	}
	/* Hack -- Heavy weapon */
	if (i == INVEN_WIELD)
	{
		object_type *o_ptr;
		o_ptr = &inventory[i];
		if (adj_str_hold[p_ptr->state.stat_ind[A_STR]] < o_ptr->weight / 10)
		{
			p = L"just lifting";
		}
	}
	/* Hack -- Heavy bow */
	if (i == INVEN_BOW)
	{
		object_type *o_ptr;
		o_ptr = &inventory[i];
		if (adj_str_hold[p_ptr->state.stat_ind[A_STR]] < o_ptr->weight / 10)
		{
			p = L"just holding";
		}
	}
	/* Return the result */
	return p;
}

/*
 * Check an item against the item tester info
 */
bool item_tester_okay(const object_type *o_ptr)
{
	/* Hack -- allow listing empty slots */
	if (item_tester_full) return (TRUE);

	/* Require an item */
	if (!o_ptr->k_idx) 
		return (FALSE);

	/* Hack -- ignore "gold" */
	if (o_ptr->tval == TV_GOLD) 
		return (FALSE);

	/* Check the tval */
	if (item_tester_tval)
	{
		if (item_tester_tval != o_ptr->tval) 
			return (FALSE);
	}

	/* Check the hook */
	if (item_tester_hook)
	{
		if (!(*item_tester_hook)(o_ptr)) 
			return (FALSE);
	}

	/* Assume okay */
	return (TRUE);
}

/*
 * Get the indexes of objects at a given floor location. -TNB-
 *
 * Return the number of object indexes acquired.
 *
 * Valid flags are any combination of the bits:
 *   0x01 -- Verify item tester
 *   0x02 -- Marked/visible items only
 *   0x04 -- Only the top item
 */
int scan_floor(int *items, int max_size, int y, int x, int mode)
{
	int this_o_idx, next_o_idx;

	int num = 0;

	/* Sanity */
	if (!in_bounds(y, x)) return 0;

	/* Scan all objects in the grid */
	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
	{
		object_type *o_ptr;

		/* XXX Hack -- Enforce limit */
		if (num >= max_size) 
			break;

		/* Get the object */
		o_ptr = &o_list[this_o_idx];

		/* Get the next object */
		next_o_idx = o_ptr->next_o_idx;

		/* Item tester */
		if ((mode & 0x01) && !item_tester_okay(o_ptr)) 
			continue;

		/* Marked */
		if ((mode & 0x02) && (!o_ptr->marked || squelch_hide_item(o_ptr)))
			continue;

		/* Accept this item */
		items[num++] = this_o_idx;

		/* Only one */
		if (mode & 0x04) 
			break;
	}
	return num;
}

/*
 * Excise a dungeon object from any stacks
 */
void excise_object_idx(int o_idx)
{
	object_type *j_ptr;

	s16b this_o_idx, next_o_idx = 0;

	s16b prev_o_idx = 0;

	/* Object */
	j_ptr = &o_list[o_idx];

	/* Monster */
	if (j_ptr->held_m_idx)
	{
		monster_type *m_ptr;

		/* Monster */
		m_ptr = &mon_list[j_ptr->held_m_idx];

		/* Scan all objects in the grid */
		for (this_o_idx = m_ptr->hold_o_idx; this_o_idx; this_o_idx = next_o_idx)
		{
			object_type *o_ptr;

			/* Get the object */
			o_ptr = &o_list[this_o_idx];

			/* Get the next object */
			next_o_idx = o_ptr->next_o_idx;

			/* Done */
			if (this_o_idx == o_idx)
			{
				/* No previous */
				if (prev_o_idx == 0)
				{
					/* Remove from list */
					m_ptr->hold_o_idx = next_o_idx;
				}
				else /* Real previous */
				{
					object_type *i_ptr;

					/* Previous object */
					i_ptr = &o_list[prev_o_idx];

					/* Remove from list */
					i_ptr->next_o_idx = next_o_idx;
				}
				/* Forget next pointer */
				o_ptr->next_o_idx = 0;

				/* Done */
				break;
			}
			/* Save prev_o_idx */
			prev_o_idx = this_o_idx;
		}
	}
	else /* Dungeon */
	{
		int y = j_ptr->iy;
		int x = j_ptr->ix;

		/* Scan all objects in the grid */
		for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
		{
			object_type *o_ptr;

			/* Get the object */
			o_ptr = &o_list[this_o_idx];

			/* Get the next object */
			next_o_idx = o_ptr->next_o_idx;

			/* Done */
			if (this_o_idx == o_idx)
			{
				/* No previous */
				if (prev_o_idx == 0)
				{
					/* Remove from list */
					cave_o_idx[y][x] = next_o_idx;
				}
				else /* Real previous */
				{
					object_type *i_ptr;

					/* Previous object */
					i_ptr = &o_list[prev_o_idx];

					/* Remove from list */
					i_ptr->next_o_idx = next_o_idx;
				}
				/* Forget next pointer */
				o_ptr->next_o_idx = 0;

				/* Done */
				break;
			}
			/* Save prev_o_idx */
			prev_o_idx = this_o_idx;
		}
	}
}

/*
 * Delete a dungeon object
 *
 * Handle "stacks" of objects correctly.
 */
void delete_object_idx(int o_idx)
{
	object_type *j_ptr;

	/* Excise */
	excise_object_idx(o_idx);

	/* Object */
	j_ptr = &o_list[o_idx];

	/* Dungeon floor */
	if (!(j_ptr->held_m_idx))
	{
		int y, x;

		/* Location */
		y = j_ptr->iy;
		x = j_ptr->ix;

		/* Visual update */
		lite_spot(y, x);
	}
	/* Wipe the object */
	object_wipe(j_ptr);

	/* Count objects */
	o_cnt--;
}

/*
 * Deletes all objects at given location
 */
void delete_object(int y, int x)
{
	s16b this_o_idx, next_o_idx = 0;

	/* Paranoia */
	if (!in_bounds(y, x)) 
		return;

	/* Scan all objects in the grid */
	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
	{
		object_type *o_ptr;

		/* Get the object */
		o_ptr = &o_list[this_o_idx];

		/* Get the next object */
		next_o_idx = o_ptr->next_o_idx;

		/* Wipe the object */
		object_wipe(o_ptr);

		/* Count objects */
		o_cnt--;
	}

	/* Objects are gone */
	cave_o_idx[y][x] = 0;

	/* Visual update */
	lite_spot(y, x);
}

/*
 * Move an object from index i1 to index i2 in the object list
 */
static void compact_objects_aux(int i1, int i2)
{
	int i;

	object_type *o_ptr;

	/* Do nothing */
	if (i1 == i2) 
		return;

	/* Repair objects */
	for (i = 1; i < o_max; i++)
	{
		/* Get the object */
		o_ptr = &o_list[i];

		/* Skip "dead" objects */
		if (!o_ptr->k_idx) continue;

		/* Repair "next" pointers */
		if (o_ptr->next_o_idx == i1)
		{
			/* Repair */
			o_ptr->next_o_idx = i2;
		}
	}
	/* Get the object */
	o_ptr = &o_list[i1];

	/* Monster */
	if (o_ptr->held_m_idx)
	{
		monster_type *m_ptr;

		/* Get the monster */
		m_ptr = &mon_list[o_ptr->held_m_idx];

		/* Repair monster */
		if (m_ptr->hold_o_idx == i1)
		{
			/* Repair */
			m_ptr->hold_o_idx = i2;
		}
	}
	else /* Dungeon */
	{
		int y, x;

		/* Get location */
		y = o_ptr->iy;
		x = o_ptr->ix;

		/* Repair grid */
		if (cave_o_idx[y][x] == i1)
		{
			/* Repair */
			cave_o_idx[y][x] = i2;
		}
	}
	/* Hack -- move object */
	COPY(&o_list[i2], &o_list[i1], object_type);

	/* Hack -- wipe hole */
	object_wipe(o_ptr);
}

/*
 * Compact and reorder the object list
 *
 * This function can be very dangerous, use with caution!
 *
 * When compacting objects, we first destroy gold, on the basis that by the
 * time item compaction becomes an issue, the player really won't care.
 * We also nuke items marked as squelch.
 *
 * When compacting other objects, we base the saving throw on a combination of
 * object level, distance from player, and current "desperation".
 *
 * After compacting, we "reorder" the objects into a more compact order, and we
 * reset the allocation info, and the "live" array.
 */
void compact_objects(int size)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int i, y, x, cnt;

	int cur_lev, cur_dis, chance;

	/* Reorder objects when not passed a size */
	if (!size)
	{
		/* Excise dead objects (backwards!) */
		for (i = o_max - 1; i >= 1; i--)
		{
			object_type *o_ptr = &o_list[i];

			/* Skip real objects */
			if (o_ptr->k_idx) 
				continue;

			/* Move last object into open hole */
			compact_objects_aux(o_max - 1, i);

			/* Compress "o_max" */
			o_max--;
		}
		return;
	}
	/* Message */
	msg_print(__T("Compacting objects..."));

	/*** Try destroying objects ***/

	/* First do gold */
	for (i = 1; (i < o_max) && (size); i++)
	{
		object_type *o_ptr = &o_list[i];

		/* Nuke gold or squelched items */
		if (o_ptr->tval == TV_GOLD || squelch_item_ok(o_ptr))
		{
			delete_object_idx(i);
			size--;
		}
	}
	/* Compact at least 'size' objects */
	for (cnt = 1; size; cnt++)
	{
		/* Get more vicious each iteration */
		cur_lev = 5 * cnt;

		/* Get closer each iteration */
		cur_dis = 5 * (20 - cnt);

		/* Examine the objects */
		for (i = 1; (i < o_max) && (size); i++)
		{
			object_type *o_ptr = &o_list[i];
			object_kind *k_ptr = &k_info[o_ptr->k_idx];

			/* Skip dead objects */
			if (!o_ptr->k_idx) 
				continue;

			/* Hack -- High level objects start out "immune" */
			if (k_ptr->level > cur_lev && !k_ptr->squelch)
				continue;

			/* Monster */
			if (o_ptr->held_m_idx)
			{
				monster_type *m_ptr;

				/* Get the monster */
				m_ptr = &mon_list[o_ptr->held_m_idx];

				/* Get the location */
				y = m_ptr->fy;
				x = m_ptr->fx;

				/* Monsters protect their objects */
				if ((randint0(100) < 90) && !k_ptr->squelch)
					continue;
			}
			else /* Dungeon */
			{
				/* Get the location */
				y = o_ptr->iy;
				x = o_ptr->ix;
			}
			/* Nearby objects start out "immune" */
			if ((cur_dis > 0) && (distance(py, px, y, x) < cur_dis) && !k_ptr->squelch)
				continue;

			/* Saving throw */
			chance = 90;

			/* Hack -- only compact artifacts in emergencies */
			if (artifact_p(o_ptr) && (cnt < 1000)) chance = 100;

			/* Apply the saving throw */
			if (randint0(100) < chance) continue;

			/* Delete the object */
			delete_object_idx(i);
			size--;
		}
	}
	/* Reorder objects */
	compact_objects(0);
}

/*
 * Delete all the items when player leaves the level
 *
 * Note -- we do NOT visually reflect these (irrelevant) changes
 *
 * Hack -- we clear the "cave_o_idx[y][x]" field for every grid,
 * and the "m_ptr->next_o_idx" field for every monster, since
 * we know we are clearing every object.  Technically, we only
 * clear those fields for grids/monsters containing objects,
 * and we clear it once for every such object.
 */
void wipe_o_list(void)
{
	int i;

	/* Delete the existing objects */
	for (i = 1; i < o_max; i++)
	{
		object_type *o_ptr = &o_list[i];

		/* Skip dead objects */
		if (!o_ptr->k_idx) continue;

		/* Preserve artifacts */
		if (!character_dungeon || !OPT(adult_no_preserve))
		{
			/* Preserve only unknown artifacts */
			if (artifact_p(o_ptr) && !object_known_p(o_ptr))
				a_info[o_ptr->name1].cur_num = 0;
		}
		/* Mark artifacts as lost in logs */
		if (artifact_p(o_ptr))
			history_lose_artifact(o_ptr->name1);

		/* Monster */
		if (o_ptr->held_m_idx)
		{
			monster_type *m_ptr;

			/* Monster */
			m_ptr = &mon_list[o_ptr->held_m_idx];

			/* Hack -- see above */
			m_ptr->hold_o_idx = 0;
		}
		else /* Dungeon */
		{
			/* Get the location */
			int y = o_ptr->iy;
			int x = o_ptr->ix;

			/* Hack -- see above */
			cave_o_idx[y][x] = 0;
		}
		/* Wipe the object */
		(void)WIPE(o_ptr, object_type);
	}
	/* Reset "o_max" */
	o_max = 1;

	/* Reset "o_cnt" */
	o_cnt = 0;
}

/*
 * Get and return the index of a "free" object.
 *
 * This routine should almost never fail, but in case it does,
 * we must be sure to handle "failure" of this routine.
 */
s16b o_pop(void)
{
	int i;

	/* Initial allocation */
	if (o_max < z_info->o_max)
	{
		/* Get next space */
		i = o_max;

		/* Expand object array */
		o_max++;

		/* Count objects */
		o_cnt++;

		/* Use this object */
		return (i);
	}
	/* Recycle dead objects */
	for (i = 1; i < o_max; i++)
	{
		object_type *o_ptr;

		/* Get the object */
		o_ptr = &o_list[i];

		/* Skip live objects */
		if (o_ptr->k_idx) continue;

		/* Count objects */
		o_cnt++;

		/* Use this object */
		return (i);
	}
	/* Warn the player (except during dungeon creation) */
	if (character_dungeon) 
		msg_print(__T("Too many objects!"));

	/* Oops */
	return (0);
}

/*
 * Get the first object at a dungeon location
 * or NULL if there isn't one.
 */
object_type *get_first_object(int y, int x)
{
	s16b o_idx = cave_o_idx[y][x];

	if (o_idx) return (&o_list[o_idx]);

	/* No object */
	return (NULL);
}

/*
 * Get the next object in a stack or NULL if there isn't one.
 */
object_type *get_next_object(const object_type *o_ptr)
{
	if (o_ptr->next_o_idx) 
		return (&o_list[o_ptr->next_o_idx]);

	/* No more objects */
	return (NULL);
}

/*
 * Determine if a weapon is 'blessed'
 */
bool is_blessed(const object_type *o_ptr)
{
	u32b f[OBJ_FLAG_N];

	/* Get the flags */
	object_flags(o_ptr, f);

	/* Is the object blessed? */
	return ((f[2] & TR2_BLESSED) ? TRUE : FALSE);
}

/*
 * Return the "value" of an "unknown" item
 * Make a guess at the value of non-aware items
 */
static s32b object_value_base(const object_type *o_ptr)
{
	object_kind *k_ptr = &k_info[o_ptr->k_idx];

	/* Use template cost for aware objects */
	if (object_aware_p(o_ptr)) 
		return (k_ptr->cost);

	/* Analyze the type */
	switch (o_ptr->tval)
	{
		/* Un-aware Food */
		case TV_FOOD: 
			return (5L);

		/* Un-aware Potions */
		case TV_POTION:
			return (20L);

		/* Un-aware Scrolls */
		case TV_SCROLL: 
			return (20L);

		/* Un-aware Staffs */
		case TV_STAFF: 
			return (70L);

		/* Un-aware Wands */
		case TV_WAND: 
			return (50L);

		/* Un-aware Rods */
		case TV_ROD: 
			return (90L);

		/* Un-aware Rings */
		case TV_RING: 
			return (45L);

		/* Un-aware Amulets */
		case TV_AMULET: 
			return (45L);
	}
	/* Paranoia -- Oops */
	return (0L);
}

/*
 * Return the "real" price of a "known" item, not including discounts.
 *
 * Wand and staffs get cost for each charge.
 *
 * Wearable items (weapons, launchers, jewelry, lights, armour) and ammo 
 * are priced according to their power rating. All ammo, and normal (non-ego) 
 * torches are scaled down by AMMO_RESCALER to reflect their impermanence. 
 */
static s32b object_value_real(const object_type *o_ptr, int qty)
{
	s32b value, total_value;

	u32b f[OBJ_FLAG_N];

	object_kind *k_ptr = &k_info[o_ptr->k_idx];

	s32b power;
	int a = 2;
	int b = 1;
	static file_mode pricing_mode = MODE_WRITE;

	if (wearable_p(o_ptr))
	{
 		_TCHAR buf[1024];
		int verbose = 1;
		ang_file *log_file = NULL;

		if (verbose)
		{                
			path_build(buf, _countof(buf), ANGBAND_DIR_USER, L"pricing.log");
                	log_file = file_open(buf, pricing_mode, FTYPE_TEXT);
                	if (!log_file)
                	{
                		msg_print(__T("Error - can't open pricing.log for writing."));
                	        exit(1);
                	}
			pricing_mode = MODE_APPEND;
		}
		LOG_PRINT1(L"object is %s", k_name + k_ptr->name);
		power = object_power(o_ptr, verbose, log_file);
		value = sign(power) * ((a * power * power) + (b * power));

		if ( (o_ptr->tval == TV_SHOT) || (o_ptr->tval == TV_ARROW) ||
  			(o_ptr->tval == TV_BOLT) || ((o_ptr->tval == TV_LITE) &&
			(o_ptr->sval == SV_LITE_TORCH)) )
		{
			value = value / AMMO_RESCALER;
			if (value < 1) value = 1;
		}
		LOG_PRINT2(L"a is %d and b is %d\n", a, b);
		LOG_PRINT1(L"value is %d\n", value);
		total_value = value * qty;

        if (verbose)
        {
            if (!file_close(log_file))
            {
                msg_print(__T("Error - can't close pricing.log file."));
                exit(1);
            }
        }
		if (total_value < 0) total_value = 0;

		return (total_value);
	}

	/* Hack -- "worthless" items */
	if (!k_ptr->cost) return (0L);

	/* Base cost */
	value = k_ptr->cost;

	/* Extract some flags */
	object_flags(o_ptr, f);

	/* Analyze the item type and quantity*/
	switch (o_ptr->tval)
	{
		/* Wands/Staffs */
		case TV_WAND:
		case TV_STAFF:
		{
			int charges;

			total_value = value * qty;

			/* Calculate number of charges, rounded up */
			charges = o_ptr->pval * qty / o_ptr->number;
			if ((o_ptr->pval * qty) % o_ptr->number != 0) charges++;

			/* Pay extra for charges, depending on standard number of charges */
			total_value += value * charges / 20;

			/* Done */
			break;
		}
		default:
		{
			total_value = value * qty;
			break;
		}
	}
	/* No negative value */
	if (total_value < 0) total_value = 0;

	/* Return the value */
	return (total_value);
}

/*
 * Return the price of an item including plusses (and charges).
 *
 * This function returns the "value" of the given item (qty one).
 *
 * Never notice "unknown" bonuses or properties, including "curses",
 * since that would give the player information he did not have.
 *
 * Note that discounted items stay discounted forever.
 */
s32b object_value(const object_type *o_ptr, int qty)
{
	s32b value;

	if (object_known_p(o_ptr))
	{
		if (cursed_p(o_ptr)) return (0L);

		value = object_value_real(o_ptr, qty);
	}
	else if (wearable_p(o_ptr))
	{
		object_type object_type_body;
		object_type *j_ptr = &object_type_body;

		/* Hack -- Felt cursed items */
		if ((o_ptr->ident & IDENT_SENSE) && cursed_p(o_ptr)) return (0L);

		memcpy(j_ptr, o_ptr, sizeof(object_type));
		memcpy(j_ptr->flags, j_ptr->known_flags, sizeof(j_ptr->flags));
		if (!(o_ptr->ident & IDENT_ATTACK))
			j_ptr->to_h = j_ptr->to_d = 0;
		else if (!(o_ptr->ident & IDENT_DEFENCE))
			j_ptr->to_a = 0;

		value = object_value_real(j_ptr, qty);
	}
	else 
		value = object_value_base(o_ptr) * qty;

	/* Return the final value */
	return (value);
}

/*
 * Determine if an item can "absorb" a second item
 *
 * See "object_absorb()" for the actual "absorption" code.
 *
 * If permitted, we allow weapons/armor to stack, if "known".
 *
 * Missiles will combine if both stacks have the same "known" status.
 * This is done to make unidentified stacks of missiles useful.
 *
 * Food, potions, scrolls, and "easy know" items always stack.
 *
 * Chests, and activatable items, except rods, never stack (for various
 * reasons).
 */
bool object_similar(const object_type *o_ptr, const object_type *j_ptr)
{
	int total = o_ptr->number + j_ptr->number;

	/* Require identical object types */
	if (o_ptr->k_idx != j_ptr->k_idx) return (0);

	/* Analyze the items */
	switch (o_ptr->tval)
	{
		/* Chests */
		case TV_CHEST:
		{
			/* Never okay */
			return (0);
		}

		/* Food and Potions and Scrolls */
		case TV_FOOD:
		case TV_POTION:
		case TV_SCROLL:
		{
			/* Assume okay */
			break;
		}

		/* Staves and Wands */
		case TV_STAFF:
		case TV_WAND:
		{
			/* Assume okay */
			break;
		}

		/* Rods */
		case TV_ROD:
		{
			/* Assume okay */
			break;
		}

		/* Weapons and Armor */
		case TV_BOW:
		case TV_DIGGING:
		case TV_HAFTED:
		case TV_POLEARM:
		case TV_SWORD:
		case TV_BOOTS:
		case TV_GLOVES:
		case TV_HELM:
		case TV_CROWN:
		case TV_SHIELD:
		case TV_CLOAK:
		case TV_SOFT_ARMOR:
		case TV_HARD_ARMOR:
		case TV_DRAG_ARMOR:
		{
			/* Fall through */
		}

		/* Rings, Amulets, Lites */
		case TV_RING:
		case TV_AMULET:
		case TV_LITE:
		{
			/* Require both items to be known */
			if (!object_known_p(o_ptr) || !object_known_p(j_ptr)) return (0);

			/* Fall through */
		}

		/* Missiles */
		case TV_BOLT:
		case TV_ARROW:
		case TV_SHOT:
		{
			/* Require identical knowledge of both items */
			if (object_known_p(o_ptr) != object_known_p(j_ptr)) return (0);

			/* Require identical "bonuses" */
			if (o_ptr->to_h != j_ptr->to_h) return (FALSE);
			if (o_ptr->to_d != j_ptr->to_d) return (FALSE);
			if (o_ptr->to_a != j_ptr->to_a) return (FALSE);

			/* Require identical "pval" code */
			if (o_ptr->pval != j_ptr->pval) return (FALSE);

			/* Require identical "artifact" names */
			if (o_ptr->name1 != j_ptr->name1) return (FALSE);

			/* Require identical "ego-item" names */
			if (o_ptr->name2 != j_ptr->name2) return (FALSE);

			/* Hack - Never stack recharging items */
			if ((o_ptr->timeout || j_ptr->timeout) && o_ptr->tval != TV_LITE) 
				return FALSE;

			/* Lites must have same amount of fuel */
			else if (o_ptr->timeout != j_ptr->timeout && o_ptr->tval == TV_LITE)
				return FALSE;

			/* Require identical "values" */
			if (o_ptr->ac != j_ptr->ac) return (FALSE);
			if (o_ptr->dd != j_ptr->dd) return (FALSE);
			if (o_ptr->ds != j_ptr->ds) return (FALSE);

			/* Probably okay */
			break;
		}

		/* Various */
		default:
		{
			/* Require knowledge */
			if (!object_known_p(o_ptr) || !object_known_p(j_ptr)) return (0);

			/* Probably okay */
			break;
		}
	}

	/* Hack -- Require compatible inscriptions */
	if (o_ptr->note != j_ptr->note)
	{
		/* Never combine different inscriptions */
		if (o_ptr->note && j_ptr->note) return (0);
	}

	/* Different flags */
	if (o_ptr->flags[0] != j_ptr->flags[0] ||
		o_ptr->flags[1] != j_ptr->flags[1] ||
		o_ptr->flags[2] != j_ptr->flags[2])
		return FALSE;

	/* Maximal "stacking" limit */
	if (total >= MAX_STACK_SIZE) return (0);

	/* They match, so they must be similar */
	return (TRUE);
}

/*
 * Allow one item to "absorb" another, assuming they are similar.
 *
 * The blending of the "note" field assumes that either (1) one has an
 * inscription and the other does not, or (2) neither has an inscription.
 * In both these cases, we can simply use the existing note, unless the
 * blending object has a note, in which case we use that note.
 *
 * The blending of the "discount" field assumes that either (1) one is a
 * special inscription and one is nothing, or (2) one is a discount and
 * one is a smaller discount, or (3) one is a discount and one is nothing,
 * or (4) both are nothing.  In all of these cases, we can simply use the
 * "maximum" of the two "discount" fields.
 *
 * These assumptions are enforced by the "object_similar()" code.
 */
void object_absorb(object_type *o_ptr, const object_type *j_ptr)
{
	object_kind *k_ptr = &k_info[o_ptr->k_idx];

	int total = o_ptr->number + j_ptr->number;

	/* Add together the item counts */
	o_ptr->number = ((total < MAX_STACK_SIZE) ? total : (MAX_STACK_SIZE - 1));

	/* Hack -- Blend "known" status */
	if (object_known_p(j_ptr)) object_known(o_ptr);

	/* Hack -- Blend store status */
	if (j_ptr->ident & (IDENT_STORE)) o_ptr->ident |= (IDENT_STORE);

	/* Hack -- Blend "notes" */
	if (j_ptr->note != 0) o_ptr->note = j_ptr->note;

	/*
	 * Hack -- if rods are stacking, re-calculate the
	 * pvals (maximum timeouts) and current timeouts together
	 */
	if (o_ptr->tval == TV_ROD)
	{
		o_ptr->pval = total * k_ptr->pval;
		o_ptr->timeout += j_ptr->timeout;
	}

	/* Hack -- if wands or staves are stacking, combine the charges */
	if ((o_ptr->tval == TV_WAND) || (o_ptr->tval == TV_STAFF))
	{
		o_ptr->pval += j_ptr->pval;
	}

	if ((o_ptr->origin != j_ptr->origin) ||
	    (o_ptr->origin_depth != j_ptr->origin_depth) ||
	    (o_ptr->origin_xtra != j_ptr->origin_xtra))
	{
		int act = 2;

		if ((o_ptr->origin == ORIGIN_DROP) && (o_ptr->origin == j_ptr->origin))
		{
			monster_race *r_ptr = &r_info[o_ptr->origin_xtra];
			monster_race *s_ptr = &r_info[j_ptr->origin_xtra];

			bool r_uniq = (r_ptr->flags[0] & RF0_UNIQUE) ? TRUE : FALSE;
			bool s_uniq = (s_ptr->flags[0] & RF0_UNIQUE) ? TRUE : FALSE;

			if (r_uniq && !s_uniq) act = 0;
			else if (s_uniq && !r_uniq) act = 1;
			else act = 2;
		}

		switch (act)
		{
			/* Overwrite with j_ptr */
			case 1:
			{
				o_ptr->origin = j_ptr->origin;
				o_ptr->origin_depth = j_ptr->origin_depth;
				o_ptr->origin_xtra = j_ptr->origin_xtra;
			}

			/* Set as "mixed" */
			case 2:
			{
				o_ptr->origin = ORIGIN_MIXED;
			}
		}
	}
}

/*
 * Wipe an object clean.
 */
void object_wipe(object_type *o_ptr)
{
	/* Wipe the structure */
	(void)WIPE(o_ptr, object_type);
}

/*
 * Prepare an object based on an existing object
 */
void object_copy(object_type *o_ptr, const object_type *j_ptr)
{
	/* Copy the structure */
	COPY(o_ptr, j_ptr, object_type);
}

/* 
 * Prepare an object `dst` representing `amt` objects,  based on an existing  
 * object `src` representing at least `amt` objects. 
 * 
 * Takes care of the charge redistribution concerns of stacked items. 
 */ 
void object_copy_amt(object_type *dst, object_type *src, int amt) 
{ 
	/* Get a copy of the object */ 
	object_copy(dst, src); 

	/* Modify quantity */ 
	dst->number = amt; 
	 
	/*  
	 * If the item has charges/timeouts, set them to the correct level  
	 * too. We split off the same amount as distribute_charges. 
	 */ 
	if (src->tval == TV_WAND || src->tval == TV_STAFF) 
	{ 
		dst->pval = src->pval * amt / src->number; 
	} 

	if (src->tval == TV_ROD) 
	{ 
		int max_time = k_info[src->k_idx].time_base * amt; 

		if (src->timeout > max_time) 
			dst->timeout = max_time; 
		else 
			dst->timeout = src->timeout; 
	} 
} 

/*
 * Prepare an object based on an object kind.
 */
void object_prep(object_type *o_ptr, int k_idx)
{
	object_kind *k_ptr = &k_info[k_idx];

	/* Clear the record */
	(void)WIPE(o_ptr, object_type);

	/* Save the kind index */
	o_ptr->k_idx = k_idx;

	/* Efficiency -- tval/sval */
	o_ptr->tval = k_ptr->tval;
	o_ptr->sval = k_ptr->sval;

	/* Default "pval" */
	o_ptr->pval = k_ptr->pval;

	/* Default number */
	o_ptr->number = 1;

	/* Default weight */
	o_ptr->weight = k_ptr->weight;

	/* Default magic */
	o_ptr->to_h = k_ptr->to_h;
	o_ptr->to_d = k_ptr->to_d;
	o_ptr->to_a = k_ptr->to_a;

	/* Default power */
	o_ptr->ac = k_ptr->ac;
	o_ptr->dd = k_ptr->dd;
	o_ptr->ds = k_ptr->ds;

	/* Hack -- cursed items are always "cursed" */
	if (k_ptr->flags[2] & TR2_LIGHT_CURSE)
	    o_ptr->flags[2] |= TR2_LIGHT_CURSE;
}

/**
 * Find and return the index to the oldest object on the given grid marked as
 * "squelch".
 */
static s16b floor_get_idx_oldest_squelched(int y, int x)
{
	s16b squelch_idx = 0;
	s16b this_o_idx;

	object_type *o_ptr = NULL;

	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = o_ptr->next_o_idx)
	{
		o_ptr = &o_list[this_o_idx];

		if (squelch_hide_item(o_ptr))
			squelch_idx = this_o_idx;
	}

	return squelch_idx;
}

/*
 * Let the floor carry an object, deleting old squelched items if necessary
 */
s16b floor_carry(int y, int x, object_type *j_ptr)
{
	int n = 0;

	s16b o_idx;

	s16b this_o_idx, next_o_idx = 0;

	/* Scan objects in that grid for combination */
	for (this_o_idx = cave_o_idx[y][x]; this_o_idx; this_o_idx = next_o_idx)
	{
		object_type *o_ptr = &o_list[this_o_idx];

		/* Get the next object */
		next_o_idx = o_ptr->next_o_idx;

		/* Check for combination */
		if (object_similar(o_ptr, j_ptr))
		{
			/* Combine the items */
			object_absorb(o_ptr, j_ptr);

			/* Result */
			return (this_o_idx);
		}

		/* Count objects */
		n++;
	}

	/* Option -- disallow stacking */
	if (OPT(adult_no_stacking) && n) return (0);

	/* The stack is already too large */
	if (n >= MAX_FLOOR_STACK)
	{
		/* Squelch the oldest squelched object */
		s16b squelch_idx = floor_get_idx_oldest_squelched(y, x);

		if (squelch_idx)
			delete_object_idx(squelch_idx);
		else
			return 0;
	}

	/* Make an object */
	o_idx = o_pop();

	/* Success */
	if (o_idx)
	{
		object_type *o_ptr;

		/* Get the object */
		o_ptr = &o_list[o_idx];

		/* Structure Copy */
		object_copy(o_ptr, j_ptr);

		/* Location */
		o_ptr->iy = y;
		o_ptr->ix = x;

		/* Forget monster */
		o_ptr->held_m_idx = 0;

		/* Link the object to the pile */
		o_ptr->next_o_idx = cave_o_idx[y][x];

		/* Link the floor to the object */
		cave_o_idx[y][x] = o_idx;

		/* Notice */
		note_spot(y, x);

		/* Redraw */
		lite_spot(y, x);
	}

	/* Result */
	return (o_idx);
}

/*
 * Let an object fall to the ground at or near a location.
 *
 * The initial location is assumed to be "in_bounds_fully()".
 *
 * This function takes a parameter "chance".  This is the percentage
 * chance that the item will "disappear" instead of drop.  If the object
 * has been thrown, then this is the chance of disappearance on contact.
 *
 * Hack -- this function uses "chance" to determine if it should produce
 * some form of "description" of the drop event (under the player).
 *
 * We check several locations to see if we can find a location at which
 * the object can combine, stack, or be placed.  Artifacts will try very
 * hard to be placed, including "teleporting" to a useful grid if needed.
 */
void drop_near(object_type *j_ptr, int chance, int y, int x)
{
	int i, k, n, d, s;

	int bs, bn;
	int by, bx;
	int dy, dx;
	int ty, tx;

	object_type *o_ptr;

	_TCHAR o_name[80];

	bool flag = FALSE;

	bool plural = FALSE;

	/* Extract plural */
	if (j_ptr->number != 1) plural = TRUE;

	/* Describe object */
	object_desc(o_name, _countof(o_name), j_ptr, FALSE, ODESC_BASE);

	/* Handle normal "breakage" */
	if (!artifact_p(j_ptr) && (randint0(100) < chance))
	{
		/* Message */
		msg_format(__T("The %s break%s."), o_name, PLURAL(plural));

		/* Failure */
		return;
	}

	/* Score */
	bs = -1;

	/* Picker */
	bn = 0;

	/* Default */
	by = y;
	bx = x;

	/* Scan local grids */
	for (dy = -3; dy <= 3; dy++)
	{
		/* Scan local grids */
		for (dx = -3; dx <= 3; dx++)
		{
			bool comb = FALSE;

			/* Calculate actual distance */
			d = (dy * dy) + (dx * dx);

			/* Ignore distant grids */
			if (d > 10) continue;

			/* Location */
			ty = y + dy;
			tx = x + dx;

			/* Skip illegal grids */
			if (!in_bounds_fully(ty, tx)) continue;

			/* Require line of sight */
			if (!los(y, x, ty, tx)) continue;

			/* Require floor space */
			if (cave_feat[ty][tx] != FEAT_FLOOR) continue;

			/* No objects */
			k = 0;
			n = 0;

			/* Scan objects in that grid */
			for (o_ptr = get_first_object(ty, tx); o_ptr;
					o_ptr = get_next_object(o_ptr))
			{
				/* Check for possible combination */
				if (object_similar(o_ptr, j_ptr)) comb = TRUE;

				/* Count objects */
				if (!squelch_hide_item(o_ptr))
					k++;
				else
					n++;
			}

			/* Add new object */
			if (!comb) k++;

			/* Option -- disallow stacking */
			if (OPT(adult_no_stacking) && (k > 1)) continue;
			
			/* Paranoia? */
			if ((k + n) > MAX_FLOOR_STACK &&
					!floor_get_idx_oldest_squelched(ty, tx)) continue;

			/* Calculate score */
			s = 1000 - (d + k * 5);

			/* Skip bad values */
			if (s < bs) continue;

			/* New best value */
			if (s > bs) bn = 0;

			/* Apply the randomizer to equivalent values */
			if ((++bn >= 2) && (randint0(bn) != 0)) continue;

			/* Keep score */
			bs = s;

			/* Track it */
			by = ty;
			bx = tx;

			/* Okay */
			flag = TRUE;
		}
	}

	/* Handle lack of space */
	if (!flag && !artifact_p(j_ptr))
	{
		/* Message */
		msg_format(__T("The %s disappear%s."), o_name, PLURAL(plural));

		/* Debug */
		if (p_ptr->wizard) msg_print(__T("Breakage (no floor space)."));

		/* Failure */
		return;
	}

	/* Find a grid */
	for (i = 0; !flag; i++)
	{
		/* Bounce around */
		if (i < 1000)
		{
			ty = rand_spread(by, 1);
			tx = rand_spread(bx, 1);
		}

		/* Random locations */
		else
		{
			ty = randint0(DUNGEON_HGT);
			tx = randint0(DUNGEON_WID);
		}

		/* Require floor space */
		if (cave_feat[ty][tx] != FEAT_FLOOR) continue;

		/* Bounce to that location */
		by = ty;
		bx = tx;

		/* Require floor space */
		if (!cave_clean_bold(by, bx)) continue;

		/* Okay */
		flag = TRUE;
	}

	/* Give it to the floor */
	if (!floor_carry(by, bx, j_ptr))
	{
		/* Message */
		msg_format(__T("The %s disappear%s."), o_name, PLURAL(plural));

		/* Debug */
		if (p_ptr->wizard) msg_print(__T("Breakage (too many objects)."));

		/* Hack -- Preserve artifacts */
		a_info[j_ptr->name1].cur_num = 0;

		/* Failure */
		return;
	}

	/* Sound */
	sound(MSG_DROP);

	/* Mega-Hack -- no message if "dropped" by player */
	/* Message when an object falls under the player */
	if (chance && (cave_m_idx[by][bx] < 0))
	{
		msg_print(__T("You feel something roll beneath your feet."));
	}
}

/*
 * Scatter some "great" objects near the player
 */
void acquirement(int y1, int x1, int level, int num, bool great)
{
	object_type *i_ptr;
	object_type object_type_body;

	/* Acquirement */
	while (num--)
	{
		/* Get local object */
		i_ptr = &object_type_body;

		/* Wipe the object */
		object_wipe(i_ptr);

		/* Make a good (or great) object (if possible) */
		if (!make_object(i_ptr, level, TRUE, great)) continue;
		i_ptr->origin = ORIGIN_ACQUIRE;
		i_ptr->origin_depth = (byte) p_ptr->depth;

		/* Drop the object */
		drop_near(i_ptr, -1, y1, x1);
	}
}

/*
 * Describe the charges on an item in the inventory.
 */
void inven_item_charges(int item)
{
	object_type *o_ptr = &inventory[item];

	/* Require staff/wand */
	if ((o_ptr->tval != TV_STAFF) && (o_ptr->tval != TV_WAND)) return;

	/* Require known item */
	if (!object_known_p(o_ptr)) return;

	/* Print a message */
	msg_format(__T("You have %d charge%s remaining."), o_ptr->pval,
	           (o_ptr->pval != 1) ? L"s" : __T(""));
}

/*
 * Describe an item in the inventory.
 */
void inven_item_describe(int item)
{
	object_type *o_ptr = &inventory[item];

	_TCHAR o_name[80];

	if (artifact_p(o_ptr) && object_known_p(o_ptr))
	{
		/* Get a description */
		object_desc(o_name, _countof(o_name), o_ptr, FALSE, ODESC_FULL);

		/* Print a message */
		msg_format(__T("You no longer have the %s (%c)."), o_name, index_to_label(item));
	}
	else
	{
		/* Get a description */
		object_desc(o_name, _countof(o_name), o_ptr, TRUE, ODESC_FULL);

		/* Print a message */
		msg_format(__T("You have %s (%c)."), o_name, index_to_label(item));
	}
}

/*
 * Increase the "number" of an item in the inventory
 */
void inven_item_increase(int item, int num)
{
	object_type *o_ptr = &inventory[item];

	/* Apply */
	num += o_ptr->number;

	/* Bounds check */
	if (num > 255) num = 255;
	else if (num < 0) num = 0;

	/* Un-apply */
	num -= o_ptr->number;

	/* Change the number and weight */
	if (num)
	{
		/* Add the number */
		o_ptr->number += num;

		/* Add the weight */
		p_ptr->total_weight += (num * o_ptr->weight);

		/* Recalculate bonuses */
		p_ptr->update |= (PU_BONUS);

		/* Recalculate mana XXX */
		p_ptr->update |= (PU_MANA);

		/* Combine the pack */
		p_ptr->notice |= (PN_COMBINE);

		/* Redraw stuff */
		p_ptr->redraw |= (PR_INVEN | PR_EQUIP);
	}
}

/*
 * Erase an inventory slot if it has no more items
 */
void inven_item_optimize(int item)
{
	object_type *o_ptr = &inventory[item];

	/* Only optimize real items */
	if (!o_ptr->k_idx) return;

	/* Only optimize empty items */
	if (o_ptr->number) return;

	/* The item is in the pack */
	if (item < INVEN_WIELD)
	{
		int i;

		/* One less item */
		p_ptr->inven_cnt--;

		/* Slide everything down */
		for (i = item; i < INVEN_PACK; i++)
		{
			/* Hack -- slide object */
			COPY(&inventory[i], &inventory[i+1], object_type);
		}

		/* Hack -- wipe hole */
		(void)WIPE(&inventory[i], object_type);

		/* Redraw stuff */
		p_ptr->redraw |= (PR_INVEN);
	}

	/* The item is being wielded */
	else
	{
		/* One less item */
		p_ptr->equip_cnt--;

		/* Erase the empty slot */
		object_wipe(&inventory[item]);

		/* Recalculate bonuses */
		p_ptr->update |= (PU_BONUS);

		/* Recalculate torch */
		p_ptr->update |= (PU_TORCH);

		/* Recalculate mana XXX */
		p_ptr->update |= (PU_MANA);

		/* Redraw stuff */
		p_ptr->redraw |= (PR_EQUIP);
	}
}

/*
 * Describe the charges on an item on the floor.
 */
void floor_item_charges(int item)
{
	object_type *o_ptr = &o_list[item];

	/* Require staff/wand */
	if ((o_ptr->tval != TV_STAFF) && (o_ptr->tval != TV_WAND)) return;

	/* Require known item */
	if (!object_known_p(o_ptr)) return;

	/* Print a message */
	msg_format(__T("There are %d charge%s remaining."), o_ptr->pval,
	           (o_ptr->pval != 1) ? L"s" : __T(""));
}

/*
 * Describe an item in the inventory.
 */
void floor_item_describe(int item)
{
	object_type *o_ptr = &o_list[item];

	_TCHAR o_name[80];

	/* Get a description */
	object_desc(o_name, _countof(o_name), o_ptr, TRUE, ODESC_FULL);

	/* Print a message */
	msg_format(__T("You see %s."), o_name);
}

/*
 * Increase the "number" of an item on the floor
 */
void floor_item_increase(int item, int num)
{
	object_type *o_ptr = &o_list[item];

	/* Apply */
	num += o_ptr->number;

	/* Bounds check */
	if (num > 255) num = 255;
	else if (num < 0) num = 0;

	/* Un-apply */
	num -= o_ptr->number;

	/* Change the number */
	o_ptr->number += num;
}

/*
 * Optimize an item on the floor (destroy "empty" items)
 */
void floor_item_optimize(int item)
{
	object_type *o_ptr = &o_list[item];

	/* Paranoia -- be sure it exists */
	if (!o_ptr->k_idx) return;

	/* Only optimize empty items */
	if (o_ptr->number) return;

	/* Delete the object */
	delete_object_idx(item);
}

/*
 * Check if we have space for an item in the pack without overflow
 */
bool inven_carry_okay(const object_type *o_ptr)
{
	/* Empty slot? */
	if (p_ptr->inven_cnt < INVEN_PACK) return TRUE;

	/* Check if it can stack */
	if (inven_stack_okay(o_ptr)) return TRUE;

	/* Nope */
	return FALSE;
}

/*
 * Check to see if an item is stackable in the inventory
 */
bool inven_stack_okay(const object_type *o_ptr)
{
	int j;

	/* Similar slot? */
	for (j = 0; j < INVEN_PACK; j++)
	{
		object_type *j_ptr = &inventory[j];

		/* Skip non-objects */
		if (!j_ptr->k_idx) continue;

		/* Check if the two items can be combined */
		if (object_similar(j_ptr, o_ptr)) return (TRUE);
	}

	/* Nope */
	return (FALSE);
}

/*
 * Add an item to the players inventory, and return the slot used.
 *
 * If the new item can combine with an existing item in the inventory,
 * it will do so, using "object_similar()" and "object_absorb()", else,
 * the item will be placed into the "proper" location in the inventory.
 *
 * This function can be used to "over-fill" the player's pack, but only
 * once, and such an action must trigger the "overflow" code immediately.
 * Note that when the pack is being "over-filled", the new item must be
 * placed into the "overflow" slot, and the "overflow" must take place
 * before the pack is reordered, but (optionally) after the pack is
 * combined.  This may be tricky.  See "dungeon.c" for info.
 *
 * Note that this code must remove any location/stack information
 * from the object once it is placed into the inventory.
 */
s16b inven_carry(object_type *o_ptr)
{
	int i, j, k;
	int n = -1;

	object_type *j_ptr;

	/* Apply an autoinscription */
	apply_autoinscription(o_ptr);

	/* Check for combining */
	for (j = 0; j < INVEN_PACK; j++)
	{
		j_ptr = &inventory[j];

		/* Skip non-objects */
		if (!j_ptr->k_idx) continue;

		/* Hack -- track last item */
		n = j;

		/* Check if the two items can be combined */
		if (object_similar(j_ptr, o_ptr))
		{
			/* Combine the items */
			object_absorb(j_ptr, o_ptr);

			/* Increase the weight */
			p_ptr->total_weight += (o_ptr->number * o_ptr->weight);

			/* Recalculate bonuses */
			p_ptr->update |= (PU_BONUS);

			/* Redraw stuff */
			p_ptr->redraw |= (PR_INVEN);

			/* Success */
			return (j);
		}
	}

	/* Paranoia */
	if (p_ptr->inven_cnt > INVEN_PACK) return (-1);

	/* Find an empty slot */
	for (j = 0; j <= INVEN_PACK; j++)
	{
		j_ptr = &inventory[j];

		/* Use it if found */
		if (!j_ptr->k_idx) break;
	}

	/* Use that slot */
	i = j;

	/* Reorder the pack */
	if (i < INVEN_PACK)
	{
		s32b o_value, j_value;

		/* Get the "value" of the item */
		o_value = object_value(o_ptr, 1);

		/* Scan every occupied slot */
		for (j = 0; j < INVEN_PACK; j++)
		{
			j_ptr = &inventory[j];

			/* Use empty slots */
			if (!j_ptr->k_idx) break;

			/* Hack -- readable books always come first */
			if ((o_ptr->tval == cp_ptr->spell_book) &&
			    (j_ptr->tval != cp_ptr->spell_book)) break;
			if ((j_ptr->tval == cp_ptr->spell_book) &&
			    (o_ptr->tval != cp_ptr->spell_book)) continue;

			/* Objects sort by decreasing type */
			if (o_ptr->tval > j_ptr->tval) break;
			if (o_ptr->tval < j_ptr->tval) continue;

			/* Non-aware (flavored) items always come last */
			if (!object_aware_p(o_ptr)) continue;
			if (!object_aware_p(j_ptr)) break;

			/* Objects sort by increasing sval */
			if (o_ptr->sval < j_ptr->sval) break;
			if (o_ptr->sval > j_ptr->sval) continue;

			/* Unidentified objects always come last */
			if (!object_known_p(o_ptr)) continue;
			if (!object_known_p(j_ptr)) break;

			/* Lites sort by decreasing fuel */
			if (o_ptr->tval == TV_LITE)
			{
				if (o_ptr->pval > j_ptr->pval) break;
				if (o_ptr->pval < j_ptr->pval) continue;
			}

			/* Determine the "value" of the pack item */
			j_value = object_value(j_ptr, 1);

			/* Objects sort by decreasing value */
			if (o_value > j_value) break;
			if (o_value < j_value) continue;
		}

		/* Use that slot */
		i = j;

		/* Slide objects */
		for (k = n; k >= i; k--)
		{
			/* Hack -- Slide the item */
			object_copy(&inventory[k+1], &inventory[k]);
		}

		/* Wipe the empty slot */
		object_wipe(&inventory[i]);
	}

	/* Copy the item */
	object_copy(&inventory[i], o_ptr);

	/* Get the new object */
	j_ptr = &inventory[i];

	/* Forget stack */
	j_ptr->next_o_idx = 0;

	/* Forget monster */
	j_ptr->held_m_idx = 0;

	/* Forget location */
	j_ptr->iy = j_ptr->ix = 0;

	/* No longer marked */
	j_ptr->marked = FALSE;

	/* Increase the weight */
	p_ptr->total_weight += (j_ptr->number * j_ptr->weight);

	/* Count the items */
	p_ptr->inven_cnt++;

	/* Recalculate bonuses */
	p_ptr->update |= (PU_BONUS);

	/* Combine and Reorder pack */
	p_ptr->notice |= (PN_COMBINE | PN_REORDER);

	/* Redraw stuff */
	p_ptr->redraw |= (PR_INVEN);

	/* Return the slot */
	return (i);
}

/*
 * Take off (some of) a non-cursed equipment item
 *
 * Note that only one item at a time can be wielded per slot.
 *
 * Note that taking off an item when "full" may cause that item
 * to fall to the ground.
 *
 * Return the inventory slot into which the item is placed.
 */
s16b inven_takeoff(int item, int amt)
{
	int slot;

	object_type *o_ptr;

	object_type *i_ptr;
	object_type object_type_body;

	const _TCHAR *act;

	_TCHAR o_name[80];

	/* Get the item to take off */
	o_ptr = &inventory[item];

	/* Paranoia */
	if (amt <= 0) return (-1);

	/* Verify */
	if (amt > o_ptr->number) amt = o_ptr->number;

	/* Get local object */
	i_ptr = &object_type_body;

	/* Obtain a local object */
	object_copy(i_ptr, o_ptr);

	/* Modify quantity */
	i_ptr->number = amt;

	/* Describe the object */
	object_desc(o_name, _countof(o_name), i_ptr, TRUE, ODESC_FULL);

	/* Took off weapon */
	if (item == INVEN_WIELD)
	{
		act = __T("You were wielding");
	}

	/* Took off bow */
	else if (item == INVEN_BOW)
	{
		act = __T("You were holding");
	}

	/* Took off light */
	else if (item == INVEN_LITE)
	{
		act = __T("You were holding");
	}

	/* Took off something */
	else
	{
		act = __T("You were wearing");
	}

	/* Modify, Optimize */
	inven_item_increase(item, -amt);
	inven_item_optimize(item);

	/* Carry the object */
	slot = inven_carry(i_ptr);

	/* Message */
	sound(MSG_WIELD);
	msg_format(__T("%s %s (%c)."), act, o_name, index_to_label(slot));

	p_ptr->notice |= PN_SQUELCH;

	/* Return slot */
	return (slot);
}

/*
 * Drop (some of) a non-cursed inventory/equipment item
 *
 * The object will be dropped "near" the current location
 */
void inven_drop(int item, int amt)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	object_type *o_ptr;

	object_type *i_ptr;
	object_type object_type_body;

	_TCHAR o_name[80];

	/* Get the original object */
	o_ptr = &inventory[item];

	/* Error check */
	if (amt <= 0) return;

	/* Not too many */
	if (amt > o_ptr->number) amt = o_ptr->number;

	/* Take off equipment */
	if (item >= INVEN_WIELD)
	{
		/* Take off first */
		item = inven_takeoff(item, amt);

		/* Get the original object */
		o_ptr = &inventory[item];
	}

	/* Get local object */
	i_ptr = &object_type_body;

	/* Obtain local object */
	object_copy(i_ptr, o_ptr);

	/* Distribute charges of wands, staves, or rods */
	distribute_charges(o_ptr, i_ptr, amt);

	/* Modify quantity */
	i_ptr->number = amt;

	/* Describe local object */
	object_desc(o_name, _countof(o_name), i_ptr, TRUE, ODESC_FULL);

	/* Message */
	msg_format(__T("You drop %s (%c)."), o_name, index_to_label(item));

	/* Drop it near the player */
	drop_near(i_ptr, 0, py, px);

	/* Modify, Describe, Optimize */
	inven_item_increase(item, -amt);
	inven_item_describe(item);
	inven_item_optimize(item);
}

/*
 * Combine items in the pack
 * Also "pick up" any gold in the inventory by accident
 *
 * Note special handling of the "overflow" slot
 */
void combine_pack(void)
{
	int i, j, k;

	object_type *o_ptr;
	object_type *j_ptr;

	bool flag = FALSE;

	/* Combine the pack (backwards) */
	for (i = INVEN_PACK; i > 0; i--)
	{
		bool slide = FALSE;

		/* Get the item */
		o_ptr = &inventory[i];

		/* Skip empty items */
		if (!o_ptr->k_idx) continue;

		/* Absorb gold */
		if (o_ptr->tval == TV_GOLD)
		{
			/* Count the gold */
			slide = TRUE;
			p_ptr->au += o_ptr->pval;
		}

		/* Scan the items above that item */
		else for (j = 0; j < i; j++)
		{
			/* Get the item */
			j_ptr = &inventory[j];

			/* Skip empty items */
			if (!j_ptr->k_idx) continue;

			/* Can we drop "o_ptr" onto "j_ptr"? */
			if (object_similar(j_ptr, o_ptr))
			{
				/* Take note */
				flag = slide = TRUE;

				/* Add together the item counts */
				object_absorb(j_ptr, o_ptr);

				break;
			}
		}

		/* Compact the inventory */
		if (slide)
		{
			/* One object is gone */
			p_ptr->inven_cnt--;

			/* Slide everything down */
			for (k = i; k < INVEN_PACK; k++)
			{
				/* Hack -- slide object */
				COPY(&inventory[k], &inventory[k+1], object_type);
			}

			/* Hack -- wipe hole */
			object_wipe(&inventory[k]);

			/* Redraw stuff */
			p_ptr->redraw |= (PR_INVEN);
		}
	}

	/* Message */
	if (flag) msg_print(__T("You combine some items in your pack."));
}

/*
 * Reorder items in the pack
 *
 * Note special handling of the "overflow" slot
 */
void reorder_pack(void)
{
	int i, j, k;

	s32b o_value;
	s32b j_value;

	object_type *o_ptr;
	object_type *j_ptr;

	object_type *i_ptr;
	object_type object_type_body;

	bool flag = FALSE;

	/* Re-order the pack (forwards) */
	for (i = 0; i < INVEN_PACK; i++)
	{
		/* Get the item */
		o_ptr = &inventory[i];

		/* Skip empty slots */
		if (!o_ptr->k_idx) continue;

		/* Get the "value" of the item */
		o_value = object_value(o_ptr, 1);

		/* Scan every occupied slot */
		for (j = 0; j < INVEN_PACK; j++)
		{
			/* Get the item already there */
			j_ptr = &inventory[j];

			/* Use empty slots */
			if (!j_ptr->k_idx) break;

			/* Hack -- readable books always come first */
			if ((o_ptr->tval == cp_ptr->spell_book) &&
			    (j_ptr->tval != cp_ptr->spell_book)) break;
			if ((j_ptr->tval == cp_ptr->spell_book) &&
			    (o_ptr->tval != cp_ptr->spell_book)) continue;

			/* Objects sort by decreasing type */
			if (o_ptr->tval > j_ptr->tval) break;
			if (o_ptr->tval < j_ptr->tval) continue;

			/* Non-aware (flavored) items always come last */
			if (!object_aware_p(o_ptr)) continue;
			if (!object_aware_p(j_ptr)) break;

			/* Objects sort by increasing sval */
			if (o_ptr->sval < j_ptr->sval) break;
			if (o_ptr->sval > j_ptr->sval) continue;

			/* Unidentified objects always come last */
			if (!object_known_p(o_ptr)) continue;
			if (!object_known_p(j_ptr)) break;

			/* Lites sort by decreasing fuel */
			if (o_ptr->tval == TV_LITE)
			{
				if (o_ptr->pval > j_ptr->pval) break;
				if (o_ptr->pval < j_ptr->pval) continue;
			}

			/* Determine the "value" of the pack item */
			j_value = object_value(j_ptr, 1);

			/* Objects sort by decreasing value */
			if (o_value > j_value) break;
			if (o_value < j_value) continue;
		}

		/* Never move down */
		if (j >= i) continue;

		/* Take note */
		flag = TRUE;

		/* Get local object */
		i_ptr = &object_type_body;

		/* Save a copy of the moving item */
		object_copy(i_ptr, &inventory[i]);

		/* Slide the objects */
		for (k = i; k > j; k--)
		{
			/* Slide the item */
			object_copy(&inventory[k], &inventory[k-1]);
		}

		/* Insert the moving item */
		object_copy(&inventory[j], i_ptr);

		/* Redraw stuff */
		p_ptr->redraw |= (PR_INVEN);
	}

	/* Message */
	if (flag)  
 	{ 
 		msg_print(__T("You reorder some items in your pack."));
 	
 		/* Stop "repeat last command" from working. */ 
 		cmd_disable_repeat(); 
 	} 
}

/*
 * Distribute charges of rods, staves, or wands.
 *
 * o_ptr = source item
 * q_ptr = target item, must be of the same type as o_ptr
 * amt   = number of items that are transfered
 */
void distribute_charges(object_type *o_ptr, object_type *q_ptr, int amt)
{
	int max_time;

	/*
	 * Hack -- If rods, staves, or wands are dropped, the total maximum
	 * timeout or charges need to be allocated between the two stacks.
	 * If all the items are being dropped, it makes for a neater message
	 * to leave the original stack's pval alone. -LM-
	 */
	if ((o_ptr->tval == TV_WAND) ||
	    (o_ptr->tval == TV_STAFF))
	{
		q_ptr->pval = o_ptr->pval * amt / o_ptr->number;

		if (amt < o_ptr->number) o_ptr->pval -= q_ptr->pval;
	}

	/*
	 * Hack -- Rods also need to have their timeouts distributed.
	 *
	 * The dropped stack will accept all time remaining to charge up to
	 * its maximum.
	 */
	if (o_ptr->tval == TV_ROD)
	{
		max_time = k_info[o_ptr->k_idx].time_base * amt;

		if (o_ptr->timeout > max_time)
			q_ptr->timeout = max_time;
		else
			q_ptr->timeout = o_ptr->timeout;

		if (amt < o_ptr->number)
			o_ptr->timeout -= q_ptr->timeout;
	}
}

void reduce_charges(object_type *o_ptr, int amt)
{
	/*
	 * Hack -- If rods or wand are destroyed, the total maximum timeout or
	 * charges of the stack needs to be reduced, unless all the items are
	 * being destroyed. -LM-
	 */
	if (((o_ptr->tval == TV_WAND) ||
	     (o_ptr->tval == TV_STAFF)) &&
	    (amt < o_ptr->number))
	{
		o_ptr->pval -= o_ptr->pval * amt / o_ptr->number;
	}

	if ((o_ptr->tval == TV_ROD) &&
	    (amt < o_ptr->number))
	{
		o_ptr->timeout -= o_ptr->timeout * amt / o_ptr->number;
	}
}

/*
 * Looks if "inscrip" is present on the given object.
 */
unsigned check_for_inscrip(const object_type *o_ptr, const _TCHAR *inscrip)
{
	unsigned i = 0;
	const _TCHAR *s;

	if (!o_ptr->note) return 0;

	s = quark_str(o_ptr->note);

	do
	{
		s = _tcsstr(s, inscrip);
		if (!s) break;

		i++;
		s++;
	}
	while (s);

	return i;
}

/*** Object kind lookup functions ***/

/**
 * Return the k_idx of the object kind with the given `tval` and `sval`, or 0.
 */
int lookup_kind(int tval, int sval)
{
	int k;

	/* Look for it */
	for (k = 1; k < z_info->k_max; k++)
	{
		object_kind *k_ptr = &k_info[k];

		/* Found a match */
		if ((k_ptr->tval == tval) && (k_ptr->sval == sval)) return (k);
	}

	/* Failure */
	msg_format(__T("No object (%s,%d)"), tval_find_name(tval), tval, sval);
	return 0;
}

/**
 * Find the tval and sval of object kind `k_idx`, and return via the pointers
 * `tval` and `sval`.
 */
bool lookup_reverse(s16b k_idx, int *tval, int *sval)
{
	object_kind *k_ptr;

	/* Validate k_idx */
	if ((k_idx < 1) || (k_idx > z_info->k_max))
		return FALSE;

	/* Get pointer */
	k_ptr = &k_info[k_idx];
	*tval = k_ptr->tval;
	*sval = k_ptr->sval;

	/* Done */
	return TRUE;
}

/*** Textual<->numeric conversion ***/

/**
 * List of { tval, name } pairs.
 */
static const grouper tval_names[] =
{
	{ TV_SKELETON,    L"skeleton" },
	{ TV_BOTTLE,      L"bottle" },
	{ TV_JUNK,        L"junk" },
	{ TV_SPIKE,       L"spike" },
	{ TV_CHEST,       L"chest" },
	{ TV_SHOT,        L"shot" },
	{ TV_ARROW,       L"arrow" },
	{ TV_BOLT,        L"bolt" },
	{ TV_BOW,         L"bow" },
	{ TV_DIGGING,     L"digger" },
	{ TV_HAFTED,      L"hafted" },
	{ TV_POLEARM,     L"polearm" },
	{ TV_SWORD,       L"sword" },
	{ TV_BOOTS,       L"boots" },
	{ TV_GLOVES,      L"gloves" },
	{ TV_HELM,        L"helm" },
	{ TV_CROWN,       L"crown" },
	{ TV_SHIELD,      L"shield" },
	{ TV_CLOAK,       L"cloak" },
	{ TV_SOFT_ARMOR,  L"soft armor" },
	{ TV_SOFT_ARMOR,  L"soft armour" },
	{ TV_HARD_ARMOR,  L"hard armor" },
	{ TV_HARD_ARMOR,  L"hard armour" },
	{ TV_DRAG_ARMOR,  L"dragon armor" },
	{ TV_DRAG_ARMOR,  L"dragon armour" },
	{ TV_LITE,        L"light" },
	{ TV_AMULET,      L"amulet" },
	{ TV_RING,        L"ring" },
	{ TV_STAFF,       L"staff" },
	{ TV_WAND,        L"wand" },
	{ TV_ROD,         L"rod" },
	{ TV_SCROLL,      L"scroll" },
	{ TV_POTION,      L"potion" },
	{ TV_FLASK,       L"flask" },
	{ TV_FOOD,        L"food" },
	{ TV_MAGIC_BOOK,  L"magic book" },
	{ TV_PRAYER_BOOK, L"prayer book" },
	{ TV_GOLD,        L"gold" },
};

/**
 * Return the k_idx of the object kind with the given `tval` and name `name`.
 */
int lookup_name(int tval, const _TCHAR *name)
{
	int k;

	/* Look for it */
	for (k = 1; k < z_info->k_max; k++)
	{
		object_kind *k_ptr = &k_info[k];
		const _TCHAR *nm = k_name + k_ptr->name;

		if (*nm == '&' && *(nm+1))
			nm += 2;

		/* Found a match */
		if (k_ptr->tval == tval && !_tcscmp(name, nm))
			return k;
	}

	msg_format(__T("No object (\"%s\",\"%s\")"), tval_find_name(tval), name);
	return -1;
}

/**
 * Return the a_idx of the artifact with the given name
 */
int lookup_artifact_name(const _TCHAR *name)
{
	int i;
	
	/* Look for it */
	for (i = 1; i < z_info->a_max; i++)
	{
		artifact_type *a_ptr = &a_info[i];
		const _TCHAR *nm = a_name + a_ptr->name;
		
		/* Found a match */
		if (streq(name, nm))
			return i;
		
	} 
	
	return -1;
}

/**
 * Return the numeric sval of the object kind with the given `tval` and name `name`.
 */
int lookup_sval(int tval, const _TCHAR *name)
{
	int k;

	/* Look for it */
	for (k = 1; k < z_info->k_max; k++)
	{
		object_kind *k_ptr = &k_info[k];
		const _TCHAR *nm = k_name + k_ptr->name;

		if (*nm == '&' && *(nm+1))
			nm += 2;

		/* Found a match */
		if (k_ptr->tval == tval && !_tcscmp(name, nm))
			return k_ptr->sval;
	}

	msg_format(__T("No object (\"%s\",\"%s\")"), tval_find_name(tval), name);
	return -1;
}

/**
 * Returns the numeric equivalent tval of the textual tval `name`.
 */
int tval_find_idx(const _TCHAR *name)
{
	size_t i = 0;

	for (i = 0; i < N_ELEMENTS(tval_names); i++)
	{
		if (!my_stricmp(name, tval_names[i].name))
			return tval_names[i].tval;
	}

	return -1;
}

/**
 * Returns the textual equivalent tval of the numeric tval `name`.
 */
const _TCHAR *tval_find_name(int tval)
{
	size_t i = 0;

	for (i = 0; i < N_ELEMENTS(tval_names); i++)
	{
		if (tval == tval_names[i].tval)
			return tval_names[i].name;
	}

	return L"unknown";
}

/**
 * Sort comparator for objects using only tval and sval.
 * -1 if o1 should be first
 *  1 if o2 should be first
 *  0 if it doesn't matter
 */
static int compare_types(const object_type *o1, const object_type *o2)
{
	if (o1->tval == o2->tval)
		return CMP(o1->sval, o2->sval);
	else
		return CMP(o1->tval, o2->tval);
}

/* some handy macros for sorting */
#define object_is_known_artifact(o) (artifact_p(o) && object_known_p(o))
#define object_is_worthless(o) (object_value(o, 1) == 0)

/**
 * Sort comparator for objects
 * -1 if o1 should be first
 *  1 if o2 should be first
 *  0 if it doesn't matter
 *
 * The sort order is designed with the "list items" command in mind.
 */
static int compare_items(const object_type *o1, const object_type *o2)
{
	/* known artifacts will sort first */
	if (object_is_known_artifact(o1) && object_is_known_artifact(o2))
		return compare_types(o1, o2);
	if (object_is_known_artifact(o1)) return -1;
	if (object_is_known_artifact(o2)) return 1;

	/* unknown objects will sort next */
	if (!object_aware_p(o1) && !object_aware_p(o2))
		return compare_types(o1, o2);
	if (!object_aware_p(o1)) return -1;
	if (!object_aware_p(o2)) return 1;

	/* if only one of them is worthless, the other comes first */
	if (object_is_worthless(o1) && !object_is_worthless(o2)) return 1;
	if (!object_is_worthless(o1) && object_is_worthless(o2)) return -1;

	/* otherwise, just compare tvals and svals */
	/* NOTE: arguably there could be a better order than this */
	return compare_types(o1, o2);
}

/*
 * Display visible items, similar to display_monlist
 */
void display_itemlist(void)
{
	int max;
	int mx, my;
	unsigned num;
	int line = 1, x = 0;
	int cur_x;
	unsigned i;
	unsigned disp_count = 0;
	byte a;
	char c;

	object_type *types[MAX_ITEMLIST];
	int counts[MAX_ITEMLIST];
	unsigned counter = 0;

	int dungeon_hgt = p_ptr->depth == 0 ? TOWN_HGT : DUNGEON_HGT;
	int dungeon_wid = p_ptr->depth == 0 ? TOWN_WID : DUNGEON_WID;

	byte attr;
	_TCHAR buf[80];
	_TCHAR langbuf[200];

	int floor_list[MAX_FLOOR_STACK];

	/* Clear the term if in a subwindow, set x otherwise */
	if (Term != angband_term[0])
	{
		clear_from(0);
		max = Term->hgt - 1;
	}
	else
	{
		x = 13;
		max = Term->hgt - 2;
	}
	/* Look at each square of the dungeon for items */
	for (my = 0; my < dungeon_hgt; my++)
	{
		for (mx = 0; mx < dungeon_wid; mx++)
		{
			num = scan_floor(floor_list, MAX_FLOOR_STACK, my, mx, 0x02);

			/* Iterate over all the items found on this square */
			for (i = 0; i < num; i++)
			{
				object_type *o_ptr = &o_list[floor_list[i]];
				unsigned j;

				/* Skip gold/squelched */
				if (o_ptr->tval == TV_GOLD || squelch_item_ok(o_ptr))
					continue;

				/* See if we've already seen a similar item; if so, just add */
				/* to its count */
				for (j = 0; j < counter; j++)
				{
					if (object_similar(o_ptr, types[j]))
					{
						counts[j] += o_ptr->number;
						break;
					}
				}
				/* We saw a new item. So insert it at the end of the list and */
				/* then sort it forward using compare_items(). The types list */
				/* is always kept sorted. */
				if (j == counter)
				{
					types[counter] = o_ptr;
					counts[counter] = o_ptr->number;

					while (j > 0 && compare_items(types[j - 1], types[j]) > 0)
					{
						object_type *tmp_o = types[j - 1];
						int tmpcount;

						types[j - 1] = types[j];
						types[j] = tmp_o;
						tmpcount = counts[j - 1];
						counts[j - 1] = counts[j];
						counts[j] = tmpcount;
						j--;
					}
					counter++;
				}
			}
		}
	}
	/* Note no visible items */
	if (!counter)
	{
		/* Clear display and print note */
		c_prt(TERM_SLATE, L"You see no items.", 0, 0);
		if (Term == angband_term[0])
			Term_addstr(-1, TERM_WHITE, LS(__T("  (Press any key to continue.)"),
			spaceme(langbuf,__T("  (何かキーを押してください)"))));
		/* Done */
		return;
	}
	else
	{
		/* Reprint Message */
		prt(format(__T("You can see %d item%s:"),
				   counter, (counter > 1 ? L"s" : __T(""))), 0, 0);
	}
	for (i = 0; i < counter; i++)
	{
		/* o_name will hold the object_desc() name for the object. */
		/* o_desc will also need to put a (x4) behind it. */
		/* can there be more than 999 stackable items on a level? */
		_TCHAR o_name[80];
		_TCHAR o_desc[86];

		object_type *o_ptr = types[i];

		/* We shouldn't list coins or squelched items */
		if (o_ptr->tval == TV_GOLD || squelch_item_ok(o_ptr))
			continue;

		object_desc(o_name, _countof(o_name), o_ptr, FALSE, ODESC_FULL);
		if (counts[i] > 1)
			strnfmt(o_desc, _countof(o_desc), L"%s (x%d)", o_name, counts[i]);
		else
			strnfmt(o_desc, _countof(o_desc), L"%s", o_name);

		/* Reset position */
		cur_x = x;

		/* See if we need to scroll or not */
		if (Term == angband_term[0] && (line == max) && disp_count != counter)
		{
			prt(__T("-- more --"), line, x);
			anykey();

			/* Clear the screen */
			for (line = 1; line <= max; line++)
				prt(__T(""), line, x);

			/* Reprint Message */
			prt(format(__T("You can see %d item%s:"),
					   counter, (counter > 1 ? L"s" : __T(""))), 0, 0);

			/* Reset */
			line = 1;
		}
		else if (line == max)
		{
			continue;
		}

		/* Note that the number of items actually displayed */
		disp_count++;

		if (artifact_p(o_ptr) && object_known_p(o_ptr))
			/* known artifact */
			attr = TERM_VIOLET;
		else if (!object_aware_p(o_ptr))
			/* unaware of kind */
			attr = TERM_RED;
		else if (object_value(o_ptr, 1) == 0)
			/* worthless */
			attr = TERM_SLATE;
		else
			/* default */
			attr = TERM_WHITE;

		a = object_kind_attr(o_ptr->k_idx);
		c = object_kind_char(o_ptr->k_idx);

		/* Display the pict */
		Term_putch(cur_x++, line, a, c);
		if (use_bigtile) Term_putch(cur_x++, line, 255, -1);
		Term_putch(cur_x++, line, TERM_WHITE, ' ');

		/* Print and bump line counter */
		c_prt(attr, o_desc, line, cur_x);
		line++;
	}

	if (disp_count != counter)
	{
		/* Print "and others" message if we've run out of space */
		strnfmt(buf, sizeof buf, L"  ...and %d others.", counter - disp_count);
		c_prt(TERM_WHITE, buf, line, x);
	}
	else
	{
		/* Otherwise clear a line at the end, for main-term display */
		prt(__T(""), line, x);
	}

	if (Term == angband_term[0])
		Term_addstr(-1, TERM_WHITE, LS(__T("  (Press any key to continue.)"),
			spaceme(langbuf,__T("  (何かキーを押してください)"))));
}

/* Basic tval testers */ 
bool obj_is_staff(const object_type *o_ptr)  { return o_ptr->tval == TV_STAFF; } 
bool obj_is_wand(const object_type *o_ptr)   { return o_ptr->tval == TV_WAND; } 
bool obj_is_rod(const object_type *o_ptr)    { return o_ptr->tval == TV_ROD; } 
bool obj_is_potion(const object_type *o_ptr) { return o_ptr->tval == TV_POTION; } 
bool obj_is_scroll(const object_type *o_ptr) { return o_ptr->tval == TV_SCROLL; } 
bool obj_is_food(const object_type *o_ptr)   { return o_ptr->tval == TV_FOOD; } 
 
/* Determine if an object is zappable */ 
bool obj_can_zap(const object_type *o_ptr) 
{ 
        const object_kind *k_ptr = &k_info[o_ptr->k_idx]; 
        if (o_ptr->tval != TV_ROD) return FALSE; 
 
        /* All still charging? */ 
        if (o_ptr->number <= (o_ptr->timeout + (k_ptr->time_base - 1)) / k_ptr->time_base) return FALSE; 
 
        /* Otherwise OK */ 
        return TRUE; 
} 
 
/* Determine if an object is activatable */ 
bool obj_is_activatable(const object_type *o_ptr)
{ 
        u32b f[OBJ_FLAG_N]; 
 
        /* Not known */ 
        if (!object_known_p(o_ptr)) 
			return (FALSE); 
 
        /* Extract the flags */ 
        object_flags(o_ptr, f); 
 
        /* Check activation flag */ 
        return (f[2] & TR2_ACTIVATE) ? TRUE : FALSE; 
} 
 
/* Determine if an object can be activated now */ 
bool obj_can_activate(const object_type *o_ptr) 
{ 
	if (obj_is_activatable(o_ptr)) 
	{ 
		/* Check the recharge */ 
		if (!o_ptr->timeout) 
			return TRUE; 
	} 
	return FALSE; 
} 

bool obj_can_refill(const object_type *o_ptr) 
{ 
        u32b f[OBJ_FLAG_N]; 
        const object_type *j_ptr = &inventory[INVEN_LITE]; 
 
        /* Get flags */ 
        object_flags(o_ptr, f); 
 
        if (j_ptr->sval == SV_LITE_LANTERN) 
        { 
                /* Flasks of oil are okay */ 
                if (o_ptr->tval == TV_FLASK) return (TRUE); 
        } 
 
        /* Non-empty, non-everburning sources are okay */ 
        if ((o_ptr->tval == TV_LITE) && 
            (o_ptr->sval == j_ptr->sval) && 
            (o_ptr->timeout > 0) && 
                !(f[2] & TR2_NO_FUEL)) 
        { 
                return (TRUE); 
        } 
 
        /* Assume not okay */ 
        return (FALSE); 
} 
 
 
bool obj_can_browse(const object_type *o_ptr) 
{ 
        if (o_ptr->tval != cp_ptr->spell_book) return FALSE; 
        return TRUE; 
} 
 
 
 
/* Can only take off non-cursed items */ 
bool obj_can_takeoff(const object_type *o_ptr) 
{ 
        return !cursed_p(o_ptr); 
} 
 
/* Can only put on wieldable items */ 
bool obj_can_wear(const object_type *o_ptr) 
{ 
        return (wield_slot(o_ptr) >= INVEN_WIELD); 
} 
 
/* Can has inscrip pls */ 
bool obj_has_inscrip(const object_type *o_ptr) 
{ 
        return (o_ptr->note ? TRUE : FALSE); 
} 
 
/*** Generic utility functions ***/ 
 
/* Get an o_ptr from an item number */ 
object_type *object_from_item_idx(int item) 
{ 
        if (item >= 0) 
                return &inventory[item]; 
        else 
                return &o_list[0 - item]; 
} 
 
 
/* 
 * Does the given object need to be aimed? 
 */  
bool obj_needs_aim(object_type *o_ptr) 
{ 
        int effect; 
 
        /* Figure out effect the object would use */ 
        if (o_ptr->name1) 
                effect = a_info[o_ptr->name1].effect; 
        else 
                effect = k_info[o_ptr->k_idx].effect; 
 
        /* If the effect needs aiming, or if the object type needs 
           aiming, this object needs aiming. */ 
        if (effect_aim(effect) || 
            (o_ptr->tval == TV_WAND) || 
            (o_ptr->tval == TV_ROD && !object_aware_p(o_ptr))) 
        { 
                return TRUE; 
        } 
        else 
        { 
                return FALSE; 
        } 
} 

/* 
 * Verify the "okayness" of a given item. 
 * 
 * The item can be negative to mean "item on floor". 
 */ 
bool get_item_okay(int item)
{ 
	/* Verify the item */ 
	return (item_tester_okay(object_from_item_idx(item))); 
} 
 
/* 
 * Get a list of "valid" item indexes. 
 * 
 * Fills item_list[] with items that are "okay" as defined by the 
 * current item_tester_hook, etc.  mode determines what combination of 
 * inventory, equipment and player's floor location should be used 
 * when drawing up the list. 
 * 
 * Returns the number of items placed into the list. 
 * 
 * Maximum space that can be used is [INVEN_TOTAL + MAX_FLOOR_STACK], 
 * though practically speaking much smaller numbers are likely. 
 */ 
int scan_items(int *item_list, size_t item_list_max, int mode) 
{ 
	bool use_inven = ((mode & USE_INVEN) ? TRUE : FALSE); 
	bool use_equip = ((mode & USE_EQUIP) ? TRUE : FALSE); 
	bool use_floor = ((mode & USE_FLOOR) ? TRUE : FALSE); 

	int floor_list[MAX_FLOOR_STACK]; 
	int floor_num; 

	u16b i; 
	u16b item_list_num = 0; 

	if (use_inven) 
	{ 
		for (i = 0; (i < INVEN_PACK) && (item_list_num < item_list_max); i++) 
		{ 
				if (get_item_okay(i)) 
						item_list[item_list_num++] = i; 
		} 
	} 
	if (use_equip) 
	{ 
		for (i = INVEN_WIELD; i < INVEN_TOTAL && item_list_num < item_list_max; i++) 
		{ 
				if (get_item_okay(i)) 
						item_list[item_list_num++] = i; 
		} 
	} 
	/* Scan all non-gold objects in the grid */ 
	if (use_floor) 
	{ 
		floor_num = scan_floor(floor_list, N_ELEMENTS(floor_list), p_ptr->py, p_ptr->px, 0x03); 

		for (i = 0; i < floor_num && item_list_num < item_list_max; i++) 
		{ 
			if (get_item_okay(-floor_list[i])) 
				item_list[item_list_num++] = -floor_list[i];
		} 
	} 
	/* Forget the item_tester_tval and item_tester_hook  restrictions */ 
	item_tester_tval = 0; 
	item_tester_hook = NULL; 

	return item_list_num; 
} 
 
/*  
 * Check if the given item is available for the player to use.  
 * 
 * 'mode' defines which areas we should look at, a la scan_items(). 
 */ 
bool item_is_available(int item, bool (*tester)(const object_type *), int mode) 
{ 
	int item_list[INVEN_TOTAL + MAX_FLOOR_STACK]; 
	int item_num; 
	int i; 

	item_tester_hook = tester; 
	item_tester_tval = 0; 
	item_num = scan_items(item_list, N_ELEMENTS(item_list), mode); 

	for (i = 0; i < item_num; i++) 
	{ 
		if (item_list[i] == item) 
				return TRUE; 
	} 

	return FALSE; 
}
