﻿/* テスト */
/*
 * File: wizard2.c
 * Purpose: Debug mode commands
 *
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This work is free software; you can redistribute it and/or modify it
 * under the terms of either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation, version 2, or
 *
 * b) the "Angband licence":
 *    This software may be copied and distributed for educational, research,
 *    and not for profit purposes provided that this copyright and statement
 *    are included in all such copies.  Other copyrights may also apply.
 */
extern "C"
{
#include "angband.h"
#include "wizard.h"
#include "cmds.h"
#include "object/tvalsval.h"
}

#ifdef ALLOW_DEBUG

extern bool get_item(int *cp, const _TCHAR *pmt, const _TCHAR *str, int mode);
extern void prt(const _TCHAR *str, int row, int col);
extern size_t strnfmt(_TCHAR *buf, size_t max, const _TCHAR *fmt, ...);
extern size_t object_desc(_TCHAR *buf, size_t max, const object_type *o_ptr, bool prefix, int mode);
extern void message_flush(void);
extern void msg_print(const _TCHAR *msg);
extern bool get_check(const _TCHAR *prompt);
extern void screen_save(void);
extern void screen_load(void);
extern void do_cmd_spoilers(void);

/*
 * This is a nice utility function; it determines if a (NULL-terminated)
 * string consists of only digits (starting with a non-zero digit).
 */
s16b get_idx_from_name(_TCHAR *s)
{
	_TCHAR *endptr = NULL;
	long l = _tcstol(s, &endptr, 10);
	return *endptr == 0 ? (s16b)l : 0;
}

/*
 * Hack -- quick debugging hook
 */
static void do_cmd_wiz_hack_ben(void)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int i, y, x;

	for (i = 0; i < MONSTER_FLOW_DEPTH; ++i)
	{
		/* Update map */
		for (y = Term->offset_y; y < Term->offset_y + SCREEN_HGT; y++)
		{
			for (x = Term->offset_x; x < Term->offset_x + SCREEN_WID; x++)
			{
				byte a = TERM_RED;

				if (!in_bounds_fully(y, x)) continue;

				/* Display proper cost */
				if (cave_cost[y][x] != i) continue;

				/* Reliability in yellow */
				if (cave_when[y][x] == cave_when[py][px])
				{
					a = TERM_YELLOW;
				}

				/* Display player/floors/walls */
				if ((y == py) && (x == px))
				{
					print_rel('@', a, y, x);
				}
				else if (cave_floor_bold(y, x))
				{
					print_rel('*', a, y, x);
				}
				else
				{
					print_rel('#', a, y, x);
				}
			}
		}
		/* Prompt */
		prt(format(__T("Depth %d: "), i), 0, 0);

		/* Get key */
		if (inkey() == ESCAPE) break;

		/* Redraw map */
		prt_map();
	}
	/* Done */
	prt(__T(""), 0, 0);

	/* Redraw map */
	prt_map();
}

/*
 * Output a long int in binary format.
 */
static void prt_binary(u32b flags, int row, int col)
{
	int i;
	u32b bitmask;

	/* Scan the flags */
	for (i = bitmask = 1; i <= 32; i++, bitmask *= 2)
	{
		/* Dump set bits */
		if (flags & bitmask)
		{
			Term_putch(col++, row, TERM_BLUE, '*');
		}
		else /* Dump unset bits */
		{
			Term_putch(col++, row, TERM_WHITE, '-');
		}
	}
}

/*
 * Hack -- Teleport to the target
 */
static void do_cmd_wiz_bamf(void)
{
	s16b x, y;

	/* Must have a target */
	if (!target_okay()) return;

	/* Teleport to the target */
	target_get(&x, &y);
	teleport_player_to(y, x);
}

/*
 * Aux function for "do_cmd_wiz_change()"
 */
static void do_cmd_wiz_change_aux(void)
{
	int i;

	int tmp_int;

	long tmp_long;

	_TCHAR tmp_val[160];

	_TCHAR ppp[80];

	/* Query the stats */
	for (i = 0; i < A_MAX; i++)
	{
		/* Prompt */
		strnfmt(ppp, _countof(ppp), L"%s (3-118): ", stat_names[i]);

		/* Default */
		strnfmt(tmp_val, _countof(tmp_val), L"%d", p_ptr->stat_max[i]);

		/* Query */
		if (!get_string(ppp, tmp_val, 4)) return;

		/* Extract */
		tmp_int = _tstoi(tmp_val);

		/* Verify */
		if (tmp_int > 18+100) tmp_int = 18+100;
		else if (tmp_int < 3) tmp_int = 3;

		/* Save it */
		p_ptr->stat_cur[i] = p_ptr->stat_max[i] = tmp_int;
	}
	/* Default */
	strnfmt(tmp_val, _countof(tmp_val), L"%ld", (long)(p_ptr->au));

	/* Query */
	if (!get_string(L"Gold: ", tmp_val, 10)) return;

	/* Extract */
	tmp_long = _tstol(tmp_val);

	/* Verify */
	if (tmp_long < 0) tmp_long = 0L;

	/* Save */
	p_ptr->au = tmp_long;

	/* Default */
	strnfmt(tmp_val, _countof(tmp_val), L"%ld", (long)(p_ptr->exp));

	/* Query */
	if (!get_string(L"Experience: ", tmp_val, 10)) return;

	/* Extract */
	tmp_long = _tstol(tmp_val);

	/* Verify */
	if (tmp_long < 0) tmp_long = 0L;

	/* Save */
	p_ptr->exp = tmp_long;

	/* Update */
	check_experience();

	/* Default */
	strnfmt(tmp_val, _countof(tmp_val), L"%ld", (long)(p_ptr->max_exp));

	/* Query */
	if (!get_string(L"Max Exp: ", tmp_val, 10)) return;

	/* Extract */
	tmp_long = _tstol(tmp_val);

	/* Verify */
	if (tmp_long < 0) tmp_long = 0L;

	/* Save */
	p_ptr->max_exp = tmp_long;

	/* Update */
	check_experience();
}

/*
 * Change various "permanent" player variables.
 */
static void do_cmd_wiz_change(void)
{
	/* Interact */
	do_cmd_wiz_change_aux();

	/* Redraw everything */
	do_cmd_redraw();
}

/*
 * Wizard routines for creating objects and modifying them
 *
 * This has been rewritten to make the whole procedure
 * of debugging objects much easier and more comfortable.
 *
 * Here are the low-level functions
 *
 * - wiz_display_item()
 *     display an item's debug-info
 * - wiz_create_itemtype()
 *     specify tval and sval (type and subtype of object)
 * - wiz_tweak_item()
 *     specify pval, +AC, +tohit, +todam
 *     Note that the wizard can leave this function anytime,
 *     thus accepting the default-values for the remaining values.
 *     pval comes first now, since it is most important.
 * - wiz_reroll_item()
 *     apply some magic to the item or turn it into an artifact.
 * - wiz_roll_item()
 *     Get some statistics about the rarity of an item:
 *     We create a lot of fake items and see if they are of the
 *     same type (tval and sval), then we compare pval and +AC.
 *     If the fake-item is better or equal it is counted.
 *     Note that cursed items that are better or equal (absolute values)
 *     are counted, too.
 *     HINT: This is *very* useful for balancing the game!
 * - wiz_quantity_item()
 *     change the quantity of an item, but be sane about it.
 *
 * And now the high-level functions
 * - do_cmd_wiz_play()
 *     play with an existing object
 * - wiz_create_item()
 *     create a new object
 *
 * Note -- You do not have to specify "pval" and other item-properties
 * directly. Just apply magic until you are satisfied with the item.
 *
 * Note -- For some items (such as wands, staffs, some rings, etc), you
 * must apply magic, or you will get "broken" or "uncharged" objects.
 *
 * Note -- Redefining artifacts via "do_cmd_wiz_play()" may destroy
 * the artifact.  Be careful.
 *
 * Hack -- this function will allow you to create multiple artifacts.
 * This "feature" may induce crashes or other nasty effects.
 */

/*
 * Display an item's properties
 */
static void wiz_display_item(const object_type *o_ptr)
{
	int j = 0;

	u32b f[OBJ_FLAG_N];

	_TCHAR buf[256];

	/* Extract the flags */
	object_flags(o_ptr, f);

	/* Clear screen */
	Term_clear();

	/* Describe fully */
	object_desc(buf, _countof(buf), o_ptr, TRUE, ODESC_FULL | ODESC_SPOIL);

	prt(buf, 2, j);

	prt(format(__T("kind = %-5d  level = %-4d  tval = %-5d  sval = %-5d"),
	           o_ptr->k_idx, k_info[o_ptr->k_idx].level,
	           o_ptr->tval, o_ptr->sval), 4, j);

	prt(format(__T("number = %-3d  wgt = %-6d  ac = %-5d    damage = %dd%d"),
	           o_ptr->number, o_ptr->weight,
	           o_ptr->ac, o_ptr->dd, o_ptr->ds), 5, j);

	prt(format(__T("pval = %-5d  toac = %-5d  tohit = %-4d  todam = %-4d"),
	           o_ptr->pval, o_ptr->to_a, o_ptr->to_h, o_ptr->to_d), 6, j);

	prt(format(__T("name1 = %-4d  name2 = %-4d  cost = %ld"),
	           o_ptr->name1, o_ptr->name2, (long)object_value(o_ptr, 1)), 7, j);

	prt(format(__T("ident = %04x  timeout = %-d"),
	           o_ptr->ident, o_ptr->timeout), 8, j);

	prt(__T("+------------FLAGS0------------+"), 10, j);
	prt(__T("AFFECT..........SLAY.......BRAND"), 11, j);
	prt(__T("                ae      xxxpaefc"), 12, j);
	prt(__T("siwdcc  ssidsasmnvudotgddduoclio"), 13, j);
	prt(__T("tnieoh  trnipthgiinmrrnrrmniierl"), 14, j);
	prt(__T("rtsxna..lcfgdkttmldncltggndsdced"), 15, j);
	prt_binary(f[0], 16, j);

	prt(__T("+------------FLAGS1------------+"), 17, j);
	prt(__T("SUST........IMM.RESIST........."), 18, j);
	prt(__T("            afecaefcpfldbc s n  "), 19, j);
	prt(__T("siwdcc      cilocliooeialoshnecd"), 20, j);
	prt(__T("tnieoh      irelierliatrnnnrethi"), 21, j);
	prt(__T("rtsxna......decddcedsrekdfddxhss"), 22, j);
	prt_binary(f[1], 23, j);

	prt(__T("+------------FLAGS2------------+"), 10, j+32);
	prt(__T("s   ts hn    tadiiii   aiehs  hp"), 11, j+32);
	prt(__T("lf  eefoo    egrgggg  bcnaih  vr"), 12, j+32);
	prt(__T("we  lerlf   ilgannnn  ltssdo  ym"), 13, j+32);
	prt(__T("da reiedu   merirrrr  eityew ccc"), 14, j+32);
	prt(__T("itlepnele   ppanaefc  svaktm uuu"), 15, j+32);
	prt(__T("ghigavail   aoveclio  saanyo rrr"), 16, j+32);
	prt(__T("seteticf    craxierl  etropd sss"), 17, j+32);
	prt(__T("trenhste    tttpdced  detwes eee"), 18, j+32);
	prt_binary(f[2], 19, j+32);
}

/*
 * A structure to hold a tval and its description
 */
typedef struct tval_desc
{
	int tval;
	const _TCHAR *desc;
} tval_desc;

/*
 * A list of tvals and their textual names
 */
static const tval_desc tvals[] =
{
	{ TV_SWORD,             L"Sword"                },
	{ TV_POLEARM,           L"Polearm"              },
	{ TV_HAFTED,            L"Hafted Weapon"        },
	{ TV_BOW,               L"Bow"                  },
	{ TV_ARROW,             L"Arrows"               },
	{ TV_BOLT,              L"Bolts"                },
	{ TV_SHOT,              L"Shots"                },
	{ TV_SHIELD,            L"Shield"               },
	{ TV_CROWN,             L"Crown"                },
	{ TV_HELM,              L"Helm"                 },
	{ TV_GLOVES,            L"Gloves"               },
	{ TV_BOOTS,             L"Boots"                },
	{ TV_CLOAK,             L"Cloak"                },
	{ TV_DRAG_ARMOR,        L"Dragon Scale Mail"    },
	{ TV_HARD_ARMOR,        L"Hard Armor"           },
	{ TV_SOFT_ARMOR,        L"Soft Armor"           },
	{ TV_RING,              L"Ring"                 },
	{ TV_AMULET,            L"Amulet"               },
	{ TV_LITE,              L"Lite"                 },
	{ TV_POTION,            L"Potion"               },
	{ TV_SCROLL,            L"Scroll"               },
	{ TV_WAND,              L"Wand"                 },
	{ TV_STAFF,             L"Staff"                },
	{ TV_ROD,               L"Rod"                  },
	{ TV_PRAYER_BOOK,       L"Priest Book"          },
	{ TV_MAGIC_BOOK,        L"Magic Book"           },
	{ TV_SPIKE,             L"Spikes"               },
	{ TV_DIGGING,           L"Digger"               },
	{ TV_CHEST,             L"Chest"                },
	{ TV_FOOD,              L"Food"                 },
	{ TV_FLASK,             L"Flask"                },
	{ TV_SKELETON,          L"Skeletons"            },
	{ TV_BOTTLE,            L"Empty bottle"         },
	{ TV_JUNK,              L"Junk"                 },
	{ TV_GOLD,              L"Gold"                 },
	{ 0,                    NULL                   }
};

/*
 * Get an object kind for creation (or zero)
 *
 * List up to 60 choices in three columns
 */
static int wiz_create_itemtype(void)
{
	int i, num, max_num;
	int col, row;
	int tval;

	const _TCHAR *tval_desc;
	_TCHAR ch;

	int choice[60];
	static const _TCHAR choice_name[] = __T("abcdefghijklmnopqrst")
	                                  __T("ABCDEFGHIJKLMNOPQRST")
	                                  __T("0123456789:;<=>?@%&*");
	const _TCHAR *cp;

	_TCHAR buf[160];

	/* Clear screen */
	Term_clear();

	/* Print all tval's and their descriptions */
	for (num = 0; (num < 60) && tvals[num].tval; num++)
	{
		row = 2 + (num % 20);
		col = 30 * (num / 20);
		ch  = choice_name[num];
		prt(format(__T("[%c] %s"), ch, tvals[num].desc), row, col);
	}
	/* We need to know the maximal possible tval_index */
	max_num = num;

	/* Choose! */
	if (!get_com(L"Get what type of object? ", &ch)) return (0);

	/* Analyze choice */
	num = -1;
	if ((cp = _tcschr(choice_name, ch)) != NULL)
		num = cp - choice_name;

	/* Bail out if choice is illegal */
	if ((num < 0) || (num >= max_num)) return (0);

	/* Base object type chosen, fill in tval */
	tval = tvals[num].tval;
	tval_desc = tvals[num].desc;

	/*** And now we go for k_idx ***/

	/* Clear screen */
	Term_clear();

	/* We have to search the whole itemlist. */
	for (num = 0, i = 1; (num < 60) && (i < z_info->k_max); i++)
	{
		object_kind *k_ptr = &k_info[i];

		/* Analyze matching items */
		if (k_ptr->tval == tval)
		{
			/* Hack -- Skip instant artifacts */
			if (k_ptr->flags[2] & (TR2_INSTA_ART)) continue;

			/* Prepare it */
			row = 2 + (num % 20);
			col = 30 * (num / 20);
			ch  = choice_name[num];

			/* Get the "name" of object "i" */
			object_kind_name(buf, sizeof buf, i, TRUE);

			/* Print it */
			prt(format(__T("[%c] %s"), ch, buf), row, col);

			/* Remember the object index */
			choice[num++] = i;
		}
	}
	/* Me need to know the maximal possible remembered object_index */
	max_num = num;

	/* Choose! */
	if (!get_com(format(__T("What Kind of %s? "), tval_desc), &ch)) return (0);

	/* Analyze choice */
	num = -1;
	if ((cp = _tcschr(choice_name, ch)) != NULL)
		num = cp - choice_name;

	/* Bail out if choice is "illegal" */
	if ((num < 0) || (num >= max_num)) 
		return (0);

	/* And return successful */
	return (choice[num]);
}

/*
 * Tweak an item
 */
static void wiz_tweak_item(object_type *o_ptr)
{
	const _TCHAR *p;
	_TCHAR tmp_val[80];

	/* Hack -- leave artifacts alone */
	if (artifact_p(o_ptr)) 
		return;

	p = L"Enter new 'pval' setting: ";
	strnfmt(tmp_val, _countof(tmp_val), L"%d", o_ptr->pval);
	if (!get_string(p, tmp_val, 6)) 
		return;
	o_ptr->pval = _tstoi(tmp_val);
	wiz_display_item(o_ptr);

	p = L"Enter new 'to_a' setting: ";
	strnfmt(tmp_val, _countof(tmp_val), L"%d", o_ptr->to_a);
	if (!get_string(p, tmp_val, 6)) 
		return;
	o_ptr->to_a = _tstoi(tmp_val);
	wiz_display_item(o_ptr);

	p = L"Enter new 'to_h' setting: ";
	strnfmt(tmp_val, _countof(tmp_val), L"%d", o_ptr->to_h);
	if (!get_string(p, tmp_val, 6)) 
		return;
	o_ptr->to_h = _tstoi(tmp_val);
	wiz_display_item(o_ptr);

	p = L"Enter new 'to_d' setting: ";
	strnfmt(tmp_val, _countof(tmp_val), L"%d", o_ptr->to_d);
	if (!get_string(p, tmp_val, 6)) 
		return;
	o_ptr->to_d = _tstoi(tmp_val);
	wiz_display_item(o_ptr);
}

/*
 * Apply magic to an item or turn it into an artifact. -Bernd-
 */
static void wiz_reroll_item(object_type *o_ptr)
{
	object_type *i_ptr;
	object_type object_type_body;

	_TCHAR ch;

	bool changed = FALSE;

	/* Hack -- leave artifacts alone */
	if (artifact_p(o_ptr)) 
		return;

	/* Get local object */
	i_ptr = &object_type_body;

	/* Copy the object */
	object_copy(i_ptr, o_ptr);

	/* Main loop. Ask for magification and artifactification */
	while (TRUE)
	{
		/* Display full item debug information */
		wiz_display_item(i_ptr);

		/* Ask wizard what to do. */
		if (!get_com(L"[a]ccept, [n]ormal, [g]ood, [e]xcellent? ", &ch))
			break;

		/* Create/change it! */
		if (ch == 'A' || ch == 'a')
		{
			changed = TRUE;
			break;
		}
		/* Apply normal magic, but first clear object */
		else if (ch == 'n' || ch == 'N')
		{
			object_prep(i_ptr, o_ptr->k_idx);
			apply_magic(i_ptr, p_ptr->depth, FALSE, FALSE, FALSE);
		}
		/* Apply good magic, but first clear object */
		else if (ch == 'g' || ch == 'g')
		{
			object_prep(i_ptr, o_ptr->k_idx);
			apply_magic(i_ptr, p_ptr->depth, FALSE, TRUE, FALSE);
		}
		/* Apply great magic, but first clear object */
		else if (ch == 'e' || ch == 'e')
		{
			object_prep(i_ptr, o_ptr->k_idx);
			apply_magic(i_ptr, p_ptr->depth, FALSE, TRUE, TRUE);
		}
	}
	/* Notice change */
	if (changed)
	{
		/* Mark as cheat */
		i_ptr->origin = ORIGIN_CHEAT;

		/* Restore the position information */
		i_ptr->iy = o_ptr->iy;
		i_ptr->ix = o_ptr->ix;
		i_ptr->next_o_idx = o_ptr->next_o_idx;
		i_ptr->marked = o_ptr->marked;

		/* Apply changes */
		object_copy(o_ptr, i_ptr);

		/* Recalculate bonuses */
		p_ptr->update |= (PU_BONUS);

		/* Combine / Reorder the pack (later) */
		p_ptr->notice |= (PN_COMBINE | PN_REORDER);

		/* Window stuff */
		p_ptr->redraw |= (PR_INVEN | PR_EQUIP );
	}
}

/*
 * Maximum number of rolls
 */
#define TEST_ROLL 100000

/*
 * Try to create an item again. Output some statistics.    -Bernd-
 *
 * The statistics are correct now.  We acquire a clean grid, and then
 * repeatedly place an object in this grid, copying it into an item
 * holder, and then deleting the object.  We fiddle with the artifact
 * counter flags to prevent weirdness.  We use the items to collect
 * statistics on item creation relative to the initial item.
 */
static void wiz_statistics(object_type *o_ptr, int level)
{
	long i, matches, better, worse, other;

	_TCHAR ch;
	const _TCHAR *quality;

	bool good, great;

	object_type *i_ptr;
	object_type object_type_body;

	const _TCHAR *q = L"Rolls: %ld, Matches: %ld, Better: %ld, Worse: %ld, Other: %ld";


	/* Mega-Hack -- allow multiple artifacts XXX XXX XXX */
	if (artifact_p(o_ptr)) a_info[o_ptr->name1].cur_num = 0;

	/* Interact */
	while (TRUE)
	{
		const _TCHAR *pmt = L"Roll for [n]ormal, [g]ood, or [e]xcellent treasure? ";

		/* Display item */
		wiz_display_item(o_ptr);

		/* Get choices */
		if (!get_com(pmt, &ch)) break;

		if (ch == 'n' || ch == 'N')
		{
			good = FALSE;
			great = FALSE;
			quality = L"normal";
		}
		else if (ch == 'g' || ch == 'G')
		{
			good = TRUE;
			great = FALSE;
			quality = L"good";
		}
		else if (ch == 'e' || ch == 'E')
		{
			good = TRUE;
			great = TRUE;
			quality = L"excellent";
		}
		else
		{
#if 0 /* unused */
			good = FALSE;
			great = FALSE;
#endif /* unused */
			break;
		}

		/* Let us know what we are doing */
		msg_format(__T("Creating a lot of %s items. Base level = %d."),
		           quality, p_ptr->depth);
		message_flush();

		/* Set counters to zero */
		matches = better = worse = other = 0;

		/* Let's rock and roll */
		for (i = 0; i <= TEST_ROLL; i++)
		{
			/* Output every few rolls */
			if ((i < 100) || (i % 100 == 0))
			{
				/* Do not wait */
				inkey_scan = SCAN_INSTANT;

				/* Allow interupt */
				if (inkey())
				{
					/* Flush */
					flush();

					/* Stop rolling */
					break;
				}
				/* Dump the stats */
				prt(format(q, i, matches, better, worse, other), 0, 0);
				Term_fresh();
			}
			/* Get local object */
			i_ptr = &object_type_body;

			/* Wipe the object */
			object_wipe(i_ptr);

			/* Create an object */
			make_object(i_ptr, level, good, great);

			/* Mega-Hack -- allow multiple artifacts XXX XXX XXX */
			if (artifact_p(i_ptr)) 
				a_info[i_ptr->name1].cur_num = 0;

			/* Test for the same tval and sval. */
			if ((o_ptr->tval) != (i_ptr->tval)) 
				continue;
			if ((o_ptr->sval) != (i_ptr->sval)) 
				continue;

			/* Check for match */
			if ((i_ptr->pval == o_ptr->pval) &&
			    (i_ptr->to_a == o_ptr->to_a) &&
			    (i_ptr->to_h == o_ptr->to_h) &&
			    (i_ptr->to_d == o_ptr->to_d))
			{
				matches++;
			}
			/* Check for better */
			else if ((i_ptr->pval >= o_ptr->pval) &&
			         (i_ptr->to_a >= o_ptr->to_a) &&
			         (i_ptr->to_h >= o_ptr->to_h) &&
			         (i_ptr->to_d >= o_ptr->to_d))
			{
				better++;
			}
			/* Check for worse */
			else if ((i_ptr->pval <= o_ptr->pval) &&
			         (i_ptr->to_a <= o_ptr->to_a) &&
			         (i_ptr->to_h <= o_ptr->to_h) &&
			         (i_ptr->to_d <= o_ptr->to_d))
			{
				worse++;
			}
			/* Assume different */
			else
			{
				other++;
			}
		}
		/* Final dump */
		msg_format(q, i, matches, better, worse, other);
		message_flush();
	}
	/* Hack -- Normally only make a single artifact */
	if (artifact_p(o_ptr)) a_info[o_ptr->name1].cur_num = 1;
}


/*
 * Change the quantity of a the item
 */
static void wiz_quantity_item(object_type *o_ptr, bool carried)
{
	int tmp_int;

	_TCHAR tmp_val[3];

	/* Never duplicate artifacts */
	if (artifact_p(o_ptr)) 
		return;

	/* Default */
	strnfmt(tmp_val, _countof(tmp_val), L"%d", o_ptr->number);

	/* Query */
	if (get_string(L"Quantity: ", tmp_val, 3))
	{
		/* Extract */
		tmp_int = _tstoi(tmp_val);

		/* Paranoia */
		if (tmp_int < 1) tmp_int = 1;
		if (tmp_int > 99) tmp_int = 99;

		/* Adjust total weight being carried */
		if (carried)
		{
			/* Remove the weight of the old number of objects */
			p_ptr->total_weight -= (o_ptr->number * o_ptr->weight);

			/* Add the weight of the new number of objects */
			p_ptr->total_weight += (tmp_int * o_ptr->weight);
		}
		/* Adjust charge for rods */
		if (o_ptr->tval == TV_ROD)
		{
			o_ptr->pval = (o_ptr->pval / o_ptr->number) * tmp_int;
		}
		/* Accept modifications */
		o_ptr->number = tmp_int;
	}
}

/**
 * Tweak the cursed status of an object.
 *
 * \param o_ptr is the object to curse or decurse
 */
static void wiz_tweak_curse(object_type *o_ptr)
{
	if (cursed_p(o_ptr))
	{
		msg_print(__T("Resetting existing curses."));
		o_ptr->flags[2] &= ~TR2_CURSE_MASK;
	}
	if (get_check(L"Set light curse? "))
		o_ptr->flags[2] |= TR2_LIGHT_CURSE;
	else if (get_check(L"Set heavy curse? "))
		o_ptr->flags[2] |= (TR2_LIGHT_CURSE | TR2_HEAVY_CURSE);
	else if (get_check(L"Set permanent curse? "))
		o_ptr->flags[2] |= (TR2_LIGHT_CURSE | TR2_HEAVY_CURSE | TR2_PERMA_CURSE);
}

/*
 * Play with an item. Options include:
 *   - Output statistics (via wiz_roll_item)
 *   - Reroll item (via wiz_reroll_item)
 *   - Change properties (via wiz_tweak_item)
 *   - Change the number of items (via wiz_quantity_item)
 */
static void do_cmd_wiz_play(void)
{
	int item;

	object_type *i_ptr;
	object_type object_type_body;

	object_type *o_ptr;

	_TCHAR ch;

	const _TCHAR *q; 
	const _TCHAR *s;

	bool changed = FALSE;

	/* Get an item */
	q = L"Play with which object? ";
	s = L"You have nothing to play with.";
	if (!get_item(&item, q, s, (USE_EQUIP | USE_INVEN | USE_FLOOR))) return;

	/* Get the item */
	o_ptr = object_from_item_idx(item);

	/* Save screen */
	screen_save();

	/* Get local object */
	i_ptr = &object_type_body;

	/* Copy object */
	object_copy(i_ptr, o_ptr);

	/* The main loop */
	while (TRUE)
	{
		/* Display the item */
		wiz_display_item(i_ptr);

		/* Get choice */
		if (!get_com(L"[a]ccept [s]tatistics [r]eroll [t]weak [c]urse [q]uantity? ", &ch))
			break;

		if (ch == 'A' || ch == 'a')
		{
			changed = TRUE;
			break;
		}
		else if (ch == 'c' || ch == 'C')
			wiz_tweak_curse(i_ptr);
		else if (ch == 's' || ch == 'S')
			wiz_statistics(i_ptr, p_ptr->depth);
		else if (ch == 'r' || ch == 'r')
			wiz_reroll_item(i_ptr);
		else if (ch == 't' || ch == 'T')
			wiz_tweak_item(i_ptr);
		else if (ch == 'q' || ch == 'Q')
		{
			bool carried = (item >= 0) ? TRUE : FALSE;
			wiz_quantity_item(i_ptr, carried);
		}
	}
	/* Load screen */
	screen_load();

	/* Accept change */
	if (changed)
	{
		/* Message */
		msg_print(__T("Changes accepted."));

		/* Change */
		object_copy(o_ptr, i_ptr);

		/* Recalculate bonuses */
		p_ptr->update |= (PU_BONUS);

		/* Combine / Reorder the pack (later) */
		p_ptr->notice |= (PN_COMBINE | PN_REORDER);

		/* Window stuff */
		p_ptr->redraw |= (PR_INVEN | PR_EQUIP );
	}

	/* Ignore change */
	else
	{
		msg_print(__T("Changes ignored."));
	}
}

/*
 * Wizard routine for creating objects
 *
 * Note that wizards cannot create objects on top of other objects.
 *
 * Hack -- this routine always makes a "dungeon object", and applies
 * magic to it, and attempts to decline cursed items. XXX XXX XXX
 */
static void wiz_create_item(void)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	object_type *i_ptr;
	object_type object_type_body;

	int k_idx;

	/* Save screen */
	screen_save();

	/* Get object base type */
	k_idx = wiz_create_itemtype();

	/* Load screen */
	screen_load();

	/* Return if failed */
	if (!k_idx) 
		return;

	/* Get local object */
	i_ptr = &object_type_body;

	/* Create the item */
	object_prep(i_ptr, k_idx);

	/* Apply magic (no messages, no artifacts) */
	apply_magic(i_ptr, p_ptr->depth, FALSE, FALSE, FALSE);

	/* Mark as cheat, and where created */
	i_ptr->origin = ORIGIN_CHEAT;
	i_ptr->origin_depth = (byte) p_ptr->depth;

	if (k_info[k_idx].tval == TV_GOLD)
		make_gold(i_ptr, p_ptr->depth, k_info[k_idx].sval);

	/* Drop the object from heaven */
	drop_near(i_ptr, -1, py, px);

	/* All done */
	msg_print(__T("Allocated."));
}

/*
 * Create the artifact with the specified number
 */
static void wiz_create_artifact(int a_idx)
{
	object_type *i_ptr;
	object_type object_type_body;
	int k_idx;

	artifact_type *a_ptr = &a_info[a_idx];

	/* Ignore "empty" artifacts */
	if (!a_ptr->name) return;

	/* Get local object */
	i_ptr = &object_type_body;

	/* Wipe the object */
	object_wipe(i_ptr);

	/* Acquire the "kind" index */
	k_idx = lookup_kind(a_ptr->tval, a_ptr->sval);

	/* Oops */
	if (!k_idx) return;

	/* Create the artifact */
	object_prep(i_ptr, k_idx);

	/* Save the name */
	i_ptr->name1 = a_idx;

	/* Extract the fields */
	i_ptr->pval = a_ptr->pval;
	i_ptr->ac = a_ptr->ac;
	i_ptr->dd = a_ptr->dd;
	i_ptr->ds = a_ptr->ds;
	i_ptr->to_a = a_ptr->to_a;
	i_ptr->to_h = a_ptr->to_h;
	i_ptr->to_d = a_ptr->to_d;
	i_ptr->weight = a_ptr->weight;

	/* Hack -- extract the "cursed" flags */
	if (cursed_p(a_ptr))
		i_ptr->flags[2] |= (a_ptr->flags[2] & TR2_CURSE_MASK);

	/* Mark that the artifact has been created. */
	a_ptr->cur_num = 1;

	/* Mark as cheat */
	i_ptr->origin = ORIGIN_CHEAT;

	/* Drop the artifact from heaven */
	drop_near(i_ptr, -1, p_ptr->py, p_ptr->px);

	/* All done */
	msg_print(__T("Allocated."));
}

/*
 * Cure everything instantly
 */
static void do_cmd_wiz_cure_all(void)
{
	/* Remove curses */
	(void)remove_all_curse();

	/* Restore stats */
	(void)res_stat(A_STR);
	(void)res_stat(A_INT);
	(void)res_stat(A_WIS);
	(void)res_stat(A_CON);
	(void)res_stat(A_DEX);
	(void)res_stat(A_CHR);

	/* Restore the level */
	(void)restore_level();

	/* Heal the player */
	p_ptr->chp = p_ptr->mhp;
	p_ptr->chp_frac = 0;

	/* Restore mana */
	p_ptr->csp = p_ptr->msp;
	p_ptr->csp_frac = 0;

	/* Cure stuff */
	(void)clear_timed(TMD_BLIND, TRUE);
	(void)clear_timed(TMD_CONFUSED, TRUE);
	(void)clear_timed(TMD_POISONED, TRUE);
	(void)clear_timed(TMD_AFRAID, TRUE);
	(void)clear_timed(TMD_PARALYZED, TRUE);
	(void)clear_timed(TMD_IMAGE, TRUE);
	(void)clear_timed(TMD_STUN, TRUE);
	(void)clear_timed(TMD_CUT, TRUE);
	(void)clear_timed(TMD_SLOW, TRUE);
	(void)clear_timed(TMD_AMNESIA, TRUE);

	/* No longer hungry */
	(void)set_food(PY_FOOD_MAX - 1);

	/* Redraw everything */
	do_cmd_redraw();
}

/*
 * Go to any level
 */
static void do_cmd_wiz_jump(void)
{
	/* Ask for level */
	if (p_ptr->command_arg <= 0)
	{
		_TCHAR ppp[80];

		_TCHAR tmp_val[160];

		/* Prompt */
		strnfmt(ppp, _countof(ppp), L"Jump to level (0-%d): ", MAX_DEPTH-1);

		/* Default */
		strnfmt(tmp_val, _countof(tmp_val), L"%d", p_ptr->depth);

		/* Ask for a level */
		if (!get_string(ppp, tmp_val, 11)) return;

		/* Extract request */
		p_ptr->command_arg = _tstoi(tmp_val);
	}

	/* Paranoia */
	if (p_ptr->command_arg < 0) p_ptr->command_arg = 0;

	/* Paranoia */
	if (p_ptr->command_arg > MAX_DEPTH - 1) p_ptr->command_arg = MAX_DEPTH - 1;

	/* Accept request */
	msg_format(__T("You jump to dungeon level %d."), p_ptr->command_arg);

	/* New depth */
	p_ptr->depth = p_ptr->command_arg;

	/* Leaving */
	p_ptr->leaving = TRUE;
}

/*
 * Become aware of a lot of objects
 */
static void do_cmd_wiz_learn(void)
{
	int i;

	object_type *i_ptr;
	object_type object_type_body;

	/* Scan every object */
	for (i = 1; i < z_info->k_max; i++)
	{
		object_kind *k_ptr = &k_info[i];

		/* Induce awareness */
		if (k_ptr->level <= p_ptr->command_arg)
		{
			/* Get local object */
			i_ptr = &object_type_body;

			/* Prepare object */
			object_prep(i_ptr, i);

			/* Awareness */
			object_aware(i_ptr);
		}
	}
}

/*
 * Hack -- Rerate Hitpoints
 */
static void do_cmd_rerate(void)
{
	int min_value, max_value, i, percent;

	min_value = (PY_MAX_LEVEL * 3 * (p_ptr->hitdie - 1)) / 8;
	min_value += PY_MAX_LEVEL;

	max_value = (PY_MAX_LEVEL * 5 * (p_ptr->hitdie - 1)) / 8;
	max_value += PY_MAX_LEVEL;

	p_ptr->player_hp[0] = p_ptr->hitdie;

	/* Rerate */
	while (1)
	{
		/* Collect values */
		for (i = 1; i < PY_MAX_LEVEL; i++)
		{
			p_ptr->player_hp[i] = (s16b) randint1(p_ptr->hitdie);
			p_ptr->player_hp[i] += p_ptr->player_hp[i - 1];
		}

		/* Legal values */
		if ((p_ptr->player_hp[PY_MAX_LEVEL - 1] >= min_value) &&
		    (p_ptr->player_hp[PY_MAX_LEVEL - 1] <= max_value)) break;
	}

	percent = (int)(((long)p_ptr->player_hp[PY_MAX_LEVEL - 1] * 200L) /
	                (p_ptr->hitdie + ((PY_MAX_LEVEL - 1) * p_ptr->hitdie)));

	/* Update and redraw hitpoints */
	p_ptr->update |= (PU_HP);
	p_ptr->redraw |= (PR_HP);

	/* Handle stuff */
	handle_stuff();

	/* Message */
	msg_format(__T("Current Life Rating is %d/100."), percent);
}


/*
 * Summon some creatures
 */
static void do_cmd_wiz_summon(int num)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int i;

	for (i = 0; i < num; i++)
	{
		(void)summon_specific(py, px, p_ptr->depth, 0);
	}
}

/*
 * Summon a creature of the specified type
 *
 * This function is rather dangerous XXX XXX XXX
 */
static void do_cmd_wiz_named(int r_idx, bool slp)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int i, x, y;

	/* Paranoia */
	if (!r_idx) 
		return;
	if (r_idx >= z_info->r_max-1) 
		return;

	/* Try 10 times */
	for (i = 0; i < 10; i++)
	{
		int d = 1;

		/* Pick a location */
		scatter(&y, &x, py, px, d, 0);

		/* Require empty grids */
		if (!cave_empty_bold(y, x)) 
			continue;

		/* Place it (allow groups) */
		if (place_monster_aux(y, x, r_idx, slp, TRUE)) 
			break;
	}
}

/*
 * Hack -- Delete all nearby monsters
 */
static void do_cmd_wiz_zap(int d)
{
	int i;

	/* Banish everyone nearby */
	for (i = 1; i < mon_max; i++)
	{
		monster_type *m_ptr = &mon_list[i];

		/* Skip dead monsters */
		if (!m_ptr->r_idx) continue;

		/* Skip distant monsters */
		if (m_ptr->cdis > d) continue;

		/* Delete the monster */
		delete_monster_idx(i);
	}

	/* Update monster list window */
	p_ptr->redraw |= PR_MONLIST;
}

/*
 * Un-hide all monsters
 */
static void do_cmd_wiz_unhide(int d)
{
	int i;

	/* Process monsters */
	for (i = 1; i < mon_max; i++)
	{
		monster_type *m_ptr = &mon_list[i];

		/* Skip dead monsters */
		if (!m_ptr->r_idx) continue;

		/* Skip distant monsters */
		if (m_ptr->cdis > d) continue;

		/* Optimize -- Repair flags */
		repair_mflag_mark = repair_mflag_show = TRUE;

		/* Detect the monster */
		m_ptr->mflag |= (MFLAG_MARK | MFLAG_SHOW);

		/* Update the monster */
		update_mon(i, FALSE);
	}
}

/*
 * Query the dungeon
 */
static void do_cmd_wiz_query(void)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	int y, x;
	_TCHAR langbuf[200];

	_TCHAR cmd;

	u16b mask = 0x00;

	/* Get a "debug command" */
	if (!get_com(L"Debug Command Query: ", &cmd)) 
		return;

	/* Extract a flag */
	switch (cmd)
	{
		case '0': mask = (1 << 0); break;
		case '1': mask = (1 << 1); break;
		case '2': mask = (1 << 2); break;
		case '3': mask = (1 << 3); break;
		case '4': mask = (1 << 4); break;
		case '5': mask = (1 << 5); break;
		case '6': mask = (1 << 6); break;
		case '7': mask = (1 << 7); break;

		case 'm': mask |= (CAVE_MARK); break;
		case 'g': mask |= (CAVE_GLOW); break;
		case 'r': mask |= (CAVE_ROOM); break;
		case 'i': mask |= (CAVE_ICKY); break;
		case 's': mask |= (CAVE_SEEN); break;
		case 'v': mask |= (CAVE_VIEW); break;
		case 't': mask |= (CAVE_TEMP); break;
		case 'w': mask |= (CAVE_WALL); break;
	}
	/* Scan map */
	for (y = Term->offset_y; y < Term->offset_y + SCREEN_HGT; y++)
	{
		for (x = Term->offset_x; x < Term->offset_x + SCREEN_WID; x++)
		{
			byte a = TERM_RED;

			if (!in_bounds_fully(y, x)) continue;

			/* Given mask, show only those grids */
			if (mask && !(cave_info[y][x] & mask)) continue;

			/* Given no mask, show unknown grids */
			if (!mask && (cave_info[y][x] & (CAVE_MARK))) continue;

			/* Color */
			if (cave_floor_bold(y, x)) a = TERM_YELLOW;

			/* Display player/floors/walls */
			if ((y == py) && (x == px))
			{
				print_rel('@', a, y, x);
			}
			else if (cave_floor_bold(y, x))
			{
				print_rel('*', a, y, x);
			}
			else
			{
				print_rel('#', a, y, x);
			}
		}
	}
	/* Get keypress */
	msg_print(LS(__T("Press any key."),
			spaceme(langbuf,__T("何かキーを押してください。"))));

	message_flush();

	/* Redraw map */
	prt_map();
}

/*
 * Create lots of items.
 */
static void wiz_test_kind(int tval)
{
	int py = p_ptr->py;
	int px = p_ptr->px;
	int sval;

	object_type object_type_body;
	object_type *i_ptr = &object_type_body;

	for (sval = 0; sval < 255; sval++)
	{
		int k_idx = lookup_kind(tval, sval);

		if (k_idx)
		{
			/* Create the item */
			object_prep(i_ptr, k_idx);

			/* Apply magic (no messages, no artifacts) */
			apply_magic(i_ptr, p_ptr->depth, FALSE, FALSE, FALSE);

			/* Mark as cheat, and where created */
			i_ptr->origin = ORIGIN_CHEAT;
			i_ptr->origin_depth = (byte) p_ptr->depth;

			if (k_info[k_idx].tval == TV_GOLD)
				make_gold(i_ptr, p_ptr->depth, sval);

			/* Drop the object from heaven */
			drop_near(i_ptr, -1, py, px);
		}
	}
	msg_print(__T("Done."));
}

/*
 * Display the debug commands help file.
 */
static void do_cmd_wiz_help(void) 
{
	_TCHAR buf[80];
	strnfmt(buf, _countof(buf), L"debug.txt");
	screen_save();
	show_file(buf, NULL, 0, 0);
	screen_load();
}

/*
 * Ask for and parse a "debug command"
 *
 * The "p_ptr->command_arg" may have been set.
 */
void do_cmd_debug(void)
{
	int py = p_ptr->py;
	int px = p_ptr->px;

	_TCHAR cmd;

	/* Get a "debug command" */
	if (!get_com(L"Debug Command: ", &cmd)) 
		return;

	/* Analyze the command */
	switch (cmd)
	{
		/* Ignore */
		case ESCAPE:
		case ' ':
		case '\n':
		case '\r':
		{
			break;
		}
#ifdef ALLOW_SPOILERS

		/* Hack -- Generate Spoilers */
		case '"':
		{
			do_cmd_spoilers();
			break;
		}
#endif
		/* Hack -- Help */
		case '?':
		{
			do_cmd_wiz_help();
			break;
		}
		/* Cure all maladies */
		case 'a':
		{
			do_cmd_wiz_cure_all();
			break;
		}
		/* Teleport to target */
		case 'b':
		{
			do_cmd_wiz_bamf();
			break;
		}
		/* Create any object */
		case 'c':
		{
			wiz_create_item();
			break;
		}
		/* Create an artifact */
		case 'C':
		{
			if (p_ptr->command_arg > 0)
			{
				wiz_create_artifact(p_ptr->command_arg);
			}
			else
			{
				_TCHAR name[80] = __T("");
				int a_idx = -1;

				/* Avoid the prompt getting in the way */
				screen_save();

				/* Prompt */
				prt(__T("Create which artifact? "), 0, 0);

				/* Get the name */
				if (askfor_aux(name, _countof(name), NULL))
				{
					/* See if an a_idx was entered */
					a_idx = get_idx_from_name(name);
					
					/* If not, find the artifact with that name */
					if (a_idx < 1)
						a_idx = lookup_artifact_name(name); 
					
					/* Did we find a valid artifact? */
					if (a_idx != -1)
						wiz_create_artifact(a_idx);
				}
				
				/* Reload the screen */
				screen_load();
			}
			break;
		}
		/* Detect everything */
		case 'd':
		{
			detect_all(TRUE);
			break;
		}
		/* Edit character */
		case 'e':
		{
			do_cmd_wiz_change();
			break;
		}
		case 'f':
		{
			stats_collect();
			break;
		}
		/* Good Objects */
		case 'g':
		{
			if (p_ptr->command_arg <= 0) p_ptr->command_arg = 1;
			acquirement(py, px, p_ptr->depth, p_ptr->command_arg, FALSE);
			break;
		}
		/* Hitpoint rerating */
		case 'h':
		{
			do_cmd_rerate();
			break;
		}
		/* Identify */
		case 'i':
		{
			(void)ident_spell();
			break;
		}
		/* Go up or down in the dungeon */
		case 'j':
		{
			do_cmd_wiz_jump();
			break;
		}
		/* Self-Knowledge */
		case 'k':
		{
			self_knowledge(TRUE);
			break;
		}
		/* Learn about objects */
		case 'l':
		{
			do_cmd_wiz_learn();
			break;
		}
		/* Magic Mapping */
		case 'm':
		{
			map_area();
			break;
		}
		/* Summon Named Monster */
		case 'n':
		{
			if (p_ptr->command_arg > 0)
			{
				do_cmd_wiz_named(p_ptr->command_arg, TRUE);
			}
			else
			{
				_TCHAR name[80] = __T("");
				s16b r_idx;

				/* Avoid the prompt getting in the way */
				screen_save();

				/* Prompt */
				prt(__T("Summon which monster? "), 0, 0);

				/* Get the name */
				if (askfor_aux(name, _countof(name), NULL))
				{
					/* See if a r_idx was entered */
					r_idx = get_idx_from_name(name);
					
					/* If not, find the monster with that name */
					if (r_idx < 1)
						r_idx = lookup_monster(name); 
					
					/* Did we find a valid monster? */
					if (r_idx != -1)
						do_cmd_wiz_named(r_idx, TRUE);
				}
				/* Reload the screen */
				screen_load();
			}
			break;
		}
		/* Object playing routines */
		case 'o':
		{
			do_cmd_wiz_play();
			break;
		}
		/* Phase Door */
		case 'p':
		{
			teleport_player(10);
			break;
		}
		/* Query the dungeon */
		case 'q':
		{
			do_cmd_wiz_query();
			break;
		}
		/* Summon Random Monster(s) */
		case 's':
		{
			if (p_ptr->command_arg <= 0) p_ptr->command_arg = 1;
			do_cmd_wiz_summon(p_ptr->command_arg);
			break;
		}
		/* Teleport */
		case 't':
		{
			teleport_player(100);
			break;
		}
		/* Create a trap */
		case 'T':
		{
			cave_set_feat(p_ptr->py, p_ptr->px, FEAT_INVIS);
			break;
		}
		/* Un-hide all monsters */
		case 'u':
		{
			if (p_ptr->command_arg <= 0) p_ptr->command_arg = 255;
			do_cmd_wiz_unhide(p_ptr->command_arg);
			break;
		}
		/* Very Good Objects */
		case 'v':
		{
			if (p_ptr->command_arg <= 0) p_ptr->command_arg = 1;
			acquirement(py, px, p_ptr->depth, p_ptr->command_arg, TRUE);
			break;
		}
		case 'V':
		{
			wiz_test_kind(p_ptr->command_arg);
			break;
		}

		/* Wizard Light the Level */
		case 'w':
		{
			wiz_lite();
			break;
		}
		/* Increase Experience */
		case 'x':
		{
			if (p_ptr->command_arg)
			{
				gain_exp(p_ptr->command_arg);
			}
			else
			{
				gain_exp(p_ptr->exp + 1);
			}
			break;
		}
		/* Zap Monsters (Banishment) */
		case 'z':
		{
			if (p_ptr->command_arg <= 0) p_ptr->command_arg = MAX_SIGHT;
			do_cmd_wiz_zap(p_ptr->command_arg);
			break;
		}
		/* Hack */
		case '_':
		{
			do_cmd_wiz_hack_ben();
			break;
		}
		/* Oops */
		default:
		{
			msg_print(__T("That is not a valid debug command."));
			break;
		}
	}
}

#endif
