//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathTVector2.h
 * @brief		2DxNgt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathTVector2_H_
#define INCG_IRIS_MathTVector2_H_

//======================================================================
// include
#include "../fpu/MathFpuDef.h"
#include "../../c++0x/type_traits/cpp0x_value_traits.hpp"

namespace iris {
namespace math
{

//======================================================================
// declare
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Set(IrisTVec2<_TN>* pv0, _TN x, _TN y);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Copy(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2PositiveZero(IrisTVec2<_TN>* pv0);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2NegativeZero(IrisTVec2<_TN>* pv0);
template<typename _TN>IrisIVec2*		TFpuVec2Ceil(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisIVec2*		TFpuVec2Trunc(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisIVec2*		TFpuVec2Round(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisIVec2*		TFpuVec2Floor(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2FromIVec2(IrisTVec2<_TN>* pv0, const IrisIVec2* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Add(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Sub(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Mul(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Div(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Neg(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Abs(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Lerp(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, _TN t);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Scale(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, _TN s);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2ScaleAdd(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, _TN s);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Hermite(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pt1
									 		 , const IrisTVec2<_TN>* pv2, const IrisTVec2<_TN>* pt2, _TN t);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Bezier(IrisTVec2<_TN>* pv0, _TN n, const IrisTVec2<_TN>* pva, _TN t);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Clamp(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, _TN min, _TN max);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Max(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Min(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>_TN				TFpuVec2InnerProduct(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>_TN				TFpuVec2OuterProduct(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>_TN				TFpuVec2Funnel(const IrisTVec2<_TN>* pv0);
template<typename _TN>_TN				TFpuVec2Average(const IrisTVec2<_TN>* pv0);
template<typename _TN>IrisBool			TFpuVec2IsEqual(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisBool			TFpuVec2IsZero(const IrisTVec2<_TN>* pv0);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2SignFloat(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisIVec2*		TFpuVec2SignInt(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Normalize(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>_TN				TFpuVec2Length(const IrisTVec2<_TN>* pv0);
template<typename _TN>_TN				TFpuVec2Distance(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2FaceForward(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, const IrisTVec2<_TN>* pv3);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Reflect(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Refract(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, f32 eta);

template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Null(IrisTVec2<_TN>* pv0);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Zero(IrisTVec2<_TN>* pv0);
template<typename _TN>_TN				TFpuVec2Dot(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>_TN				TFpuVec2Cross(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Subtract(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Multiply(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Divide(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2);
template<typename _TN>IrisTVec2<_TN>*	TFpuVec2Inter(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, _TN t);

//======================================================================
// function
/**
 * @brief	vfݒ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	x	= xl
 * @param [in]	y	= yl
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Set(IrisTVec2<_TN>* pv0, _TN x, _TN y)
{
	MATH_NULLASSERT( pv0 );
	pv0->x = x; pv0->y = y;
	return pv0;
}

/**
 * @brief	vfRs[
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Copy(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = pv1->x; pv0->y = pv1->y;
	return pv0;
}

/**
 * @brief	+0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2PositiveZero(IrisTVec2<_TN>* pv0)
{
	MATH_NULLASSERT( pv0 );
	pv0->x = pv0->y = cpp0x::zero_traits<_TN>::value;
	return pv0;
}

/**
 * @brief	-0ɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2NegativeZero(IrisTVec2<_TN>* pv0)
{
	MATH_NULLASSERT( pv0 );
	pv0->x = pv0->y = -cpp0x::zero_traits<_TN>::value;
	return pv0;
}

/**
 * @brief	؂グ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisIVec2*	TFpuVec2Ceil(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (s32)( ceil(pv1->x) );
	pv0->y = (s32)( ceil(pv1->y) );
	return pv0;
}

/**
 * @brief	0ۂ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisIVec2*	TFpuVec2Trunc(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Trunc(pv1->x) );
	pv0->y = (s32)( F64_Trunc(pv1->y) );
	return pv0;
}

/**
 * @brief	ߖTۂ(ľܓ)
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisIVec2*	TFpuVec2Round(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Round(pv1->x) );
	pv0->y = (s32)( F64_Round(pv1->y) );
	return pv0;
}

/**
 * @brief	؂̂
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisIVec2*	TFpuVec2Floor(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (s32)( F64_Floor(pv1->x) );
	pv0->y = (s32)( F64_Floor(pv1->y) );
	return pv0;
}

/**
 * @brief	xNgϊ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2FromIVec2(IrisTVec2<_TN>* pv0, const IrisIVec2* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (_TN)( pv1->x );
	pv0->y = (_TN)( pv1->y );
	return pv0;
}

/**
 * @brief	xNg̉Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Add(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	pv0->x = pv1->x + pv2->x;
	pv0->y = pv1->y + pv2->y;
	return pv0;
}

/**
 * @brief	xNǧZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Sub(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	pv0->x = pv1->x - pv2->x;
	pv0->y = pv1->y - pv2->y;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Mul(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	pv0->x = pv1->x * pv2->x;
	pv0->y = pv1->y * pv2->y;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Div(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	pv0->x = pv1->x / pv2->x;
	pv0->y = pv1->y / pv2->y;
	return pv0;
}

/**
 * @brief	xNg̊evf̕]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Neg(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = -pv1->x;
	pv0->y = -pv1->y;
	return pv0;
}

/**
 * @brief	xNg̊evf̐Βl擾
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Abs(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (_TN)(F64_Abs(pv1->x));
	pv0->y = (_TN)(F64_Abs(pv1->y));
	return pv0;
}

/**
 * @brief	xNg̊Ԃ̓}
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Lerp(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, _TN t)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	pv0->x = pv1->x + t * (pv2->x - pv1->x);
	pv0->y = pv1->y + t * (pv2->y - pv1->y);
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Scale(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, _TN s)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = pv1->x * s;
	pv0->y = pv1->y * s;
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 *				pv0 = pv1 * s + vp2
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= ZxNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2ScaleAdd(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, _TN s)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	pv0->x = pv1->x * s + pv2->x;
	pv0->y = pv1->y * s + pv2->y;
	return pv0;
}

/**
 * @brief	xNg̃G~[gXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pt1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pt2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Hermite(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pt1
									  , const IrisTVec2<_TN>* pv2, const IrisTVec2<_TN>* pt2, _TN t)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pt1 );
	MATH_NULLASSERT( pv2 );
	MATH_NULLASSERT( pt2 );
	f32 t2 = t * t;
	f32 t3 = t * t2;
	IrisTVec2<_TN> v[4];
	TFpuVec2Scale<_TN>(&v[0], pv1, (( 2 * t3) - (3 * t2) + 1.0f));
	TFpuVec2Scale<_TN>(&v[1], pt1, ((     t3) - (2 * t2) + t       ));
	TFpuVec2Scale<_TN>(&v[3], pv2, ((-2 * t3) + (3 * t2)           ));
	TFpuVec2Scale<_TN>(&v[2], pt2, ((     t3) - (    t2)           ));
	TFpuVec2Add<_TN>(&v[0], &v[0], &v[1]);		// v[0] = v[0] + v[1]
	TFpuVec2Add<_TN>(&v[0], &v[0], &v[2]);		// v[0] = v[0] + v[2]
	TFpuVec2Add<_TN>(pv0,   &v[0], &v[3]);		// *pv0 = v[0] + v[3]
	return pv0;
}

/**
 * @brief	xNg̃xWGXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	n	= xNgz
 * @param [in]	pva	= xNgz
 * @param [in]	t	= 
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Bezier(IrisTVec2<_TN>* pv0, _TN n, const IrisTVec2<_TN>* pva, _TN t)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pva );
	IRIS_ASSERT( n >= 2 );
	if( t <= 0.0f )	{ TFpuVec2Copy<_TN>(pv0, &pva[0]); return pv0; }
	if( t >= 1.0f )	{ TFpuVec2Copy<_TN>(pv0, &pva[n-1]); return pv0; }
	TFpuVec2PositiveZero<_TN>(pv0);
	f32 tt = 1.0f;			// t^ip
	f32 it = 1.0f - t;		// (1-t)^ip
	s32 nn = n-1;
	const IrisTVec2<_TN>* pv = pva;
	for( s32 i=0; i < n; ++i, ++pv )
	{
		f32 J = Combination(nn, i) * tt * F64_Pow(it, (f64)(nn-i));
		pv0->x += (_TN)(pv->x * J);
		pv0->y += (_TN)(pv->y * J);
		tt = tt * t;
	}
	return pv0;
}

/**
 * @brief	xNg̃Nv
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	min	= ŏl
 * @param [in]	max	= ől
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Clamp(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, _TN min, _TN max)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (pv1->x < min) ? min : ((pv1->x > max) ? max : pv1->x);
	pv0->y = (pv1->y < min) ? min : ((pv1->y > max) ? max : pv1->y);
	return pv0;
}

/**
 * @brief	xNg̊evf̑傫Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Max(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (pv1->x > pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y > pv2->y) ? pv1->y : pv2->y;
	return pv0;
}

/**
 * @brief	xNg̊evf̏Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Min(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (pv1->x < pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y < pv2->y) ? pv1->y : pv2->y;
	return pv0;
}

/**
 * @brief	
 * @param [in]	pv0	= xNg0
 * @param [in]	pv1	= xNg1
 * @return	ϒl
*/
template<typename _TN>
_TN		TFpuVec2InnerProduct(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	return pv0->x * pv1->x + pv0->y * pv1->y;
}

/**
 * @brief	O
 * @param [in]	pv0	= xNg0
 * @param [in]	pv1	= xNg1
 * @return	ϒl
*/
template<typename _TN>
_TN		TFpuVec2OuterProduct(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	return pv0->x * pv1->y - pv0->y * pv1->x;
}

/**
 * @brief	xNg̊evf̑a擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̑a
*/
template<typename _TN>
_TN		TFpuVec2Funnel(const IrisTVec2<_TN>* pv0)
{
	MATH_NULLASSERT( pv0 );
	return (pv0->x + pv0->y);
}

/**
 * @brief	xNg̊evf̕ς擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̕
*/
template<typename _TN>
_TN		TFpuVec2Average(const IrisTVec2<_TN>* pv0)
{
	MATH_NULLASSERT( pv0 );
	return (pv0->x + pv0->y) / 2;
}

/**
 * @brief	xNgǂԂ
 * @param [in]	pv0	= rxNg
 * @param [in]	pv1	= rxNg
 * @return	^Ul
*/
template<typename _TN>
IrisBool	TFpuVec2IsEqual(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	if( (pv0->x != pv1->x) || (pv0->y != pv1->y) ) return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	[xNgǂԂ
 * @param [in]	pv0	= ̓xNg
 * @return	^Ul
*/
template<typename _TN>
IrisBool	TFpuVec2IsZero(const IrisTVec2<_TN>* pv0)
{
	MATH_NULLASSERT( pv0 );
	if( pv0->x == pv0->y == 0 )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	,̕IrisTVec2<_TN>ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2SignFloat(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0) ? -1 : (pv1->x > 0) ? 1 : 0;
	pv0->y = (pv1->y < 0) ? -1 : (pv1->y > 0) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	,̕IrisIVec2ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisIVec2*	TFpuVec2SignInt(IrisIVec2* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	pv0->x = (pv1->x < 0) ? -1 : (pv1->x > 0) ? 1 : 0;
	pv0->y = (pv1->y < 0) ? -1 : (pv1->y > 0) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	xNg̐K
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Normalize(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	_TN x = pv1->x, y = pv1->y;
	f64 q = F64_Sqrt( x * x + y * y );
#if IRIS_FPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != 0.0 );
	q = 1.0 / q;
#else
	if( q != 0.0 ) q = 1.0 / q;
#endif
	pv0->x = (_TN)(x * q);
	pv0->y = (_TN)(y * q);
	return pv0;
}

/**
 * @brief	xNg̒Ԃ
 * @param [in]	pv0	= ̓xNg
 * @return	xNg̒
*/
template<typename _TN>
_TN		TFpuVec2Length(const IrisTVec2<_TN>* pv0)
{
	MATH_NULLASSERT( pv0 );
	return (_TN)(F64_Sqrt( pv0->x * pv0->x + pv0->y * pv0->y ));
}

/**
 * @brief	xNg̊Ԃ̋
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	xNg̒
*/
template<typename _TN>
_TN		TFpuVec2Distance(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
{
	MATH_NULLASSERT( pv0 );
	_TN x = pv0->x - pv1->x;
	_TN y = pv0->y - pv1->y;
	return (_TN)(F64_Sqrt( x * x + y * y ));
}

/**
 * @brief	IuWFNg\ʂJɌ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pv3	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2FaceForward(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, const IrisTVec2<_TN>* pv3)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	MATH_NULLASSERT( pv3 );
	_TN d = pv2->x * pv3->x + pv2->y * pv3->y;
	if(d < 0)
	{
		pv0->x = pv1->x;
		pv0->y = pv1->y;
	}
	else
	{
		pv0->x = -pv1->x;
		pv0->y = -pv1->y;
	}
	return pv0;
}

/**
 * @brief	˃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Reflect(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	_TN d = pv1->x * pv2->x + pv1->y * pv2->y;
	pv0->x = pv1->x - 2 * d * pv2->x;
	pv0->y = pv1->y - 2 * d * pv2->y;
	return pv0;
}

/**
 * @brief	܃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	eta	= ܗ
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Refract(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, f32 eta)
{
	MATH_NULLASSERT( pv0 );
	MATH_NULLASSERT( pv1 );
	MATH_NULLASSERT( pv2 );
	_TN d, e, f;
	d = pv1->x * pv2->x + pv1->y * pv2->y;
	e = 1 - eta * eta * (1 - d * d);
	if(e < 0)
	{
		pv0->x = 0.0f;
		pv0->y = 0.0f;
	}
	else
	{
		f = eta * d - (_TN)F64_Sqrt(e);
		pv0->x = eta * pv1->x - f * pv2->x;
		pv0->y = eta * pv1->y - f * pv2->y;
	}
	return pv0;
}

template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Null(IrisTVec2<_TN>* pv0)
	{ return TFpuVec2PositiveZero<_TN>(pv0); }

template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Zero(IrisTVec2<_TN>* pv0)
	{ return TFpuVec2PositiveZero<_TN>(pv0); }

template<typename _TN>
_TN			TFpuVec2Dot(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
	{ return TFpuVec2InnerProduct<_TN>(pv0, pv1); }

template<typename _TN>
_TN			TFpuVec2Cross(const IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1)
	{ return TFpuVec2OuterProduct<_TN>(pv0, pv1); }

template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Subtract(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
	{ return TFpuVec2Sub<_TN>(pv0, pv1, pv2); }

template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Multiply(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
	{ return TFpuVec2Mul<_TN>(pv0, pv1, pv2); }

template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Divide(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2)
	{ return TFpuVec2Div<_TN>(pv0, pv1, pv2); }

template<typename _TN>
IrisTVec2<_TN>*	TFpuVec2Inter(IrisTVec2<_TN>* pv0, const IrisTVec2<_TN>* pv1, const IrisTVec2<_TN>* pv2, _TN t)
	{ return TFpuVec2Lerp<_TN>(pv0, pv1, pv2, t); }

}	// end of namespace math
}	// end of namespace iris

#endif
