//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathTMatrix44.h
 * @brief		4x4}gbNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathTMatrix44_H_
#define INCG_IRIS_MathTMatrix44_H_

//======================================================================
// include
#include "MathTMatrix33.h"
#include "MathTVector4.h"
#include "MathTPlane.h"

namespace iris {
namespace math
{

//======================================================================
// declare
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Unit(IrisTMtx44<_TN>* pm0);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Zero(IrisTMtx44<_TN>* pm0);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Copy(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44SetTransfer(IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44GetTransfer(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Transfer(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44Transform(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44TransformXYZ(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44HomogeneousTransform(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Mul(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisTMtx44<_TN>* pm2);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Scale(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN s);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Transpose(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44RotZ(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN rz);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44RotIdxZ(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, u16 idz);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44RotY(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN ry);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44RotIdxY(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, u16 idy);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44RotX(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN rx);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44RotIdxX(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, u16 idx);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Rot(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisTVec4<_TN>* pv0);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44RotIdx(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisSVec4* pv0);
template<typename _TN>IrisBool				TFpuMtx44IsUnit(const IrisTMtx44<_TN>* pm0);
template<typename _TN>_TN					TFpuMtx44Trace(const IrisTMtx44<_TN>* pm0);
template<typename _TN>_TN					TFpuMtx44Determinant(const IrisTMtx44<_TN>* pm0);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Adjoint(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Inverse(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44Inverse2(IrisTMtx44<_TN>* pm0, _TN* pDeterminant, const IrisTMtx44<_TN>* pm1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44DropShadow(IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv0, const IrisFPlane* pl0);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44NormalizeXYZ(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1);
template<typename _TN>IrisTMtx44<_TN>*		TFpuMtx44TruncatePrecision24(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1);

//======================================================================
// inline
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Identity(IrisTMtx44<_TN>* pm0)			{ return TFpuMtx44Unit(pm0); }
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Null(IrisTMtx44<_TN>* pm0)				{ return TFpuMtx44Zero(pm0); }
template<typename _TN>IrisTVec4<_TN>*	TFpuMtx44Apply(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1)		{ return TFpuMtx44Transform(pv0, pm0, pv1); }
template<typename _TN>IrisTVec4<_TN>*	TFpuMtx44ApplyXYZ(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1)	{ return TFpuMtx44TransformXYZ(pv0, pm0, pv1); }
template<typename _TN>IrisBool			TFpuMtx44IsIdentity(const IrisTMtx44<_TN>* pm0)	{ return TFpuMtx44IsUnit(pm0); }

//======================================================================
// function
/**
 * @brief	Pʃ}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>* TFpuMtx44Unit(IrisTMtx44<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN zero;
	zero -= zero;
	pm0->x.x = pm0->y.y = pm0->z.z = pm0->w.w = 1;
	pm0->x.y = pm0->x.z = pm0->x.w =
	pm0->y.x = pm0->y.z = pm0->y.w =
	pm0->z.x = pm0->z.y = pm0->z.w =
	pm0->w.x = pm0->w.y = pm0->w.z = zero;
	return pm0;
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Zero(IrisTMtx44<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	TFpuVec4Zero(&pm0->x);
	TFpuVec4Zero(&pm0->y);
	TFpuVec4Zero(&pm0->z);
	TFpuVec4Zero(&pm0->w);
	return pm0;
}

/**
 * @brief	}gbNX̃Rs[
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Copy(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
#if 1
	pm0->_00 = pm1->_00;
	pm0->_01 = pm1->_01;
	pm0->_02 = pm1->_02;
	pm0->_03 = pm1->_03;
	pm0->_10 = pm1->_10;
	pm0->_11 = pm1->_11;
	pm0->_12 = pm1->_12;
	pm0->_13 = pm1->_13;
	pm0->_20 = pm1->_20;
	pm0->_21 = pm1->_21;
	pm0->_22 = pm1->_22;
	pm0->_23 = pm1->_23;
	pm0->_30 = pm1->_30;
	pm0->_31 = pm1->_31;
	pm0->_32 = pm1->_32;
	pm0->_33 = pm1->_33;
#else
	TFpuVec4Copy(&pm0->x, &pm1->x);
	TFpuVec4Copy(&pm0->y, &pm1->y);
	TFpuVec4Copy(&pm0->z, &pm1->z);
	TFpuVec4Copy(&pm0->w, &pm1->w);
#endif
	return pm0;
}

/**
 * @brief	xNg̕sړʕݒ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pv0	= ̓xNg
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44SetTransfer(IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	pm0->w.x = pv0->x;
	pm0->w.y = pv0->y;
	pm0->w.z = pv0->z;
	pm0->w.w = pv0->w;
	return pm0;
}

/**
 * @brief	xNg̕sړʕ擾
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pv0	= ̓xNg
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44GetTransfer(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	pv0->x = pm0->w.x;
	pv0->y = pm0->w.y;
	pv0->z = pm0->w.z;
	pv0->w = pm0->w.w;
	return pv0;
}

/**
 * @brief	}gbNX̕sړ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ړxNg
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Transfer(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	MATH_FPU_NULLASSERT( pv0 );
	pm0->x.x = pm1->x.x;
	pm0->x.y = pm1->x.y;
	pm0->x.z = pm1->x.z;
	pm0->x.w = pm1->x.w;
	pm0->y.x = pm1->y.x;
	pm0->y.y = pm1->y.y;
	pm0->y.z = pm1->y.z;
	pm0->y.w = pm1->y.w;
	pm0->z.x = pm1->z.x;
	pm0->z.y = pm1->z.y;
	pm0->z.z = pm1->z.z;
	pm0->z.w = pm1->z.w;
	pm0->w.x = pm1->w.x + pv0->x;
	pm0->w.y = pm1->w.y + pv0->y;
	pm0->w.z = pm1->w.z + pv0->z;
	pm0->w.w = pv0->w;
	return pm0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44Transform(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	_TN x = pv1->x;
	_TN y = pv1->y;
	_TN z = pv1->z;
	_TN w = pv1->w;
	pv0->x = pm0->x.x * x + pm0->y.x * y + pm0->z.x * z + pm0->w.x * w;
	pv0->y = pm0->x.y * x + pm0->y.y * y + pm0->z.y * z + pm0->w.y * w;
	pv0->z = pm0->x.z * x + pm0->y.z * y + pm0->z.z * z + pm0->w.z * w;
	pv0->w = pm0->x.w * x + pm0->y.w * y + pm0->z.w * z + pm0->w.w * w;
	return pv0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44TransformXYZ(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	_TN x = pv1->x;
	_TN y = pv1->y;
	_TN z = pv1->z;
	pv0->x = pm0->x.x * x + pm0->y.x * y + pm0->z.x * z;
	pv0->y = pm0->x.y * x + pm0->y.y * y + pm0->z.y * z;
	pv0->z = pm0->x.z * x + pm0->y.z * y + pm0->z.z * z;
	pv0->w = pm0->x.w * x + pm0->y.w * y + pm0->z.w * z;
	return pv0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>IrisTVec4<_TN>*		TFpuMtx44HomogeneousTransform(IrisTVec4<_TN>* pv0, const IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	IRIS_ASSERT( pv0 != pv1 );	// TODO : AhXɖΉ
	pv0->x = pm0->x.x * pv1->x + pm0->y.x * pv1->y + pm0->z.x * pv1->z + pv1->w;
	pv0->y = pm0->x.y * pv1->x + pm0->y.y * pv1->y + pm0->z.y * pv1->z + pv1->w;
	pv0->z = pm0->x.z * pv1->x + pm0->y.z * pv1->y + pm0->z.z * pv1->z + pv1->w;
	pv0->w = pm0->x.w * pv1->x + pm0->y.w * pv1->y + pm0->z.w * pv1->z + pv1->w;
	return pv0;
}

/**
 * @brief	Q̃}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pm2	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Mul(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisTMtx44<_TN>* pm2)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	MATH_FPU_NULLASSERT( pm2 );
	IRIS_ASSERT( pm0 != pm1 );
	IRIS_ASSERT( pm0 != pm2 );
	pm0->x.x = pm1->x.x * pm2->x.x + pm1->y.x * pm2->x.y + pm1->z.x * pm2->x.z + pm1->w.x * pm2->x.w;
	pm0->x.y = pm1->x.y * pm2->x.x + pm1->y.y * pm2->x.y + pm1->z.y * pm2->x.z + pm1->w.y * pm2->x.w;
	pm0->x.z = pm1->x.z * pm2->x.x + pm1->y.z * pm2->x.y + pm1->z.z * pm2->x.z + pm1->w.z * pm2->x.w;
	pm0->x.w = pm1->x.w * pm2->x.x + pm1->y.w * pm2->x.y + pm1->z.w * pm2->x.z + pm1->w.w * pm2->x.w;
	pm0->y.x = pm1->x.x * pm2->y.x + pm1->y.x * pm2->y.y + pm1->z.x * pm2->y.z + pm1->w.x * pm2->y.w;
	pm0->y.y = pm1->x.y * pm2->y.x + pm1->y.y * pm2->y.y + pm1->z.y * pm2->y.z + pm1->w.y * pm2->y.w;
	pm0->y.z = pm1->x.z * pm2->y.x + pm1->y.z * pm2->y.y + pm1->z.z * pm2->y.z + pm1->w.z * pm2->y.w;
	pm0->y.w = pm1->x.w * pm2->y.x + pm1->y.w * pm2->y.y + pm1->z.w * pm2->y.z + pm1->w.w * pm2->y.w;
	pm0->z.x = pm1->x.x * pm2->z.x + pm1->y.x * pm2->z.y + pm1->z.x * pm2->z.z + pm1->w.x * pm2->z.w;
	pm0->z.y = pm1->x.y * pm2->z.x + pm1->y.y * pm2->z.y + pm1->z.y * pm2->z.z + pm1->w.y * pm2->z.w;
	pm0->z.z = pm1->x.z * pm2->z.x + pm1->y.z * pm2->z.y + pm1->z.z * pm2->z.z + pm1->w.z * pm2->z.w;
	pm0->z.w = pm1->x.w * pm2->z.x + pm1->y.w * pm2->z.y + pm1->z.w * pm2->z.z + pm1->w.w * pm2->z.w;
	return pm0;
}

/**
 * @brief	}gbNX̃XP[O
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	s	= XJ[l
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Scale(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN s)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	TFpuVec4Scale(&pm0->x, &pm1->x, s);
	TFpuVec4Scale(&pm0->y, &pm1->y, s);
	TFpuVec4Scale(&pm0->z, &pm1->z, s);
	TFpuVec4Scale(&pm0->w, &pm1->w, s);
	return pm0;
}

/**
 * @brief	}gbNX̓]us߂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Transpose(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	pm0->x.x = pm1->x.x;
	_TN f01 = pm1->x.y;
	_TN f02 = pm1->x.z;
	_TN f03 = pm1->x.w;
	pm0->x.y = pm1->y.x;
	_TN f11 = pm1->y.y;
	_TN f12 = pm1->y.z;
	_TN f13 = pm1->y.w;
	pm0->x.z = pm1->z.x;
	_TN f21 = pm1->z.y;
	_TN f22 = pm1->z.z;
	_TN f23 = pm1->z.w;
	pm0->x.w = pm1->w.x;
	pm0->y.x = f01;
	pm0->y.y = f11;
	pm0->y.z = f21;
	pm0->y.w = pm1->w.y;
	pm0->z.x = f02;
	pm0->z.y = f12;
	pm0->z.z = f22;
	pm0->z.w = pm1->w.z;
	pm0->w.x = f03;
	pm0->w.y = f13;
	pm0->w.z = f23;
	pm0->w.w = pm1->w.w;
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44RotZ(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN rz)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_Cos(rz);
	_TN s = F32_Sin(rz);
	_TN zero;
	zero -= zero;
	if( pm1 == nullptr )
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = zero;
		pm0->x.w = zero;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = zero;
		pm0->y.w = zero;
		pm0->z.x = zero;
		pm0->z.y = zero;
		pm0->z.z = 1;
		pm0->z.w = zero;
		pm0->w.x = zero;
		pm0->w.y = zero;
		pm0->w.z = zero;
		pm0->w.w = 1;
	}
	else
	{
		IrisTMtx44<_TN> m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = zero;
		m.x.w = zero;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = zero;
		m.y.w = zero;
		m.z.x = zero;
		m.z.y = zero;
		m.z.z = 1;
		m.z.w = zero;
		m.w.x = zero;
		m.w.y = zero;
		m.w.z = zero;
		m.w.w = 1;
		TFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44RotIdxZ(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, u16 idz)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_CosIdx(idz);
	_TN s = F32_SinIdx(idz);
	_TN zero;
	zero -= zero;
	if( pm1 == nullptr )
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = zero;
		pm0->x.w = zero;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = zero;
		pm0->y.w = zero;
		pm0->z.x = zero;
		pm0->z.y = zero;
		pm0->z.z = 1;
		pm0->z.w = zero;
		pm0->w.x = zero;
		pm0->w.y = zero;
		pm0->w.z = zero;
		pm0->w.w = 1;
	}
	else
	{
		IrisTMtx44<_TN> m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = zero;
		m.x.w = zero;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = zero;
		m.y.w = zero;
		m.z.x = zero;
		m.z.y = zero;
		m.z.z = 1;
		m.z.w = zero;
		m.w.x = zero;
		m.w.y = zero;
		m.w.z = zero;
		m.w.w = 1;
		TFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	ry	= Y]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44RotY(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN ry)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_Cos(ry);
	_TN s = F32_Sin(ry);
	_TN zero;
	zero -= zero;
	if( pm1 == nullptr )
	{
		pm0->x.x = +c;
		pm0->x.y = zero;
		pm0->x.z = +s;
		pm0->x.w = zero;
		pm0->y.x = zero;
		pm0->y.y = 1;
		pm0->y.z = zero;
		pm0->y.w = zero;
		pm0->z.x = -s;
		pm0->z.y = zero;
		pm0->z.z = +c;
		pm0->z.w = zero;
		pm0->w.x = zero;
		pm0->w.y = zero;
		pm0->w.z = zero;
		pm0->w.w = 1;
	}
	else
	{
		IrisTMtx44<_TN> m;
		m.x.x = +c;
		m.x.y = zero;
		m.x.z = +s;
		m.x.w = zero;
		m.y.x = zero;
		m.y.y = 1;
		m.y.z = zero;
		m.y.w = zero;
		m.z.x = -s;
		m.z.y = zero;
		m.z.z = +c;
		m.z.w = zero;
		m.w.x = zero;
		m.w.y = zero;
		m.w.z = zero;
		m.w.w = 1;
		TFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idy	= Y]ʁiCfbNXj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44RotIdxY(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, u16 idy)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_CosIdx(idy);
	_TN s = F32_SinIdx(idy);
	_TN zero;
	zero -= zero;
	if( pm1 == nullptr )
	{
		pm0->x.x = +c;
		pm0->x.y = zero;
		pm0->x.z = +s;
		pm0->x.w = zero;
		pm0->y.x = zero;
		pm0->y.y = 1;
		pm0->y.z = zero;
		pm0->y.w = zero;
		pm0->z.x = -s;
		pm0->z.y = zero;
		pm0->z.z = +c;
		pm0->z.w = zero;
		pm0->w.x = zero;
		pm0->w.y = zero;
		pm0->w.z = zero;
		pm0->w.w = 1;
	}
	else
	{
		IrisTMtx44<_TN> m;
		m.x.x = +c;
		m.x.y = zero;
		m.x.z = +s;
		m.x.w = zero;
		m.y.x = zero;
		m.y.y = 1;
		m.y.z = zero;
		m.y.w = zero;
		m.z.x = -s;
		m.z.y = zero;
		m.z.z = +c;
		m.z.w = zero;
		m.w.x = zero;
		m.w.y = zero;
		m.w.z = zero;
		m.w.w = 1;
		TFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rx	= X]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44RotX(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, _TN rx)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_Cos(rx);
	_TN s = F32_Sin(rx);
	_TN zero;
	zero -= zero;
	if( pm1 == nullptr )
	{
		pm0->x.x = 1;
		pm0->x.y = zero;
		pm0->x.z = zero;
		pm0->x.w = zero;
		pm0->y.x = zero;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->y.w = zero;
		pm0->z.x = zero;
		pm0->z.y = -s;
		pm0->z.z = +c;
		pm0->z.w = zero;
		pm0->w.x = zero;
		pm0->w.y = zero;
		pm0->w.z = zero;
		pm0->w.w = 1;
	}
	else
	{
		IrisTMtx44<_TN> m;
		m.x.x = 1;
		m.x.y = zero;
		m.x.z = zero;
		m.x.w = zero;
		m.y.x = zero;
		m.y.y = +c;
		m.y.z = +s;
		m.y.w = zero;
		m.z.x = zero;
		m.z.y = -s;
		m.z.z = +c;
		m.z.w = zero;
		m.w.x = zero;
		m.w.y = zero;
		m.w.z = zero;
		m.w.w = 1;
		TFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idx	= X]ʁiCfbNXj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44RotIdxX(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, u16 idx)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = F32_CosIdx(idx);
	_TN s = F32_SinIdx(idx);
	_TN zero;
	zero -= zero;
	if( pm1 == nullptr )
	{
		pm0->x.x = 1;
		pm0->x.y = zero;
		pm0->x.z = zero;
		pm0->x.w = zero;
		pm0->y.x = zero;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->y.w = zero;
		pm0->z.x = zero;
		pm0->z.y = -s;
		pm0->z.z = +c;
		pm0->z.w = zero;
		pm0->w.x = zero;
		pm0->w.y = zero;
		pm0->w.z = zero;
		pm0->w.w = 1;
	}
	else
	{
		IrisTMtx44<_TN> m;
		m.x.x = 1;
		m.x.y = zero;
		m.x.z = zero;
		m.x.w = zero;
		m.y.x = zero;
		m.y.y = +c;
		m.y.z = +s;
		m.y.w = zero;
		m.z.x = zero;
		m.z.y = -s;
		m.z.z = +c;
		m.z.w = zero;
		m.w.x = zero;
		m.w.y = zero;
		m.w.z = zero;
		m.w.w = 1;
		TFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Rot(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisTVec4<_TN>* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	TFpuMtx44RotZ(pm0, pm1, pv0->z);
	TFpuMtx44RotY(pm0, pm0, pv0->y);
	TFpuMtx44RotX(pm0, pm0, pv0->x);
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiCfbNXj
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44RotIdx(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1, const IrisSVec4* pv0)
{
	MATH_FPU_NULLASSERT( pv0 );
	TFpuMtx44RotIdxZ(pm0, pm1, (u16)pv0->z);
	TFpuMtx44RotIdxY(pm0, pm0, (u16)pv0->y);
	TFpuMtx44RotIdxX(pm0, pm0, (u16)pv0->x);
	return pm0;
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
template<typename _TN>IrisBool		TFpuMtx44IsUnit(const IrisTMtx44<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	if( pm0->x.x != 1 || pm0->y.y != 1 || pm0->z.z != 1 || pm0->w.w != 1 )
		return IRIS_FALSE;
	static const _TN ZERO = ::cpp0x::zero_traits<_TN>::value;
	if( pm0->x.y != ZERO || pm0->x.z != ZERO || pm0->x.w != ZERO
		|| pm0->y.x != ZERO || pm0->y.z != ZERO || pm0->y.w != ZERO
		|| pm0->z.x != ZERO || pm0->z.y != ZERO || pm0->z.w != ZERO
		|| pm0->w.x != ZERO || pm0->w.y != ZERO || pm0->w.z != ZERO )
		return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	}gbNX̃g[XԂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	}gbNX̃g[X
*/
template<typename _TN>_TN			TFpuMtx44Trace(const IrisTMtx44<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	return (pm0->x.x + pm0->y.y + pm0->z.z + pm0->w.w);
}

/**
 * @brief	s񎮂Ԃ
 * @param [in]	pm0	= ̓}gbNX
 * @return	s
*/
template<typename _TN>_TN			TFpuMtx44Determinant(const IrisTMtx44<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN a21 = pm0->x.y;
	_TN a31 = pm0->x.z;
	_TN a41 = pm0->x.w;
	_TN a22 = pm0->y.y;
	_TN a32 = pm0->y.z;
	_TN a42 = pm0->y.w;
	_TN a23 = pm0->z.y;
	_TN a33 = pm0->z.z;
	_TN a43 = pm0->z.w;
	_TN a24 = pm0->w.y;
	_TN a34 = pm0->w.z;
	_TN a44 = pm0->w.w;

	_TN d1 = pm0->x.x
		* (a22 * a33 * a44
		+  a23 * a34 * a42
		+  a24 * a32 * a43
		-  a24 * a33 * a42
		-  a23 * a32 * a44
		-  a22 * a34 * a43);
	_TN d2 = pm0->y.x
		* (a21 * a33 * a44
		+  a23 * a34 * a41
		+  a24 * a31 * a43
		-  a24 * a33 * a41
		-  a23 * a31 * a44
		-  a21 * a34 * a43);
	_TN d3 = pm0->z.x
		* (a21 * a32 * a44
		+  a22 * a34 * a41
		+  a24 * a31 * a42
		-  a24 * a32 * a41
		-  a22 * a31 * a44
		-  a21 * a34 * a42);
	_TN d4 = pm0->w.x
		* (a21 * a32 * a43
		+  a22 * a33 * a41
		+  a23 * a31 * a42
		-  a23 * a32 * a41
		-  a22 * a31 * a43
		-  a21 * a33 * a42);
	return (d1 - d2 + d3 - d4);
}

/**
 * @brief	]qsԂ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Adjoint(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	int i, j, k;
	IrisTVec4<_TN> v[3];
	IrisTMtx33<_TN> minor;
	IrisTMtx44<_TN>  adjoint;
	for(i=0; i < 4; ++i)
	{
		for(j=0; j < 4; ++j)
		{
			if(i==0)
			{
				v[0] = pm1->y;
				v[1] = pm1->z;
				v[2] = pm1->w;
			}
			else if(i==1)
			{
				v[0] = pm1->x;
				v[1] = pm1->z;
				v[2] = pm1->w;
			}
			else if(i==2)
			{
				v[0] = pm1->x;
				v[1] = pm1->y;
				v[2] = pm1->w;
			}
			else
			{
				v[0] = pm1->x;
				v[1] = pm1->y;
				v[2] = pm1->z;
			}
			if(j==0)
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].y;
					v[k].y = v[k].z;
					v[k].z = v[k].w;
				}
			}
			else if(j==1)
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].x;
					v[k].y = v[k].z;
					v[k].z = v[k].w;
				}
			}
			else if(j==2)
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].x;
					v[k].y = v[k].y;
					v[k].z = v[k].w;
				}
			}
			else
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].x;
					v[k].y = v[k].y;
					v[k].z = v[k].z;
				}
			}
			minor.x.x = v[0].x;
			minor.x.y = v[0].y;
			minor.x.z = v[0].z;
			minor.y.x = v[1].x;
			minor.y.y = v[1].y;
			minor.y.z = v[1].z;
			minor.z.x = v[2].x;
			minor.z.y = v[2].y;
			minor.z.z = v[2].z;

			if((i ^ j) & 1)
			{
				adjoint.m[j][i] = -TFpuMtx33Determinant(&minor);
			}
			else
			{
				adjoint.m[j][i] =  TFpuMtx33Determinant(&minor);
			}
		}
	}
	TFpuMtx44Copy(pm0, &adjoint);
	return pm0;
}

/**
 * @brief	tsԂ(s͉]+sړƉ)
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Inverse(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	IrisTMtx44<_TN> m;
	IrisTVec4<_TN> v;
	TFpuMtx44Copy(&m, pm1);
	_TN zero;
	zero -= zero;
	v.x = m.w.x; v.y = m.w.y; v.z = m.w.z;
	m.w.x = m.w.y = m.w.z = zero;

	TFpuMtx44Transpose(pm0, &m);
	TFpuMtx44TransformXYZ(&v, pm0, &v);
	pm0->w.x = -v.x;
	pm0->w.y = -v.y;
	pm0->w.z = -v.z;
	return pm0;
}

/**
 * @brief	tsԂ
 * @param [out]	pm0				= o̓}gbNX
 * @param [out]	pDeterminant	= 
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44Inverse2(IrisTMtx44<_TN>* pm0, _TN* pDeterminant, const IrisTMtx44<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	_TN zero;
	zero -= zero;
	_TN d = TFpuMtx44Determinant(pm1);
	if( pDeterminant != nullptr ) *pDeterminant = d;
	if( d == zero ) return nullptr;
	TFpuMtx44Adjoint(pm0, pm1);
	TFpuMtx44Scale(pm0, pm0, 1 / d );
	return pm0;
}

/**
 * @brief	ƕʂ蓊e}gNX߂
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pv0				= xNg
 * @param [in]	pl0				= 
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44DropShadow(IrisTMtx44<_TN>* pm0, const IrisTVec4<_TN>* pv0, const IrisFPlane* pl0)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pl0 );
	IrisTPlane<_TN> pln;
	TFpuPlaneNormalize(&pln, pl0);
	_TN d = TFpuVec4InnerProductXYZ((IrisTVec4<_TN>*)&pln, pv0);
	TFpuVec4Scale(&pm0->x, pv0, -pln.a);
	TFpuVec4Scale(&pm0->y, pv0, -pln.b);
	TFpuVec4Scale(&pm0->z, pv0, -pln.c);
	TFpuVec4Scale(&pm0->w, pv0, -pln.d);
	pm0->x.x += d;
	pm0->y.y += d;
	pm0->z.z += d;
	pm0->w.w += d;
	return pm0;
}

/**
 * @brief	}gNX̐K
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44NormalizeXYZ(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	// XY̊OςZ𓾂
	TFpuVec4OuterProductXYZ(&pm0->z, &pm1->x, &pm1->y);
	// Oς瓾ZY̊OςX𓾂
	TFpuVec4OuterProductXYZ(&pm0->x, &pm1->y, &pm0->z);
	TFpuVec4Normalize(&pm0->x, &pm0->x);
	TFpuVec4Normalize(&pm0->y, &pm1->y);
	TFpuVec4Normalize(&pm0->z, &pm0->z);
	pm0->w = pm1->w;
	return pm0;
}

/**
 * @brief	8rbg̐x؂̂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>IrisTMtx44<_TN>*	TFpuMtx44TruncatePrecision24(IrisTMtx44<_TN>* pm0, const IrisTMtx44<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	TFpuVec4TruncatePrecision24(&pm0->x, &pm1->x);
	TFpuVec4TruncatePrecision24(&pm0->y, &pm1->y);
	TFpuVec4TruncatePrecision24(&pm0->z, &pm1->z);
	TFpuVec4TruncatePrecision24(&pm0->w, &pm1->w);
	return pm0;
}

}	// end of namespace math
}	// end of namespace iris

#endif
