//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathTMatrix22.h
 * @brief		2x2}gbNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathTMatrix22_H_
#define INCG_IRIS_MathTMatrix22_H_

//======================================================================
// include
#include "MathTVector2.h"

namespace iris {
namespace math
{

//======================================================================
// declare
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Unit(IrisTMtx22<_TN>* pm0);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Zero(IrisTMtx22<_TN>* pm0);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Copy(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1);
template<typename _TN>IrisTVec2<_TN>*		TFpuMtx22Transform(IrisTVec2<_TN>* pv0, const IrisTMtx22<_TN>* pm0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Mul(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, const IrisTMtx22<_TN>* pm2);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Scale(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, _TN s);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Transpose(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22RotZ(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, _TN rz);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22RotIdxZ(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, u16 idz);
template<typename _TN>IrisBool				TFpuMtx22IsUnit(const IrisTMtx22<_TN>* pm0);
template<typename _TN>_TN					TFpuMtx22Trace(const IrisTMtx22<_TN>* pm0);
template<typename _TN>_TN					TFpuMtx22Determinant(const IrisTMtx22<_TN>* pm0);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Adjoint(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22TruncatePrecision24(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1);

template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Identity(IrisTMtx22<_TN>* pm0);
template<typename _TN>IrisTMtx22<_TN>*		TFpuMtx22Null(IrisTMtx22<_TN>* pm0);
template<typename _TN>IrisTVec2<_TN>*		TFpuMtx22Apply(IrisTVec2<_TN>* pv0, const IrisTMtx22<_TN>* pm0, const IrisTVec2<_TN>* pv1);
template<typename _TN>IrisBool				TFpuMtx22IsIdentity(const IrisTMtx22<_TN>* pm0);

//======================================================================
// function
/**
 * @brief	Pʍs̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22Unit(IrisTMtx22<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN zero;
	zero -= zero;	// make 0
	pm0->x.x = pm0->y.y = zero;
	pm0->x.y = pm0->y.x = zero;
	return pm0;
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22Zero(IrisTMtx22<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	TFpuVec2Zero(&pm0->x);
	TFpuVec2Zero(&pm0->y);
	return pm0;
}

/**
 * @brief	}gbNX̃Rs[
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22Copy(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
#if 1
	pm0->_00 = pm1->_00;
	pm0->_01 = pm1->_01;
	pm0->_10 = pm1->_10;
	pm0->_11 = pm1->_11;
#else
	FpuVec2Copy(&pm0->x, &pm1->x);
	FpuVec2Copy(&pm0->y, &pm1->y);
#endif
	return pm0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*		TFpuMtx22Transform(IrisTVec2<_TN>* pv0, const IrisTMtx22<_TN>* pm0, const IrisTVec2<_TN>* pv1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	_TN x = pv1->x;
	pv0->x = pm0->x.x * x + pm0->y.x * pv1->y;
	pv0->y = pm0->x.y * x + pm0->y.y * pv1->y;
	return pv0;
}

/**
 * @brief	Q̃}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pm2	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22Mul(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, const IrisTMtx22<_TN>* pm2)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	MATH_FPU_NULLASSERT( pm2 );
	IRIS_ASSERT( pm0 != pm1 );
	IRIS_ASSERT( pm0 != pm2 );
	pm0->x.x = pm1->x.x * pm2->x.x + pm1->y.x * pm2->x.y;
	pm0->x.y = pm1->x.y * pm2->x.x + pm1->y.y * pm2->x.y;
	pm0->y.x = pm1->x.x * pm2->y.x + pm1->y.x * pm2->y.y;
	pm0->y.y = pm1->x.y * pm2->y.x + pm1->y.y * pm2->y.y;
	return pm0;
}

/**
 * @brief	}gbNX̃XP[O
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	s	= XJ[l
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22Scale(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, _TN s)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	FpuVec2Scale(&pm0->x, &pm1->x, s);
	FpuVec2Scale(&pm0->y, &pm1->y, s);
	return pm0;
}

/**
 * @brief	}gbNX̓]us߂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22Transpose(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	pm0->x.x = pm1->x.x;
	_TN f = pm1->x.y;
	pm0->x.y = pm1->y.x;
	pm0->y.x = f;
	pm0->y.y = pm1->y.y;
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22RotZ(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, _TN rz)
{
	MATH_FPU_NULLASSERT( pm0 );
	_TN c = cos(rz);
	_TN s = sin(rz);
	if( pm1 == nullptr )
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->y.x = -s;
		pm0->y.y = +c;
	}
	else
	{
		IrisTMtx22<_TN> m;
		m.x.x = +c;
		m.x.y = +s;
		m.y.x = -s;
		m.y.y = +c;
		TFpuMtx22Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idz	= Z]ʁiCfbNXj
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22RotIdxZ(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1, u16 idz)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_CosIdx(idz);
	f32 s = F32_SinIdx(idz);
	if( pm1 == nullptr )
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->y.x = -s;
		pm0->y.y = +c;
	}
	else
	{
		IrisTMtx22<_TN> m;
		m.x.x = +c;
		m.x.y = +s;
		m.y.x = -s;
		m.y.y = +c;
		TFpuMtx22Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
template<typename _TN>
IrisBool				TFpuMtx22IsUnit(const IrisTMtx22<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	if( pm0->x.x != 1 || pm0->y.y != 1 )
		return IRIS_FALSE;
	if( pm0->x.y != 0 || pm0->y.x != 0 )
		return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	}gbNX̃g[XԂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	}gbNX̃g[X
*/
template<typename _TN>
_TN					TFpuMtx22Trace(const IrisTMtx22<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	return (pm0->x.x + pm0->y.y);
}

/**
 * @brief	s񎮂Ԃ
 * @param [in]	pm0	= ̓}gbNX
 * @return	s
*/
template<typename _TN>
_TN					TFpuMtx22Determinant(const IrisTMtx22<_TN>* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	return (pm0->x.x * pm0->y.y) - (pm0->x.y * pm0->y.x);
}

/**
 * @brief	]qsԂ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*		TFpuMtx22Adjoint(IrisTMtx22<_TN>* pm0, const IrisTMtx22<_TN>* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	pm0->x.x =  pm1->x.x;
	pm0->x.y = -pm1->x.y;
	pm0->y.x = -pm1->y.x;
	pm0->y.y =  pm1->y.y;
	return pm0;
}

/**
 * @brief	Pʍs̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*	TFpuMtx22Identity(IrisTMtx22<_TN>* pm0)
{
	return TFpuMtx22Unit(pm0);
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
template<typename _TN>
IrisTMtx22<_TN>*	TFpuMtx22Null(IrisTMtx22<_TN>* pm0)
{
	return TFpuMtx22Zero(pm0);
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
template<typename _TN>
IrisTVec2<_TN>*	TFpuMtx22Apply(IrisTVec2<_TN>* pv0, const IrisTMtx22<_TN>* pm0, const IrisTVec2<_TN>* pv1)
{
	return TFpuMtx22Transform(pv0, pm0, pv1);
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
template<typename _TN>
IrisBool			TFpuMtx22IsIdentity(const IrisTMtx22<_TN>* pm0)
{
	return TFpuMtx22IsUnit(pm0);
}

}	// end of namespace math
}	// end of namespace iris


#endif
