//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathSHA256.cpp
 * @brief		SHA256 t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_MathSHA256_CPP_
#define INCG_IRIS_MathSHAPrivate_CPP_	// private֐gp

//======================================================================
// include
#include "MathSHA256.h"
#include "MathSHACommon.h"
#include "../../misc/iris_allegrex.h"
#include "../../iris_debug.h"

namespace iris {
namespace math
{

//======================================================================
// const
static const int COUNT_HIGH			= 0;
static const int COUNT_LOW			= 1;

static const int MESSAGE_BLK_SIZE	= SHA256_MSGBLKSIZE;
static const int MESSAGE_PAD_SIZE	= 56;	// SHA256_MSGBLKSIZE-8

#define SHTR(x, n)	((x) >> n)
#define ROTL	iris_allegrex_rotl
#define ROTR	iris_allegrex_rotr

#define SHA256AddLength(ctx, len)					\
	(ctx)->uCount[COUNT_LOW] += len					\
	, ((ctx)->uCount[COUNT_LOW] != 0 ) ? 0 :		\
	(++(ctx)->uCount[COUNT_HIGH]					\
	, ((ctx)->uCount[COUNT_HIGH] != 0) ? 0 : 1 )

//======================================================================
// declare
// gXtH[֐
static void _SHA256Transform(u32* lpState, const u8* lpBuffer);

//======================================================================
// function
/**********************************************************************//**
 *
 * SHA256pReLXg̏
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @return	
*//***********************************************************************/
void	SHA256InitContext(LPSHA256CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	lpContext->uCount[0] = lpContext->uCount[1] = 0;
	lpContext->uIndex = 0;
	lpContext->uState[0] = 0x6A09E667;
	lpContext->uState[1] = 0xBB67AE85;
	lpContext->uState[2] = 0x3C6EF372;
	lpContext->uState[3] = 0xA54FF53A;
	lpContext->uState[4] = 0x510E527F;
	lpContext->uState[5] = 0x9B05688C;
	lpContext->uState[6] = 0x1F83D9AB;
	lpContext->uState[7] = 0x5BE0CD19;
	//ZeroMemory(lpContext->uBuffer, sizeof(lpContext->uBuffer));
}

/**********************************************************************//**
 *
 * SHA256pReLXg̃NA
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @return	
*//***********************************************************************/
void	SHA256ClearContext(LPSHA256CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	SHA256InitContext(lpContext);
	ZeroMemory(lpContext->uBuffer, sizeof(lpContext->uBuffer));
}

/**********************************************************************//**
 *
 * SHA256vZ
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA256Update(LPSHA256CONTEXT lpContext, const u8* lpBuffer, size_t uLength)
{
	IRIS_ASSERT( lpContext != nullptr );
	IRIS_ASSERT( lpBuffer != nullptr );
	const u8* buf = lpBuffer;
	u16 index = lpContext->uIndex;
	while( uLength-- )
	{
		lpContext->uBuffer[index++] = *buf;
		if( index == MESSAGE_BLK_SIZE )
		{
			_SHA256Transform(lpContext->uState, lpContext->uBuffer);
			index = 0;
		}
		if( SHA256AddLength(lpContext, 8) )
		{
			IRIS_WARNING("message is too long.");
			break;
		}
		++buf;
	}
	lpContext->uIndex = index;
}

/**********************************************************************//**
 *
 * ŏIISHA256vZ
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
*//***********************************************************************/
void	SHA256Final(LPSHA256CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	u16 index = lpContext->uIndex;
	if( index >= MESSAGE_PAD_SIZE )
	{
		lpContext->uBuffer[index++] = 0x80;
		while( index < MESSAGE_BLK_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
		_SHA256Transform(lpContext->uState, lpContext->uBuffer);
		index = 0;
		while( index < MESSAGE_PAD_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
	}
	else
	{
		lpContext->uBuffer[index++] = 0x80;
		while( index < MESSAGE_PAD_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
	}
	_SHAEncode32(lpContext->uBuffer+MESSAGE_PAD_SIZE, lpContext->uCount, 8);
	_SHA256Transform(lpContext->uState, lpContext->uBuffer);
	lpContext->uIndex = 0;
}

/**********************************************************************//**
 *
 * SHA256o
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [out]	lpBuffer	= o̓obt@
*//***********************************************************************/
void	SHA256Output(LPCSHA256CONTEXT lpContext, u8* lpBuffer)
{
	_SHAEncode32(lpBuffer, lpContext->uState, SHA256_HASHSIZE);
}

/**********************************************************************//**
 *
 * SHA256܂Ƃ߂ČvZ(Init ` Final)
 *
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA256Encode(LPSHA256CONTEXT lpContext, const u8* lpBuffer, size_t uLength)
{
	SHA256InitContext(lpContext);
	SHA256Update(lpContext, lpBuffer, uLength);
	SHA256Final(lpContext);
}

/**********************************************************************//**
 *
 * SHA256܂Ƃ߂ČvZ(Init ` Output)
 *
 ----------------------------------------------------------------------
 * @param [io]	lpDst		= o̓obt@
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA256Encode(u8* lpDst, const u8* lpBuffer, size_t uLength)
{
	SHA256CONTEXT ctx;
	SHA256InitContext(&ctx);
	SHA256Update(&ctx, lpBuffer, uLength);
	SHA256Final(&ctx);
	SHA256Output(&ctx, lpDst);
}

/**********************************************************************//**
 *
 * SHA256o͒l𕶎ɕϊ
 *
 ----------------------------------------------------------------------
 * @param [out]	lpString	= o̓obt@
 * @param [in]	uSize		= o̓obt@TCY
 * @param [in]	lpMD5		= ̓obt@(MD5)
*//***********************************************************************/
LPTSTR	SHA256ToString (LPTSTR lpString, size_t uSize, const u8* lpSHA256)
{
	return SHAToString (lpString, uSize, lpSHA256, SHA256_HASHSIZE);
}
LPSTR	SHA256ToStringA(LPSTR  lpString, size_t uSize, const u8* lpSHA256)
{
	return SHAToStringA(lpString, uSize, lpSHA256, SHA256_HASHSIZE);
}
LPWSTR	SHA256ToStringW(LPWSTR lpString, size_t uSize, const u8* lpSHA256)
{
	return SHAToStringW(lpString, uSize, lpSHA256, SHA256_HASHSIZE);
}

// transform
void _SHA256Transform(u32* lpState, const u8* lpBuffer)
{
#define SIGMA0(w)	(ROTR(w,  2) ^ ROTR(w, 13) ^ ROTR(w, 22))
#define SIGMA1(w)	(ROTR(w,  6) ^ ROTR(w, 11) ^ ROTR(w, 25))
#define sigma0(w)	(ROTR(w,  7) ^ ROTR(w, 18) ^ SHTR(w,  3))
#define sigma1(w)	(ROTR(w, 17) ^ ROTR(w, 19) ^ SHTR(w, 10))

#define F1(x, y, z)	_SHA_Ch(x, y, z)
#define F2(x, y, z)	_SHA_Maj(x, y, z)

	static const u32 K[64] =
	{
		  0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5
		, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5
		, 0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3
		, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174
		, 0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc
		, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da
		, 0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7
		, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967
		, 0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13
		, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85
		, 0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3
		, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070
		, 0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5
		, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3
		, 0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208
		, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
	};

	int i=0;
	u32 W[64];
	u32 A = *(lpState);
	u32 B = *(lpState+1);
	u32 C = *(lpState+2);
	u32 D = *(lpState+3);
	u32 E = *(lpState+4);
	u32 F = *(lpState+5);
	u32 G = *(lpState+6);
	u32 H = *(lpState+7);
	u32 tmp1, tmp2;

	for( ; i < 16; ++i )
	{
		W[i] = IRIS_ByteArray2DWordBE(lpBuffer+i*4);
	}
	for( ; i < 64; ++i )
	{
		W[i] = sigma1(W[i-2]) + W[i-7] + sigma0(W[i-15]) + W[i-16];
	}

	for( i=0; i < 64; ++i )
	{
		tmp1 = H + SIGMA1(E) + F1(E, F, G) + W[i] + K[i];
		tmp2 = SIGMA0(A) + F2(A, B, C);
		H = G;
		G = F;
		F = E;
		E = D + tmp1;
		D = C;
		C = B;
		B = A;
		A = tmp1 + tmp2;
	}

	*(lpState)   += A;
	*(lpState+1) += B;
	*(lpState+2) += C;
	*(lpState+3) += D;
	*(lpState+4) += E;
	*(lpState+5) += F;
	*(lpState+6) += G;
	*(lpState+7) += H;
}

}	// end of namespace math
}	// end of namespace iris

#if	(defined(_IRIS_SUPPORT_GOOGLETEST) || defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/gt/gt_inchead.h"
#include "../../iris_using.h"

TEST(CMathSHA256Test, Function)
{
	char comm[256] = "";
	u8 out[SHA256_HASHSIZE];

	SHA256CONTEXT ctx;

	// TEST1
	strcpy_s(comm, 256, "abc");
	SHA256Encode(out, comm, strlen(comm));
	SHA256ToStringA(comm, 256, out);
	ASSERT_STREQ(
		"ba7816bf8f01cfea"
		"414140de5dae2223b00361a396177a9cb410ff61f20015ad"
		, comm );

	// TEST2
	strcpy_s(comm, 256, "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq");
	SHA256Encode(out, comm, strlen(comm));
	SHA256ToStringA(comm, 256, out);
	ASSERT_STREQ(
		"248d6a61d20638b8"
        "e5c026930c3e6039a33ce45964ff2167f6ecedd419db06c1"
		, comm );

	// TEST3
	strcpy_s(comm, 256, "a");
	SHA256InitContext(&ctx);
	for( int i=0; i < 1000000; ++i )
		SHA256Update(&ctx, comm, strlen(comm));
	SHA256Final(&ctx);
	SHA256Output(&ctx, out);
	SHA256ToStringA(comm, 256, out);
	ASSERT_STREQ(
		"cdc76e5c9914fb92"
        "81a1c7e284d73e67f1809a48a497200e046d39ccc7112cd0"
		, comm );

	// TEST4
	strcpy_s(comm, 256, "01234567012345670123456701234567""01234567012345670123456701234567");
	SHA256InitContext(&ctx);
	for( int i=0; i < 10; ++i )
		SHA256Update(&ctx, comm, strlen(comm));
	SHA256Final(&ctx);
	SHA256Output(&ctx, out);
	SHA256ToStringA(comm, 256, out);
	ASSERT_STREQ(
		"594847328451bdfa"
        "85056225462cc1d867d877fb388df0ce35f25ab5562bfbb5"
		, comm );
}

#endif

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "../../iris_using.h"
#include "../../iris_iostream.h"

//======================================================================
// test
IRIS_UNITTEST(CMathSHA256UnitTest, Func)
{
	char comm[256] = "";
	u8 out[SHA256_HASHSIZE];

	std::clog << "SHA256GR[h܂B" << std::endl;
	std::safe_cin >> comm;

	SHA256Encode(out, comm, strlen(comm));

	for( int i=0; i < SHA256_HASHSIZE; ++i )
		printf("%.2x", out[i]);
	std::cout << std::endl;
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
