//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathMatrix22.inl
 * @brief		2x2}gbNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathMatrix22_inl_
#define INCG_IRIS_MathMatrix22_inl_

namespace iris {
namespace math
{

//======================================================================
// function
/**
 * @brief	Pʃ}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22* FpuMtx22Unit(IrisFMtx22* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	pm0->x.x = pm0->y.y = 1.0f;
	pm0->x.y = pm0->y.x = 0.0f;
	return pm0;
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*	FpuMtx22Zero(IrisFMtx22* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	FpuVec2Zero(&pm0->x);
	FpuVec2Zero(&pm0->y);
	return pm0;
}

/**
 * @brief	}gbNX̃Rs[
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*	FpuMtx22Copy(IrisFMtx22* pm0, const IrisFMtx22* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
#if 1
	pm0->_00 = pm1->_00;
	pm0->_01 = pm1->_01;
	pm0->_10 = pm1->_10;
	pm0->_11 = pm1->_11;
#else
	FpuVec2Copy(&pm0->x, &pm1->x);
	FpuVec2Copy(&pm0->y, &pm1->y);
#endif
	return pm0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_FPU_INLINE IrisFVec2*		FpuMtx22Transform(IrisFVec2* pv0, const IrisFMtx22* pm0, const IrisFVec2* pv1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pv0 );
	MATH_FPU_NULLASSERT( pv1 );
	f32 x = pv1->x;
	pv0->x = F32_Mul(pm0->x.x, x) + F32_Mul(pm0->y.x, pv1->y);
	pv0->y = F32_Mul(pm0->x.y, x) + F32_Mul(pm0->y.y, pv1->y);
	return pv0;
}

/**
 * @brief	Q̃}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pm2	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*		FpuMtx22Mul(IrisFMtx22* pm0, const IrisFMtx22* pm1, const IrisFMtx22* pm2)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	MATH_FPU_NULLASSERT( pm2 );
	IRIS_ASSERT( pm0 != pm1 );
	IRIS_ASSERT( pm0 != pm2 );
	pm0->x.x = F32_Mul(pm1->x.x, pm2->x.x) + F32_Mul(pm1->y.x, pm2->x.y);
	pm0->x.y = F32_Mul(pm1->x.y, pm2->x.x) + F32_Mul(pm1->y.y, pm2->x.y);
	pm0->y.x = F32_Mul(pm1->x.x, pm2->y.x) + F32_Mul(pm1->y.x, pm2->y.y);
	pm0->y.y = F32_Mul(pm1->x.y, pm2->y.x) + F32_Mul(pm1->y.y, pm2->y.y);
	return pm0;
}

/**
 * @brief	}gbNX̃XP[O
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	s	= XJ[l
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*		FpuMtx22Scale(IrisFMtx22* pm0, const IrisFMtx22* pm1, f32 s)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	FpuVec2Scale(&pm0->x, &pm1->x, s);
	FpuVec2Scale(&pm0->y, &pm1->y, s);
	return pm0;
}

/**
 * @brief	}gbNX̓]us߂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*		FpuMtx22Transpose(IrisFMtx22* pm0, const IrisFMtx22* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	pm0->x.x = pm1->x.x;
	f32 f = pm1->x.y;
	pm0->x.y = pm1->y.x;
	pm0->y.x = f;
	pm0->y.y = pm1->y.y;
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*		FpuMtx22RotZ(IrisFMtx22* pm0, const IrisFMtx22* pm1, f32 rz)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_Cos(rz);
	f32 s = F32_Sin(rz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->y.x = -s;
		pm0->y.y = +c;
	}
	else
	{
		IrisFMtx22 m;
		m.x.x = +c;
		m.x.y = +s;
		m.y.x = -s;
		m.y.y = +c;
		FpuMtx22Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idz	= Z]ʁiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*		FpuMtx22RotIdxZ(IrisFMtx22* pm0, const IrisFMtx22* pm1, u16 idz)
{
	MATH_FPU_NULLASSERT( pm0 );
	f32 c = F32_CosIdx(idz);
	f32 s = F32_SinIdx(idz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->y.x = -s;
		pm0->y.y = +c;
	}
	else
	{
		IrisFMtx22 m;
		m.x.x = +c;
		m.x.y = +s;
		m.y.x = -s;
		m.y.y = +c;
		FpuMtx22Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
IRIS_FPU_INLINE IrisBool		FpuMtx22IsUnit(const IrisFMtx22* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	const IrisMtx22* m = (const IrisMtx22*)(pm0);
	if( m->im.x.x != 0x3F800000 || m->im.y.y != 0x3F800000 )
		return IRIS_FALSE;
	if( (m->im.x.y | m->im.y.x) != 0 )
		return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	}gbNX̃g[XԂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	}gbNX̃g[X
*/
IRIS_FPU_INLINE f32				FpuMtx22Trace(const IrisFMtx22* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	return (pm0->x.x + pm0->y.y);
}

/**
 * @brief	s񎮂Ԃ
 * @param [in]	pm0	= ̓}gbNX
 * @return	s
*/
IRIS_FPU_INLINE f32				FpuMtx22Determinant(const IrisFMtx22* pm0)
{
	MATH_FPU_NULLASSERT( pm0 );
	return (F32_Mul(pm0->x.x, pm0->y.y) - F32_Mul(pm0->x.y, pm0->y.x));
}

/**
 * @brief	]qsԂ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*		FpuMtx22Adjoint(IrisFMtx22* pm0, const IrisFMtx22* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	pm0->x.x =  pm1->x.x;
	pm0->x.y = -pm1->x.y;
	pm0->y.x = -pm1->y.x;
	pm0->y.y =  pm1->y.y;
	return pm0;
}

/**
 * @brief	8rbg̐x؂̂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_FPU_INLINE IrisFMtx22*		FpuMtx22TruncatePrecision24(IrisFMtx22* pm0, const IrisFMtx22* pm1)
{
	MATH_FPU_NULLASSERT( pm0 );
	MATH_FPU_NULLASSERT( pm1 );
	FpuVec2TruncatePrecision24(&pm0->x, &pm1->x);
	FpuVec2TruncatePrecision24(&pm0->y, &pm1->y);
	return pm0;
}


}	// end of namespace math
}	// end of namespace iris

#endif
