//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		GXGL.cpp
 * @brief		gl g֐t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_GXGL_CPP_

//======================================================================
// include
#include "GXGL.h"
#include "../../math/fpu/MathPerspective.h"

#if	defined(_IRIS_SUPPORT_OPENGL)

//======================================================================
// link
#if		defined(_WIN32)
#pragma	comment( lib, "opengl32.lib")
#endif

//======================================================================
// define
#define GXGL_FVF_INDEX_TEXTURE		(0)
#define GXGL_FVF_INDEX_COLOR_RGB	(1)
#define GXGL_FVF_INDEX_COLOR_RGBA	(2)
#define GXGL_FVF_INDEX_NORMAL		(3)
#define GXGL_FVF_INDEX_VERTEX_XYZ	(4)
#define GXGL_FVF_INDEX_VERTEX_XY	(5)
#define GXGL_FVF_INDEX_VERTEX_X		(6)
#define GXGL_FVF_INDEX_VERTEX_XYZW	(7)
#define GXGL_FVF_INDEX_NUM			(8)

#define GXGL_FVF_INDEX_TABLE_NUM	(5)

namespace iris {
namespace gx {
namespace gl
{

//======================================================================
// struct
typedef struct FVFINFO
{
	GLenum type;
	int size;
}FVFINFO;

//======================================================================
// variable
static f32 s_ProjectionMatrix[16] =
{
	1.0f, 0.0f, 0.0f, 0.0f,
	0.0f, 1.0f, 0.0f, 0.0f,
	0.0f, 0.0f, 1.0f, 0.0f,
	0.0f, 0.0f, 0.0f, 1.0f,
};
static f32 s_2DProjectionMatrix[16] =
{
	1.0f, 0.0f, 0.0f, 0.0f,
	0.0f, 1.0f, 0.0f, 0.0f,
	0.0f, 0.0f, 1.0f, 0.0f,
	0.0f, 0.0f, 0.0f, 1.0f,
};
static f32 s_ViewMatrix[16] =
{
	1.0f, 0.0f, 0.0f, 0.0f,
	0.0f, 1.0f, 0.0f, 0.0f,
	0.0f, 0.0f, 1.0f, 0.0f,
	0.0f, 0.0f, 0.0f, 1.0f,
};
static f32 s_WorldMatrix[16] =
{
	1.0f, 0.0f, 0.0f, 0.0f,
	0.0f, 1.0f, 0.0f, 0.0f,
	0.0f, 0.0f, 1.0f, 0.0f,
	0.0f, 0.0f, 0.0f, 1.0f,
};
static const FVFINFO s_FVFInfoTable[GXGL_FVF_INDEX_NUM][GXGL_FVF_INDEX_TABLE_NUM] =
{
	{
		{0, 0},
		{GL_BYTE, sizeof(char) * 2},
		{GL_SHORT, sizeof(short) * 2},
		{GL_FLOAT, sizeof(float) * 2},
		{0, 0},
	},	// GXFVF_INDEX_TEXTURE
	{
		{0, 0},
		{GL_UNSIGNED_BYTE	, sizeof(char) * 3},
		{GL_UNSIGNED_SHORT	, sizeof(char) * 3},
		{GL_FLOAT			, sizeof(char) * 3},
		{0, 0},
	},	// GXFVF_INDEX_COLOR_RGB
	{
		{GL_UNSIGNED_BYTE	, sizeof(char) * 4},
		{GL_UNSIGNED_SHORT	, sizeof(char) * 4},
		{GL_FLOAT			, sizeof(char) * 4},
		{GL_UNSIGNED_BYTE	, sizeof(char) * 4},
		{0, 0},
	},	// GXFVF_INDEX_COLOR_RGBA
	{
		{0, 0},
		{GL_BYTE, sizeof(char) * 3},
		{GL_SHORT, sizeof(short) * 3},
		{GL_FLOAT, sizeof(float) * 3},
		{0, 0},
	},	// GXFVF_INDEX_NORMAL
	{
		{0, 0},
		{GL_BYTE, sizeof(char) * 3},
		{GL_SHORT, sizeof(short) * 3},
		{GL_FLOAT, sizeof(float) * 3},
		{0, 0},
	},	// GXFVF_INDEX_VERTEX_XYZ
	{
		{0, 0},
		{GL_BYTE, sizeof(char) * 2},
		{GL_SHORT, sizeof(short) * 2},
		{GL_FLOAT, sizeof(float) * 2},
		{0, 0},
	},	// GXFVF_INDEX_VERTEX_XY
	{
		{0, 0},
		{GL_BYTE, sizeof(char) * 1},
		{GL_SHORT, sizeof(short) * 1},
		{GL_FLOAT, sizeof(float) * 1},
		{0, 0},
	},	// GXFVF_INDEX_VERTEX_X
	{
		{0, 0},
		{GL_BYTE, sizeof(char) * 4},
		{GL_SHORT, sizeof(short) * 4},
		{GL_FLOAT, sizeof(float) * 4},
		{0, 0},
	},	// GXFVF_INDEX_VERTEX_XYZW
};
static const int s_VertexSizeTable[4] =
{
	3,	// GXGL_FVF_INDEX_VERTEX_XYZ
	2,	// GXGL_FVF_INDEX_VERTEX_XY
	1,	// GXGL_FVF_INDEX_VERTEX_X
	4,	// GXGL_FVF_INDEX_VERTEX_XYZW
};
static struct
{
	IrisUInt	offset;
	void*		buffer;
	IrisUInt	size;
} s_DisplayList = { 0, nullptr, 0 };

//======================================================================
// function
/**********************************************************************//**
 *
 * 擾
 *
 ----------------------------------------------------------------------
 * @return 
*//***********************************************************************/
void* gxglGetMemory(int nSize)
{
	unsigned int _size = s_DisplayList.offset + nSize;
	void *p = NULL;
	if(_size <= s_DisplayList.size)
	{
		p = pointer_cast<char*>(s_DisplayList.buffer) + s_DisplayList.offset;
		s_DisplayList.offset = _size;
	}
	return p;
}

/**********************************************************************//**
 *
 * gxgl̊Jn
 *
 ----------------------------------------------------------------------
 * @param [in]	pList		= ƃobt@
 * @param [in]	nSize		= ƃobt@TCY
*//***********************************************************************/
void gxglStart(void* pList, int nSize)
{
	s_DisplayList.offset = 0;
	s_DisplayList.buffer = pList;
	s_DisplayList.size = static_cast<unsigned int>(nSize);
}

/**********************************************************************//**
 *
 * gxgl̊Jn
 *
 ----------------------------------------------------------------------
 * @param [in]	pList		= ƃobt@
 * @param [in]	nSize		= ƃobt@TCY
*//***********************************************************************/
int gxglFinish(void)
{
	glFinish();
	return static_cast<int>(s_DisplayList.offset);
}

/**********************************************************************//**
 *
 * Begin
 *
 ----------------------------------------------------------------------
 * @param [in]	mode		= [h
*//***********************************************************************/
void gxglBegin(GLenum mode)
{
	glBegin(mode);
}

/**********************************************************************//**
 *
 * End
 *
*//***********************************************************************/
void gxglEnd(void)
{
	glEnd();
}

/**********************************************************************//**
 *
 * glClearColor wrap
 *
 ----------------------------------------------------------------------
 * @param [in]	color	= NAJ[(ABGR8888)
*//***********************************************************************/
void gxglClearColor(GLuint color)
{
	IrisF32 a = ((color >> 24) & 0xFF) / 255.0f;
	IrisF32 b = ((color >> 16) & 0xFF) / 255.0f;
	IrisF32 g = ((color >>  8) & 0xFF) / 255.0f;
	IrisF32 r = ((color >>  0) & 0xFF) / 255.0f;

	glClearColor(r, g, b, a);
}

/**********************************************************************//**
 *
 * DrawArray
 *
 ----------------------------------------------------------------------
 * @param [in]	nPrim		= v~eBu
 * @param [in]	nVtype		= ^Cv
 * @param [in]	nCount		= _
 * @param [in]	pIndices	= CfbNXobt@
 * @param [in]	pVertices	= _obt@
*//***********************************************************************/
void gxglDrawArray(GLenum nPrim, GLuint nVtype, int nCount, const void* pIndices, const void* pVertices)
{
	GLuint	vtype = nVtype;
	GLenum	type=0;
	GLint	size=0, vsize=0;
	const u8* vertices = pointer_cast<const u8*>(pVertices);
	const GLsizei vertex_size_index = static_cast<GLsizei>(GXGL_GET_VERTEX_SIZE_BITS(vtype));
	const GLint color_a = static_cast<GLint>((GXGL_GET_COLOR_BITS(vtype) >> 2) & 0x1);
	const FVFINFO *pTextureFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_TEXTURE][GXGL_GET_TEXTURE_BITS(vtype)];
	const FVFINFO *pColorFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_COLOR_RGB+color_a][GXGL_GET_COLOR_BITS(vtype) ^ 0x4];
	const FVFINFO *pNormalFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_NORMAL][GXGL_GET_NORMAL_BITS(vtype)];
	const FVFINFO *pVertexFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_VERTEX_XYZ + vertex_size_index][GXGL_GET_VERTEX_BITS(vtype)];

	vsize += pTextureFVFInfo->size;
	vsize += pColorFVFInfo->size;
	vsize += pNormalFVFInfo->size;
	vsize += pVertexFVFInfo->size;

	// texture
	type = pTextureFVFInfo->type;
	size = pTextureFVFInfo->size;
	if(size > 0)
	{
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2, type, vsize, vertices);
		vertices += size;
	}

	// color
	type = pColorFVFInfo->type;
	size = pColorFVFInfo->size;
	if(size > 0)
	{
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(3+color_a, type, vsize, vertices);
		vertices += size;
	}

	// normal
	type = pNormalFVFInfo->type;
	size = pNormalFVFInfo->size;
	if(size > 0)
	{
		glEnableClientState(GL_NORMAL_ARRAY);
		glNormalPointer(type, vsize, vertices);
		vertices += size;
	}

	// vertex
	type = pVertexFVFInfo->type;
	size = pVertexFVFInfo->size;
	if(size > 0)
	{
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(s_VertexSizeTable[vertex_size_index], type, vsize, vertices);
		vertices += size;
	}

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	if(vtype & GXGL_TRANSFORM_2D)
	{
		// 2D[h
		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadMatrixf(s_2DProjectionMatrix);
		glMatrixMode(GL_MODELVIEW);
	}
	else
	{
		// 3D[h
		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadMatrixf(s_ProjectionMatrix);
		glMatrixMode(GL_MODELVIEW);

		glMultMatrixf(s_ViewMatrix);
		glMultMatrixf(s_WorldMatrix);
	}

	if(pIndices == nullptr)	glDrawArrays(nPrim, 0, nCount);
	else					glDrawElements(nPrim, nCount, GL_UNSIGNED_SHORT, pIndices);

	// ㏈
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();

	if(vtype & GXGL_TEXTURE_BITS)	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	if(vtype & GXGL_COLOR_BITS)		glDisableClientState(GL_COLOR_ARRAY);
	if(vtype & GXGL_NORMAL_BITS)	glDisableClientState(GL_NORMAL_ARRAY);
	if(vtype & GXGL_VERTEX_BITS)	glDisableClientState(GL_VERTEX_ARRAY);
}

/**********************************************************************//**
 *
 * DrawArrayN
 *
 ----------------------------------------------------------------------
 * @param [in]	nPrim		= v~eBu
 * @param [in]	nVtype		= ^Cv
 * @param [in]	nCount		= _
 * @param [in]	nPrimCount	= v~eBu
 * @param [in]	pIndices	= CfbNXobt@
 * @param [in]	pVertices	= _obt@
*//***********************************************************************/
void gxglDrawArrayN(GLenum nPrim, GLuint nVtype, int nCount, int nPrimCount, const void* pIndices, const void* pVertices)
{
	GLuint	vtype = nVtype;
	GLenum	type=0;
	GLint	size=0, vsize=0;
	const u8* vertices = pointer_cast<const u8*>(pVertices);
	u8* temp_vertices = const_cast<u8*>(vertices);
	const GLsizei vertex_size_index = static_cast<GLsizei>(GXGL_GET_VERTEX_SIZE_BITS(vtype));
	const GLint color_a = static_cast<GLint>((GXGL_GET_COLOR_BITS(vtype) >> 2) & 0x1);
	const FVFINFO *pTextureFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_TEXTURE][GXGL_GET_TEXTURE_BITS(vtype)];
	const FVFINFO *pColorFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_COLOR_RGB+color_a][GXGL_GET_COLOR_BITS(vtype) ^ 0x4];
	const FVFINFO *pNormalFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_NORMAL][GXGL_GET_NORMAL_BITS(vtype)];
	const FVFINFO *pVertexFVFInfo	= &s_FVFInfoTable[GXGL_FVF_INDEX_VERTEX_XYZ + vertex_size_index][GXGL_GET_VERTEX_BITS(vtype)];

	vsize += pTextureFVFInfo->size;
	vsize += pColorFVFInfo->size;
	vsize += pNormalFVFInfo->size;
	vsize += pVertexFVFInfo->size;

	const int prim_vsize = vsize * nCount;

	// texture
	if(pTextureFVFInfo->size > 0)	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	// color
	if(pColorFVFInfo->size > 0)		glEnableClientState(GL_COLOR_ARRAY);
	// normal
	if(pNormalFVFInfo->size > 0)	glEnableClientState(GL_NORMAL_ARRAY);
	// vertex
	if(pVertexFVFInfo->size > 0)	glEnableClientState(GL_VERTEX_ARRAY);

	//glMatrixMode(GL_MODELVIEW);
	//glPushMatrix();
	//glLoadIdentity();
	if(vtype & GXGL_TRANSFORM_2D)
	{
		// 2D[h
		gxglLoad2DMatrix();
	}
	else
	{
		// 3D[h
		gxglLoad3DMatrix();
	}

	for( int i=0; i < nPrimCount; ++i, temp_vertices += prim_vsize )
	{
		vertices = temp_vertices;
		// texture
		type = pTextureFVFInfo->type;
		size = pTextureFVFInfo->size;
		if(size > 0)
		{
			glTexCoordPointer(2, type, vsize, vertices);
			vertices += size;
		}

		// color
		type = pColorFVFInfo->type;
		size = pColorFVFInfo->size;
		if(size > 0)
		{
			glColorPointer(3+color_a, type, vsize, vertices);
			vertices += size;
		}

		// normal
		type = pNormalFVFInfo->type;
		size = pNormalFVFInfo->size;
		if(size > 0)
		{
			glNormalPointer(type, vsize, vertices);
			vertices += size;
		}

		// vertex
		type = pVertexFVFInfo->type;
		size = pVertexFVFInfo->size;
		if(size > 0)
		{
			glVertexPointer(s_VertexSizeTable[vertex_size_index], type, vsize, vertices);
			vertices += size;
		}
		if(pIndices == nullptr)	glDrawArrays(nPrim, 0, nCount);
		else					glDrawElements(nPrim, nCount, GL_UNSIGNED_SHORT, pIndices);
	}

	// ㏈
	gxglUnload3DMatrix();

	if(vtype & GXGL_TEXTURE_BITS)	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	if(vtype & GXGL_COLOR_BITS)		glDisableClientState(GL_COLOR_ARRAY);
	if(vtype & GXGL_NORMAL_BITS)	glDisableClientState(GL_NORMAL_ARRAY);
	if(vtype & GXGL_VERTEX_BITS)	glDisableClientState(GL_VERTEX_ARRAY);
}

/**********************************************************************//**
 *
 * Viewport
 *
 ----------------------------------------------------------------------
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @param [in]	width	= 
 * @param [in]	height	= 
*//***********************************************************************/
void gxglViewport(int x, int y, int width, int height)
{
	glViewport(x, y, width, height);
	iris::math::FpuOrthoMatrix(pointer_cast<IrisFMtx44*>(s_2DProjectionMatrix)
		, (f32)(x), (f32)(y), (f32)(x+width), (f32)(y+height)
		, -1.0f, 1.0f);
}

/**********************************************************************//**
 *
 * 3d matrix load
 *
*//***********************************************************************/
void gxglLoad3DMatrix(void)
{
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadMatrixf(s_ProjectionMatrix);
	glMatrixMode(GL_MODELVIEW);

	glMultMatrixf(s_ViewMatrix);
	glMultMatrixf(s_WorldMatrix);
}

/**********************************************************************//**
 *
 * 3d matrix unload
 *
*//***********************************************************************/
void gxglUnload3DMatrix(void)
{
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}

/**********************************************************************//**
 *
 * 2d matrix load
 *
*//***********************************************************************/
void gxglLoad2DMatrix(void)
{
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadMatrixf(s_2DProjectionMatrix);
	glMatrixMode(GL_MODELVIEW);
}

/**********************************************************************//**
 *
 * 2d matrix unload
 *
*//***********************************************************************/
void gxglUnload2DMatrix(void)
{
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}

/**********************************************************************//**
 *
 * set matrix
 *
 ----------------------------------------------------------------------
 * @param [in]	type	= ݒ肷}gbNX̎(GXGL_MATRIX_***)
 * @param [in]	mtx		= }gbNX
*//***********************************************************************/
void gxglSetMatrix(int type, const f32* mtx)
{
	switch( type )
	{
	case GXGL_MATRIX_PROJECTION:
		memcpy(s_ProjectionMatrix, mtx, sizeof(s_ProjectionMatrix));
		break;
	case GXGL_MATRIX_VIEW:
		memcpy(s_ViewMatrix, mtx, sizeof(s_ViewMatrix));
		break;
	case GXGL_MATRIX_WORLD:
		memcpy(s_WorldMatrix, mtx, sizeof(s_WorldMatrix));
		break;
	case GXGL_MATRIX_TEXTURE:
		glMatrixMode(GL_TEXTURE);
		glLoadMatrixf(mtx);
		glMatrixMode(GL_MODELVIEW);
		break;
	}
}

/**********************************************************************//**
 *
 * perspective
 *
 ----------------------------------------------------------------------
 * @param [out]	mtx		= }gbNX
 * @param [in]	fovy	= AXyNg
 * @param [in]	aspect	= AXyNg
 * @param [in]	r_near	= nearʋ
 * @param [in]	r_far	= farʋ
*//***********************************************************************/
void gxglMatrixPerspectiveFov(IrisFMtx44* mtx, f32 fovy, f32 aspect, f32 r_near, f32 r_far)
{
	f32	f, asp, nf1, nf2;

	f = fovy * 0.5f;
	f = (f32)cosf(f) / (f32)sinf(f);

	asp = 1.0f / aspect;

	nf1 = 1.0f / (r_near - r_far);
	nf2 = r_near + r_far;

	mtx->x.x = f * asp;
	mtx->x.y = 0.0f;
	mtx->x.z = 0.0f;
	mtx->x.w = 0.0f;
	mtx->y.x = 0.0f;
	mtx->y.y = f;
	mtx->y.z = 0.0f;
	mtx->y.w = 0.0f;
	mtx->z.x = 0.0f;
	mtx->z.y = 0.0f;
	mtx->z.z = nf1 * nf2;
	mtx->z.w = -1.0f;
	mtx->w.x = 0.0f;
	mtx->w.y = 0.0f;
	mtx->w.z = 2.0f * r_near * r_far * nf1;
	mtx->w.w = 0.0f;
}

/**********************************************************************//**
 *
 * set matrix
 *
 ----------------------------------------------------------------------
 * @param [out]	mtx		= }gbNX
 * @param [in]	fovy	= AXyNg
 * @param [in]	aspect	= AXyNg
 * @param [in]	r_near	= nearʋ
 * @param [in]	r_far	= farʋ
*//***********************************************************************/
IrisFMtx44*	gxglMatrixLookAt(IrisFMtx44* mtx, const IrisFVec4* eye, const IrisFVec4* center, const IrisFVec4* up)
{
	IRIS_ASSERT( mtx != nullptr );
	IRIS_ASSERT( eye != nullptr );
	IRIS_ASSERT( center != nullptr );
	IRIS_ASSERT( up != nullptr );
	IrisFVec3	xvec, yvec, zvec;
	IrisFVec3	tmpvec;
	float			tmpscal;

	zvec.x = eye->x - center->x;
	zvec.y = eye->y - center->y;
	zvec.z = eye->z - center->z;

	tmpvec.x = zvec.x * zvec.x;
	tmpvec.y = zvec.y * zvec.y;
	tmpvec.z = zvec.z * zvec.z;

	tmpscal = 1.0f / (float)sqrtf(tmpvec.x + tmpvec.y + tmpvec.z);

	if (tmpscal) {
		zvec.x = zvec.x * tmpscal;
		zvec.y = zvec.y * tmpscal;
		zvec.z = zvec.z * tmpscal;
	}

	yvec.x = up->x;
	yvec.y = up->y;
	yvec.z = up->z;

	xvec.x = yvec.y * zvec.z - yvec.z * zvec.y;
	xvec.y = yvec.z * zvec.x - yvec.x * zvec.z;
	xvec.z = yvec.x * zvec.y - yvec.y * zvec.x;

	yvec.x = zvec.y * xvec.z - zvec.z * xvec.y;
	yvec.y = zvec.z * xvec.x - zvec.x * xvec.z;
	yvec.z = zvec.x * xvec.y - zvec.y * xvec.x;

	tmpvec.x = xvec.x * xvec.x;
	tmpvec.y = xvec.y * xvec.y;
	tmpvec.z = xvec.z * xvec.z;

	tmpscal = 1.0f / (float)sqrtf(tmpvec.x + tmpvec.y + tmpvec.z);

	if (tmpscal) {
		xvec.x = xvec.x * tmpscal;
		xvec.y = xvec.y * tmpscal;
		xvec.z = xvec.z * tmpscal;
	}

	tmpvec.x = yvec.x * yvec.x;
	tmpvec.y = yvec.y * yvec.y;
	tmpvec.z = yvec.z * yvec.z;

	tmpscal = 1.0f / (float)sqrtf(tmpvec.x + tmpvec.y + tmpvec.z);

	if (tmpscal) {
		yvec.x = yvec.x * tmpscal;
		yvec.y = yvec.y * tmpscal;
		yvec.z = yvec.z * tmpscal;
	}

	mtx->x.x = xvec.x;
	mtx->x.y = yvec.x;
	mtx->x.z = zvec.x;
	mtx->x.w = 0.0f;
	mtx->y.x = xvec.y;
	mtx->y.y = yvec.y;
	mtx->y.z = zvec.y;
	mtx->y.w = 0.0f;
	mtx->z.x = xvec.z;
	mtx->z.y = yvec.z;
	mtx->z.z = zvec.z;
	mtx->z.w = 0.0f;
	mtx->w.x = 0.0f;
	mtx->w.y = 0.0f;
	mtx->w.z = 0.0f;
	mtx->w.w = 1.0f;
	return mtx;
}

}	// end of namespace gl
}	// end of namespace gx
}	// end of namespace iris

#endif
